package org.mule.weave.v2.parser.phase

import org.mule.weave.v2.grammar.literals.TypeLiteral
import org.mule.weave.v2.parser.Message
import org.mule.weave.v2.parser.MessageKind
import org.mule.weave.v2.parser.TypePhaseCategory
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.functions.UsingNode
import org.mule.weave.v2.parser.ast.header.HeaderNode
import org.mule.weave.v2.parser.ast.header.directives._
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.types.NativeTypeNode

class TypeDeclValidation extends AstNodeVerifier {

  def validateHeader(directives: Seq[DirectiveNode with Product with Serializable], context: ParsingContext): Unit = {
    directives.foreach {
      case d: TypeDirective =>
        val isNativeReference = d.typeExpression.isInstanceOf[NativeTypeNode]
        if (TypeLiteral.isSystemType(d.variable.name) && !isNativeReference) {
          context.messageCollector.error(Message(MessageKind.INVALID_TYPE_DECLARATION_MESSAGE_KIND, "Cannot use `" + d.variable.name + "` as a type name. It is a system reserved type.", TypePhaseCategory), d.location())
        }
        if (isNativeReference && !TypeLiteral.isSystemType(d.variable.name)) {
          context.messageCollector.error(Message(MessageKind.INVALID_TYPE_DECLARATION_MESSAGE_KIND, "Native type `" + d.variable.name + "` doesn't exist.", TypePhaseCategory), d.location())
        }
      case d: VarDirective =>
        if (TypeLiteral.isSystemType(d.variable.name)) {
          context.messageCollector.error(Message(MessageKind.INVALID_TYPE_DECLARATION_MESSAGE_KIND, "Cannot use `" + d.variable.name + "` as a variable name. It is a system reserved type.", TypePhaseCategory), d.location())
        }
      case d: FunctionDirectiveNode =>
        if (TypeLiteral.isSystemType(d.variable.name)) {
          context.messageCollector.error(Message(MessageKind.INVALID_TYPE_DECLARATION_MESSAGE_KIND, "Cannot use `" + d.variable.name + "` as a function name. It is a system reserved type.", TypePhaseCategory), d.location())
        }
      case d: NamespaceDirective =>
        if (TypeLiteral.isSystemType(d.prefix.name)) {
          context.messageCollector.error(Message(MessageKind.INVALID_TYPE_DECLARATION_MESSAGE_KIND, "Cannot use `" + d.prefix.name + "` as a function name. It is a system reserved type.", TypePhaseCategory), d.location())
        }
      case _ =>
    }
  }

  override def verify(node: AstNode, context: ParsingContext): Unit = {
    node match {
      case HeaderNode(dirs) =>
        val directives = dirs.collect({
          case d: TypeDirective         => d
          case d: VarDirective          => d
          case d: FunctionDirectiveNode => d
          case d: NamespaceDirective    => d
        })
        validateHeader(directives, context)
      case ModuleNode(name, elements) =>
        val directives = elements.collect({
          case d: TypeDirective         => d
          case d: VarDirective          => d
          case d: FunctionDirectiveNode => d
          case d: NamespaceDirective    => d
        })
        validateHeader(directives, context)
      case UsingNode(assignments, _, _) =>
        assignments.assignmentSeq.foreach((usingAssignment) =>
          if (TypeLiteral.isSystemType(usingAssignment.name.name)) {
            context.messageCollector.error(Message(MessageKind.INVALID_TYPE_DECLARATION_MESSAGE_KIND, "Cannot use `" + usingAssignment.name + "` as a variable name. It is a system reserved type.", TypePhaseCategory), usingAssignment.location())
          })
      case _ =>
    }
  }
}