package org.mule.weave.v2.parser.phase

import org.mule.weave.v2.api.tooling.internal.DefaultDWAstNode
import org.mule.weave.v2.parser.InvalidAnnotationTarget
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNodeHelper
import org.mule.weave.v2.parser.ast.functions.FunctionParameter
import org.mule.weave.v2.parser.ast.header.directives.AnnotationDirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.DirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.FunctionDirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.ImportDirective
import org.mule.weave.v2.parser.ast.header.directives.InputDirective
import org.mule.weave.v2.parser.ast.header.directives.NamespaceDirective
import org.mule.weave.v2.parser.ast.header.directives.TypeDirective
import org.mule.weave.v2.parser.ast.header.directives.VarDirective
import org.mule.weave.v2.parser.ast.header.directives.VersionDirective
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.types.KeyTypeNode
import org.mule.weave.v2.parser.ast.types.NameTypeNode
import org.mule.weave.v2.parser.ast.types.WeaveTypeNodeWithSchema
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.scope.AstNavigator
import org.mule.weave.v2.scope.BaseVariableScope
import org.mule.weave.v2.scope.ScopesNavigator

class ScopeAnnotationProcessingPhase[R <: AstNode, T <: AstNodeResultAware[R] with ScopeNavigatorResultAware]() extends CompilationPhase[T, T] {

  override def doCall(source: T, context: ParsingContext): PhaseResult[_ <: T] = {
    val scopeNavigator: ScopesNavigator = source.scope
    val astNavigator: AstNavigator = getAstNavigator(source)
    val annotationNodes: Seq[AnnotationNode] = scopeNavigator.astNavigator().allWithType(classOf[AnnotationNode])
    annotationNodes.foreach((annotationNode) => {
      scopeNavigator
        .resolveVariable(annotationNode.name)
        .foreach((annotationRef) => {
          val annotatedNode: Option[AstNode] = astNavigator.parentOf(annotationNode)
          val annotationName: NameIdentifier = annotationRef.fqnReferenceName
          val annotationDefScopeNavigator = annotationRef.scope
          val annotationDefAstNavigation: AstNavigator = annotationDefScopeNavigator.astNavigator()
          annotationDefAstNavigation
            .parentWithType(annotationRef.referencedNode, classOf[AnnotationDirectiveNode])
            .foreach((annotationDirective) => {
              //We collect the target where this annotation is valid and validate
              val mayBeTarget = collectAnnotationTarget(annotationDirective, annotationRef.scope)
              mayBeTarget match {
                case Some(target) => {
                  val targetValues: Array[String] = collectPossibleTargets(target)
                  val passedTarget = targetValues.exists({
                    case "Parameter" => {
                      annotatedNode match {
                        case Some(_: FunctionParameter) => true
                        case _                          => false
                      }
                    }
                    case "Function" => {
                      annotatedNode match {
                        case Some(_: FunctionDirectiveNode) => true
                        case _                              => false
                      }
                    }
                    case "Variable" => {
                      annotatedNode match {
                        case Some(_: VarDirective)   => true
                        case Some(_: InputDirective) => true
                        case _                       => false
                      }
                    }
                    case "Type" => {
                      annotatedNode match {
                        case Some(_: TypeDirective) => true
                        case _                      => false
                      }
                    }
                    case "Namespace" => {
                      annotatedNode match {
                        case Some(_: NamespaceDirective) => true
                        case _                           => false
                      }
                    }
                    case "Version" => {
                      annotatedNode match {
                        case Some(_: VersionDirective) => true
                        case _                         => false
                      }
                    }
                    case "Import" => {
                      annotatedNode match {
                        case Some(_: ImportDirective) => true
                        case _                        => false
                      }
                    }
                    case "KeyType" => {
                      annotatedNode match {
                        case Some(_: KeyTypeNode) => true
                        case _                    => false
                      }
                    }
                    case "AttributeType" => {
                      annotatedNode match {
                        case Some(_: NameTypeNode) => true
                        case _                     => false
                      }
                    }
                    case "TypeExpression" => {
                      annotatedNode match {
                        case Some(_: WeaveTypeNodeWithSchema) => true
                        case _                                => false
                      }
                    }
                    case "Annotation" => {
                      annotatedNode match {
                        case Some(_: AnnotationDirectiveNode) => true
                        case _                                => false
                      }
                    }
                    case "Key" => {
                      annotatedNode match {
                        case Some(_: NameNode) => true
                        case _                 => false
                      }
                    }
                    case "Expression" => {
                      annotatedNode match {
                        case Some(_: DirectiveNode) => false
                        case _                      => true
                      }
                    }
                    case _ => false
                  })
                  if (passedTarget) {
                    dispatchToAnnotationProcessor(annotationName, annotatedNode.get, astNavigator, scopeNavigator, context, annotationNode)
                  } else {
                    context.messageCollector.error(InvalidAnnotationTarget(targetValues), annotationNode.location())
                  }
                }
                case _ => {
                  //If No target then all target are valid
                  dispatchToAnnotationProcessor(annotationName, annotatedNode.get, astNavigator, scopeNavigator, context, annotationNode)
                }
              }
            })
        })
    })
    SuccessResult(source, context)
  }

  private def getAstNavigator(source: T) = {
    source.scope.rootScope.astNavigator()
  }

  private def collectPossibleTargets(target: AnnotationNode): Array[String] = {
    AnnotationNodeHelper.argStringSeq("targets", target).getOrElse(Seq()).toArray;
  }

  private def dispatchToAnnotationProcessor(annotationFQN: NameIdentifier, astNode: AstNode, astNavigator: AstNavigator, scopeNavigator: ScopesNavigator, context: ParsingContext, annotation: AnnotationNode): Unit = {
    context
      .scopePhaseAnnotationProcessorFor(annotationFQN)
      .foreach((processor) => {
        processor.run(DefaultDWAstNode(astNode, astNavigator, None, Some(scopeNavigator)), DefaultDWAstNode(annotation, astNavigator, None, Some(scopeNavigator)), new DefaultAnnotationContext(context))
      })
  }

  private def collectAnnotationTarget(annotationDirective: AnnotationDirectiveNode, annotationScopeNavigator: BaseVariableScope): Option[AnnotationNode] = {
    annotationDirective.codeAnnotations.find((annotationNode) => {
      val maybeReference = annotationScopeNavigator.resolveVariable(annotationNode.name)
      maybeReference.exists((ref) => {
        ref.fqnReferenceName == NameIdentifier.ANNOTATION_TARGET_ANNOTATION
      })
    })
  }
}
