package org.mule.weave.v2.inspector

import org.mule.weave.v2.editor.QuickFixAction
import org.mule.weave.v2.editor.WeaveTextDocument
import org.mule.weave.v2.grammar.EqOpId
import org.mule.weave.v2.grammar.NotEqOpId
import org.mule.weave.v2.parser.SimplifyBooleanEquality
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.operators.BinaryOpNode
import org.mule.weave.v2.parser.ast.structure.BooleanNode
import org.mule.weave.v2.parser.phase.AstNodeResultAware
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.ScopeNavigatorResultAware

object EqualsBooleanInspector extends CodeInspector[AstNodeResultAware[_] with ScopeNavigatorResultAware] {

  override def inspect(node: AstNode, scopeData: AstNodeResultAware[_] with ScopeNavigatorResultAware, parsingContext: ParsingContext): Unit = {

    node match {
      case bon @ BinaryOpNode(op, _, BooleanNode("true", _), _) if (op == EqOpId) => {
        parsingContext.messageCollector.warning(SimplifyBooleanEquality(bon, "Unnecessary comparison."), node.location())
      }
      case bon @ BinaryOpNode(op, _, BooleanNode("false", _), _) if (op == EqOpId) => {
        parsingContext.messageCollector.warning(SimplifyBooleanEquality(bon, "Can be represented with a not expression."), node.location())
      }
      case bon @ BinaryOpNode(op, _, BooleanNode("false", _), _) if (op == NotEqOpId) => {
        parsingContext.messageCollector.warning(SimplifyBooleanEquality(bon, "Unnecessary comparison."), node.location())
      }
      case bon @ BinaryOpNode(op, _, BooleanNode("true", _), _) if (op == NotEqOpId) => {
        parsingContext.messageCollector.warning(SimplifyBooleanEquality(bon, "Can be represented with a not expression."), node.location())
      }
      case _ =>
    }
  }

}

class EqualsBooleanInspectorFixAction(bon: BinaryOpNode) extends QuickFixAction {
  override def run(document: WeaveTextDocument): Unit = {
    bon.rhs match {
      case BooleanNode("true", _) if (bon.opId == EqOpId) => {
        document.delete(bon.lhs.location().endPosition.index, bon.rhs.location().endPosition.index)
      }
      case BooleanNode("false", _) if (bon.opId == NotEqOpId) => {
        document.delete(bon.lhs.location().endPosition.index, bon.rhs.location().endPosition.index)
      }
      case BooleanNode("false", _) if (bon.opId == EqOpId) => {
        document.delete(bon.lhs.location().endPosition.index, bon.rhs.location().endPosition.index)
        document.insert("!", bon.location().startPosition.index)
      }
      case BooleanNode("true", _) if (bon.opId == NotEqOpId) => {
        document.delete(bon.lhs.location().endPosition.index, bon.rhs.location().endPosition.index)
        document.insert("!", bon.location().startPosition.index)
      }
      case _ =>
    }
  }
}
