package org.mule.weave.v2.editor.composer

import org.mule.weave.v2.codegen.CodeGenerator
import org.mule.weave.v2.grammar.AdditionOpId
import org.mule.weave.v2.grammar.AsOpId
import org.mule.weave.v2.grammar.BinaryOpIdentifier
import org.mule.weave.v2.grammar.DivisionOpId
import org.mule.weave.v2.grammar.MinusOpId
import org.mule.weave.v2.grammar.MultiplicationOpId
import org.mule.weave.v2.grammar.NotOpId
import org.mule.weave.v2.grammar.SubtractionOpId
import org.mule.weave.v2.grammar.UnaryOpIdentifier
import org.mule.weave.v2.parser.MappingParser
import org.mule.weave.v2.parser.annotation.EnclosedMarkAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.conditional.DefaultNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallParametersNode
import org.mule.weave.v2.parser.ast.logical.AndNode
import org.mule.weave.v2.parser.ast.logical.OrNode
import org.mule.weave.v2.parser.ast.operators.BinaryOpNode
import org.mule.weave.v2.parser.ast.operators.UnaryOpNode
import org.mule.weave.v2.parser.ast.structure.ArrayNode
import org.mule.weave.v2.parser.ast.structure.BooleanNode
import org.mule.weave.v2.parser.ast.structure.DateTimeNode
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.structure.LocalDateNode
import org.mule.weave.v2.parser.ast.structure.LocalDateTimeNode
import org.mule.weave.v2.parser.ast.structure.LocalTimeNode
import org.mule.weave.v2.parser.ast.structure.NullNode
import org.mule.weave.v2.parser.ast.structure.NumberNode
import org.mule.weave.v2.parser.ast.structure.StringInterpolationNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.structure.TimeNode
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.location.UnknownLocation
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.ParsingResult
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.utils.StringEscapeHelper

import scala.annotation.tailrec

object ComposerCodeGenerator {

  private def partitionByResult(results: Seq[Result]): (Seq[SuccessResult], Seq[FailureResult]) = {
    @tailrec
    def split(xs: Seq[Result], success: Seq[SuccessResult], failures: Seq[FailureResult]): (Seq[SuccessResult], Seq[FailureResult]) = {
      if (xs.isEmpty) {
        (success, failures)
      } else {
        val head = xs.head
        head match {
          case sr: SuccessResult =>
            split(xs.tail, success :+ sr, failures)
          case fr: FailureResult =>
            split(xs.tail, success, failures :+ fr)
        }
      }
    }
    split(results, Seq.empty[SuccessResult], Seq.empty[FailureResult])
  }

  private def createSingleAstNode: (ComposerExpressionNode, String => Result) => Result = (expression: ComposerExpressionNode, f: String => Result) => {
    if (expression.children.isEmpty) {
      if (expression.token.text != null) {
        f(expression.token.text)
      } else {
        FailureResult(s"Missing text attribute at ${expression.token.kind} node.")
      }
    } else {
      FailureResult(s"Unexpected children at ${expression.token.kind} node")
    }
  }

  private def createTimesAstNode: (ComposerExpressionNode, String => Result) => Result = (expression: ComposerExpressionNode, f: String => Result) => {
    createSingleAstNode(expression, text => {
      if (text.startsWith("|") && text.endsWith("|")) {
        val literalValue = text.substring(1, text.length - 1)
        f(literalValue)
      } else {
        FailureResult(s"${expression.token.kind} node require text attribute content wrapped by '|'.")
      }
    })
  }

  private def createNullAstNode(expression: ComposerExpressionNode): Result = {
    if (expression.children.isEmpty) {
      SuccessResult(NullNode())
    } else {
      FailureResult(s"Unexpected children at ${expression.token.kind} node")
    }
  }

  private def createArrayAstNode(expression: ComposerExpressionNode, ctx: ParsingContext): Result = {
    if (expression.children.nonEmpty) {
      val child = expression.children.map(nd => createAstNode(nd, ctx, isRootOrArrayItem = true))
      val (success, failures) = partitionByResult(child)
      if (failures.isEmpty) {
        val elements = success.map(_.astNode)
        SuccessResult(ArrayNode(elements))
      } else {
        val errors = failures.flatMap(_.errors)
        FailureResult(errors)
      }
    } else {
      SuccessResult(ArrayNode(Seq()))
    }
  }

  private def createTextInterpolationAstNode: (ComposerExpressionNode, ParsingContext) => Result = (expression: ComposerExpressionNode, ctx: ParsingContext) => {
    if (expression.children.nonEmpty) {
      val child = expression.children.map(nd => createAstNode(nd, ctx))
      val (success, failures) = partitionByResult(child)
      if (failures.isEmpty) {
        val elements = success.map(_.astNode)
        SuccessResult(StringInterpolationNode(elements))
      } else {
        val errors = failures.flatMap(_.errors)
        FailureResult(errors)
      }
    } else {
      FailureResult(s"Empty children at ${expression.token.kind} node.")
    }
  }

  private def createLogicalOptAstNode: (String, ComposerExpressionNode, ParsingContext, (AstNode, AstNode) => AstNode) => Result = (function: String, expression: ComposerExpressionNode, ctx: ParsingContext, f: (AstNode, AstNode) => AstNode) => {
    if (expression.children.length == 3) {
      val lhs = createAstNode(expression.children(1), ctx)
      val rhs = createAstNode(expression.children(2), ctx)
      (lhs, rhs) match {
        case (SuccessResult(l), SuccessResult(r))   => SuccessResult(f(l, r))
        case (_: SuccessResult, fr: FailureResult)  => fr
        case (fr: FailureResult, _: SuccessResult)  => fr
        case (FailureResult(le), FailureResult(re)) => FailureResult(le ++ re)
      }
    } else {
      FailureResult(s"Invalid children size at $function function. Allowed children are 3 but found ${expression.children.length} children")
    }
  }

  private def createUnaryOpAstNode: (String, UnaryOpIdentifier, ComposerExpressionNode, ParsingContext) => Result = (function: String, opId: UnaryOpIdentifier, expression: ComposerExpressionNode, ctx: ParsingContext) => {
    if (expression.children.length == 2) {
      val rhs = createAstNode(expression.children(1), ctx)
      rhs match {
        case SuccessResult(r) =>
          SuccessResult(UnaryOpNode(opId, r))
        case fr: FailureResult => fr
      }
    } else {
      FailureResult(s"Invalid children size at $function function. Allowed children are 2 but found ${expression.children.length} children")
    }
  }

  private def createBinaryOpAstNode: (BinaryOpIdentifier, ComposerExpressionNode, ParsingContext) => Result = (binaryOpId: BinaryOpIdentifier, expression: ComposerExpressionNode, ctx: ParsingContext) => {
    if (expression.children.length == 3) {
      val lhs = createAstNode(expression.children(1), ctx)
      val rhs = createAstNode(expression.children(2), ctx)
      (lhs, rhs) match {
        case (SuccessResult(l), SuccessResult(r))   => SuccessResult(BinaryOpNode(binaryOpId, l, r))
        case (_: SuccessResult, fr: FailureResult)  => fr
        case (fr: FailureResult, _: SuccessResult)  => fr
        case (FailureResult(le), FailureResult(re)) => FailureResult(le ++ re)
      }
    } else {
      FailureResult(s"Invalid children size at ${binaryOpId.name} function. Allowed children are 3 but found ${expression.children.length} children")
    }
  }

  private def createDefaultAstNode(expression: ComposerExpressionNode, ctx: ParsingContext): Result = {
    if (expression.children.length == 3) {
      val lhs = createAstNode(expression.children(1), ctx)
      val rhs = createAstNode(expression.children(2), ctx)
      (lhs, rhs) match {
        case (SuccessResult(l), SuccessResult(r))   => SuccessResult(DefaultNode(l, r))
        case (_: SuccessResult, fr: FailureResult)  => fr
        case (fr: FailureResult, _: SuccessResult)  => fr
        case (FailureResult(le), FailureResult(re)) => FailureResult(le ++ re)
      }
    } else {
      FailureResult(s"Invalid children size at default function. Allowed children are 3 but found ${expression.children.length} children")
    }
  }

  private def createAstNodeFromNamedNode: (String, ComposerExpressionNode, String => Result) => Result = (kind: String, expression: ComposerExpressionNode, f: String => Result) => {
    if (expression.children.nonEmpty) {
      val namedExpressionNode = expression.children.head
      if (namedExpressionNode.token != null) {
        if (namedExpressionNode.token.kind == ComposerExpressionTokenKind.NAME) {
          if (namedExpressionNode.token != null) {
            val name = expression.children.head.token.text
            f(name)
          } else {
            FailureResult(s"Missing text attribute at ${ComposerExpressionTokenKind.NAME} token.")
          }
        } else {
          FailureResult(s"Missing ${ComposerExpressionTokenKind.NAME} token child at $kind. Found ${namedExpressionNode.token.kind} token.")
        }
      } else {
        FailureResult(s"Missing token attribute at $kind node.")
      }
    } else {
      FailureResult(s"Empty children at $kind node.")
    }
  }

  private def createParenthesisAstNode: (ComposerExpressionNode, ParsingContext) => Result = (expression: ComposerExpressionNode, ctx: ParsingContext) => {
    if (expression.children.length == 1) {
      val children = createAstNode(expression.children.head, ctx)
      children match {
        case sr: SuccessResult =>
          sr.astNode.annotate(EnclosedMarkAnnotation(UnknownLocation))
          sr
        case fr: FailureResult =>
          fr
      }
    } else {
      FailureResult(s"Invalid children size at ${expression.token.kind}. Allowed children are 1 but found ${expression.children.length} children")
    }
  }

  private def createAstNode(expression: ComposerExpressionNode, ctx: ParsingContext, isRootOrArrayItem: Boolean = false): Result = {
    if (expression.token != null) {
      if (expression.children != null) {
        if (expression.token.kind != null && expression.token.kind.nonEmpty) {
          expression.token.kind match {
            case ComposerExpressionTokenKind.NULL =>
              createNullAstNode(expression)

            case ComposerExpressionTokenKind.TYPE =>
              createSingleAstNode(expression, text => SuccessResult(TypeReferenceNode(NameIdentifier(text))))

            case ComposerExpressionTokenKind.NUMBER =>
              createSingleAstNode(expression, text => SuccessResult(NumberNode(text)))

            case ComposerExpressionTokenKind.BOOLEAN =>
              createSingleAstNode(expression, text => SuccessResult(BooleanNode(text)))

            case ComposerExpressionTokenKind.TEXT =>
              createSingleAstNode(expression, text => {
                val sn = StringNode(StringEscapeHelper.escapeString(text, '\"', insertQuotes = false))
                if (isRootOrArrayItem) {
                  sn.withQuotation('\"')
                }
                SuccessResult(sn)
              })

            case ComposerExpressionTokenKind.DATE =>
              createTimesAstNode(expression, literalValue => {
                SuccessResult(LocalDateNode(literalValue))
              })
            case ComposerExpressionTokenKind.DATE_TIME =>
              createTimesAstNode(expression, literalValue => {
                SuccessResult(DateTimeNode(literalValue))
              })
            case ComposerExpressionTokenKind.TIME =>
              createTimesAstNode(expression, literalValue => {
                SuccessResult(TimeNode(literalValue))
              })
            case ComposerExpressionTokenKind.LOCAL_TIME =>
              createTimesAstNode(expression, literalValue => {
                SuccessResult(LocalTimeNode(literalValue))
              })
            case ComposerExpressionTokenKind.LOCAL_DATE_TIME =>
              createTimesAstNode(expression, literalValue => {
                SuccessResult(LocalDateTimeNode(literalValue))
              })

            case ComposerExpressionTokenKind.TEXT_INTERPOLATION =>
              createTextInterpolationAstNode(expression, ctx)

            case ComposerExpressionTokenKind.ARRAY =>
              createArrayAstNode(expression, ctx)

            case ComposerExpressionTokenKind.UNARY_OP =>
              createAstNodeFromNamedNode(ComposerExpressionTokenKind.UNARY_OP, expression, opId => {
                val node = opId match {
                  case "!" | NotOpId.name =>
                    createUnaryOpAstNode(opId, NotOpId, expression, ctx)
                  case MinusOpId.name =>
                    createUnaryOpAstNode(MinusOpId.name, MinusOpId, expression, ctx)
                  case other =>
                    FailureResult(s"Invalid unary operator: $other.")
                }
                node
              })

            case ComposerExpressionTokenKind.BINARY_OP =>
              createAstNodeFromNamedNode(ComposerExpressionTokenKind.BINARY_OP, expression, opId => {
                val node = opId match {
                  case AdditionOpId.name =>
                    createBinaryOpAstNode(AdditionOpId, expression, ctx)
                  case SubtractionOpId.name =>
                    createBinaryOpAstNode(SubtractionOpId, expression, ctx)
                  case MultiplicationOpId.name =>
                    createBinaryOpAstNode(MultiplicationOpId, expression, ctx)
                  case DivisionOpId.name =>
                    createBinaryOpAstNode(DivisionOpId, expression, ctx)
                  case AsOpId.name =>
                    createBinaryOpAstNode(AsOpId, expression, ctx)
                  case "default" =>
                    createDefaultAstNode(expression, ctx)
                  case other =>
                    FailureResult(s"Invalid binary operator: $other.")
                }
                node
              })
            case ComposerExpressionTokenKind.FUNCTION_CALL =>
              createAstNodeFromNamedNode(ComposerExpressionTokenKind.FUNCTION_CALL, expression, function => {
                val node = function match {
                  case "or" =>
                    createLogicalOptAstNode(function, expression, ctx, (lhs, rhs) => {
                      OrNode(lhs, rhs)
                    })
                  case "and" =>
                    createLogicalOptAstNode(function, expression, ctx, (lhs, rhs) => {
                      AndNode(lhs, rhs)
                    })

                  case "!" | NotOpId.name =>
                    createUnaryOpAstNode(function, NotOpId, expression, ctx)
                  case MinusOpId.name =>
                    createUnaryOpAstNode(MinusOpId.name, MinusOpId, expression, ctx)

                  case AdditionOpId.name =>
                    createBinaryOpAstNode(AdditionOpId, expression, ctx)
                  case SubtractionOpId.name =>
                    createBinaryOpAstNode(SubtractionOpId, expression, ctx)
                  case MultiplicationOpId.name =>
                    createBinaryOpAstNode(MultiplicationOpId, expression, ctx)
                  case DivisionOpId.name =>
                    createBinaryOpAstNode(DivisionOpId, expression, ctx)

                  case _ =>
                    val vrn = VariableReferenceNode(expression.children.head.token.text)
                    val results = expression.children.drop(1).map(nd => createAstNode(nd, ctx))
                    val (success, failures) = partitionByResult(results)
                    if (failures.isEmpty) {
                      val args = success.map(_.astNode)
                      val params = FunctionCallParametersNode(args)
                      val fcn = FunctionCallNode(vrn, params)
                      SuccessResult(fcn)
                    } else {
                      val errors = failures.flatMap(_.errors)
                      FailureResult(errors)
                    }
                }
                node
              })

            case ComposerExpressionTokenKind.PARENTHESIS =>
              createParenthesisAstNode(expression, ctx)

            case ComposerExpressionTokenKind.PILL =>
              createSingleAstNode(expression, text => {
                val value: PhaseResult[ParsingResult[DocumentNode]] = MappingParser.parse(MappingParser.parsingPhase(), WeaveResource("expression", text), ctx)
                SuccessResult(value.getResult().astNode.root)
              })

            case other =>
              FailureResult(s"$other is not a valid at ${expression.token.kind} node.")
          }
        } else {
          FailureResult(s"Missing kind attribute at ${expression.token.kind} node.")
        }
      } else {
        FailureResult(s"Missing children attribute at ${expression.token.kind} node.")
      }
    } else {
      FailureResult("Missing token attribute.")
    }
  }

  def generate(expression: ComposerExpressionNode, ctx: ParsingContext): ComposerDWScriptResult = {
    val result = createAstNode(expression, ctx, isRootOrArrayItem = true)
    result match {
      case SuccessResult(astNode) =>
        val code = CodeGenerator.generate(astNode)
        ComposerDWScriptResult(code = code)
      case FailureResult(errors) =>
        ComposerDWScriptResult(code = "", success = false, errors)
    }
  }
}

trait Result
case class SuccessResult(astNode: AstNode) extends Result
case class FailureResult(errors: Seq[String]) extends Result

object FailureResult {
  def apply(error: String) = new FailureResult(Seq(error))
}