package org.mule.weave.v2.parser.phase

import org.mule.weave.v2.codegen.CodeGenerator
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.module.ModuleLoaderProvider
import org.mule.weave.v2.parser.{ MessageCollector, ModuleParser }
import org.mule.weave.v2.sdk.{ WeaveResource, WeaveResourceResolver }
import org.mule.weave.v2.utils.{ BasicCache, CacheBuilder, WeaveFile }

/**
  * Builds a ast.ModuleNode for a given identifier
  */
trait ModuleLoader {

  /**
    * Builds if possible the module ast for the given NameIdentifier
    *
    * @param nameIdentifier The name of the module to be build
    * @param moduleContext  The parsing context for this module
    * @return If was able to build the ParsingResult
    */
  def loadModule(nameIdentifier: NameIdentifier, moduleContext: ParsingContext): Option[PhaseResult[ParsingResult[ModuleNode]]]

  /**
    * Returns the content of the specified module
    *
    * @param nameIdentifier The name identifier
    * @param parsingContext The parsing context
    * @return The content
    */
  def moduleContent(nameIdentifier: NameIdentifier, parsingContext: ParsingContext): Option[WeaveResource] = {
    loadModule(nameIdentifier, parsingContext).flatMap((phaseResult) => {
      phaseResult.mayBeResult.map((parsingResult) => {
        val code = CodeGenerator.generate(parsingResult.astNode)
        WeaveResource(parsingResult.input.resource.url(), code)
      })
    })
  }

  /**
    * The name of the loader if None then handles the default loader
    *
    * @return
    */
  def name(): Option[String] = None

  /**
    * Checks if the module for the given NameIdentifier can be resolved
    *
    * @param nameIdentifier The name identifier
    * @return whether the module can be resolved by this ModuleLoader or not
    */
  def canResolveModule(nameIdentifier: NameIdentifier): Boolean

}

object ModuleLoader {
  val DEFAULT_LOADER_NAME = "dw"
  val BINARY_LOADER_NAME = "bdwl"

  def apply(weaveResourceResolver: WeaveResourceResolver): ModuleLoader = {
    new ResourceBasedModuleParser(weaveResourceResolver)
  }

  def apply(weaveResourceResolver: WeaveResourceResolver, name: String): ModuleLoader = {
    new ResourceBasedModuleParser(weaveResourceResolver, Some(name))
  }
}

/**
  * Handles the loading the definition of multiple kind of module
  *
  * @param loaders The supported module loaders
  */
class ModuleLoaderManager(loaders: Seq[ModuleLoader]) {

  private val loadersByName: Map[String, Seq[ModuleLoader]] = loaders.groupBy((loader) => loader.name().getOrElse(ModuleLoader.DEFAULT_LOADER_NAME))

  def loadModule(nameIdentifier: NameIdentifier, parentContext: ParsingContext): Option[PhaseResult[ParsingResult[ModuleNode]]] = {
    val moduleLoaderName = nameIdentifier.loader.getOrElse(ModuleLoader.DEFAULT_LOADER_NAME)
    loadersByName
      .get(moduleLoaderName)
      .flatMap((loaders) => loadModule(loaders, nameIdentifier, parentContext))
  }

  private def loadModule(loaders: Seq[ModuleLoader], nameIdentifier: NameIdentifier, parentContext: ParsingContext) = {
    val mayBeModule = loaders.toStream
      .flatMap((parser) => {
        parser.loadModule(nameIdentifier, parentContext)
      })
      .headOption
    mayBeModule
  }

  def canResolveModule(nameIdentifier: NameIdentifier): Boolean = {
    val moduleLoaderName = nameIdentifier.loader.getOrElse(ModuleLoader.DEFAULT_LOADER_NAME)
    loadersByName.get(moduleLoaderName).exists(_.exists(_.canResolveModule(nameIdentifier)))
  }

  def moduleContent(nameIdentifier: NameIdentifier, parentContext: ParsingContext): Option[WeaveResource] = {
    val moduleLoaderName = nameIdentifier.loader.getOrElse(ModuleLoader.DEFAULT_LOADER_NAME)
    loadersByName
      .get(moduleLoaderName)
      .flatMap(
        (loaders) =>
          loaders.toStream
            .flatMap((parser) => {
              parser.moduleContent(nameIdentifier, parentContext)
            })
            .headOption)
  }
}

object ModuleLoaderManager {
  def apply(parsers: Seq[ModuleLoader], additionalModules: ModuleLoaderProvider): ModuleLoaderManager = {
    new ModuleLoaderManager(parsers ++ additionalModules.getModules)
  }

  def apply(loaders: ModuleLoader*): ModuleLoaderManager = new ModuleLoaderManager(loaders)
}

/**
  * Loads a weave resource base module
  *
  * @param weaveResourceResolver The resource resolver
  * @param loaderName            The loader name
  */
class ResourceBasedModuleParser(weaveResourceResolver: WeaveResourceResolver, loaderName: Option[String] = None) extends ModuleLoader {

  override def name(): Option[String] = loaderName

  override def moduleContent(nameIdentifier: NameIdentifier, parsingContext: ParsingContext): Option[WeaveResource] = {
    weaveResourceResolver.resolve(nameIdentifier)
  }

  override def loadModule(nameIdentifier: NameIdentifier, parentContext: ParsingContext): Option[PhaseResult[ParsingResult[ModuleNode]]] = {
    parentContext.notificationManager.startResolvingResource(nameIdentifier, weaveResourceResolver)
    val resolve: Option[WeaveResource] = weaveResourceResolver.resolve(nameIdentifier)
    parentContext.notificationManager.endResolvingResource(nameIdentifier, WeaveFile.fileExtension, weaveResourceResolver, resolve)
    resolve.map((resource: WeaveResource) => {
      parserAsModule(nameIdentifier, resource, parentContext)
    })
  }

  def parserAsModule(nameIdentifier: NameIdentifier, resource: WeaveResource, parsingContext: ParsingContext): PhaseResult[ParsingResult[ModuleNode]] = {
    val moduleParsingResult: PhaseResult[ParsingResult[ModuleNode]] = ModuleParser.parse(ModuleParser.parsingPhase(), resource, parsingContext.withMessageCollector(new MessageCollector()))
    if (moduleParsingResult.hasErrors()) {
      val mappingAsModule: PhaseResult[ParsingResult[ModuleNode]] = MappingToModuleAdaptor.mappingAsModuleNode(nameIdentifier, resource, parsingContext.withMessageCollector(new MessageCollector()))
      if (mappingAsModule.hasErrors()) {
        moduleParsingResult
      } else {
        mappingAsModule
      }
    } else {
      moduleParsingResult
    }
  }

  override def canResolveModule(nameIdentifier: NameIdentifier): Boolean = weaveResourceResolver.resolve(nameIdentifier).isDefined
}

class CachedResourceBasedModuleParser(weaveResourceResolver: WeaveResourceResolver, loaderName: Option[String] = None) extends ResourceBasedModuleParser(weaveResourceResolver, loaderName) {

  val cache: BasicCache[(NameIdentifier, String), PhaseResult[ParsingResult[ModuleNode]]] = CacheBuilder
    .apply[(NameIdentifier, String), PhaseResult[ParsingResult[ModuleNode]]](null)
    .build()

  override def parserAsModule(nameIdentifier: NameIdentifier, resource: WeaveResource, parsingContext: ParsingContext): PhaseResult[ParsingResult[ModuleNode]] = {
    cache.get((nameIdentifier, resource.content()), (entry) => {
      super.parserAsModule(nameIdentifier, resource, parsingContext)
    })
  }
}
