package org.mule.weave.v2.parser.phase

import org.mule.weave.v2.api.tooling.annotation.DWAnnotationContext
import org.mule.weave.v2.api.tooling.annotation.DWAnnotationProcessor
import org.mule.weave.v2.api.tooling.annotation.DWCanonicalProcessingPhase
import org.mule.weave.v2.api.tooling.annotation.DWParsingProcessingPhase
import org.mule.weave.v2.api.tooling.annotation.DWProcessingPhase
import org.mule.weave.v2.api.tooling.annotation.DWScopeProcessingPhase
import org.mule.weave.v2.api.tooling.annotation.DWTypeProcessingPhase
import org.mule.weave.v2.api.tooling.ast.DWAstNode
import org.mule.weave.v2.api.tooling.internal.DefaultDWAstNode
import org.mule.weave.v2.api.tooling.message.{ MessageCollector => ApiMessageCollector }
import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.scope.AstNavigator
import org.mule.weave.v2.scope.ScopesNavigator
import org.mule.weave.v2.ts.TypeGraph

import scala.collection.mutable

/**
  * Annotation processor executes on each node being annotated by the given annotation on the specified compilation phase
  */
trait AnnotationProcessor[ContextInformation <: AnnotationContext] extends DWAnnotationProcessor {

  /**
    * Run the annotation processing logic on the registered phase
    *
    * @param annotatedNode The node that was annotated
    * @param annotation    The annotation that was used
    * @param context       The context information available on this phase
    */
  def run(annotatedNode: AstNode, annotation: AnnotationNode, context: ContextInformation): Unit

  /**
    * The phase where this Annotation processor is going to be executed
    */
  def phase: ProcessingPhase[ContextInformation]

  def annotationContext(astNavigator: AstNavigator, scopesNavigator: Option[ScopesNavigator], typeGraph: Option[TypeGraph], messageCollector: MessageCollector): ContextInformation

  override def run(annotatedNode: DWAstNode, annotation: DWAstNode, context: DWAnnotationContext): Unit = {
    val annotatedNodeWrapper = annotatedNode.asInstanceOf[DefaultDWAstNode]
    val annotationNodeWrapper = annotation.asInstanceOf[DefaultDWAstNode]
    val messageCollector = context.getMessageCollector.asInstanceOf[MessageCollector]
    val ctx = annotationContext(annotationNodeWrapper.astNavigator, annotationNodeWrapper.scopeGraph, annotatedNodeWrapper.typeGraph, messageCollector)
    run(annotatedNodeWrapper.node, annotationNodeWrapper.node.asInstanceOf[AnnotationNode], ctx)
  }
}

/**
  * A registry of all the Annotation processors to be registered on the ParsingContext
  */
trait AnnotationProcessorRegistry {
  def annotations(): Map[NameIdentifier, AnnotationProcessor[_]]
}

object DefaultAnnotationProcessorRegistry extends AnnotationProcessorRegistry {

  override def annotations(): Map[NameIdentifier, AnnotationProcessor[_]] = {
    Map(
      NameIdentifier.STREAM_CAPABLE -> new StreamValidatorAnnotationProcessor(),
      NameIdentifier.TAIL_REC -> new TailRecValidatorAnnotationProcessor(),
      NameIdentifier.SINCE -> new SinceAnnotationProcessor(),
      NameIdentifier.LABELS -> new LabelsAnnotationProcessor(),
      NameIdentifier.INTERNAL_ANNOTATION -> new InternalAnnotationProcessor(),
      NameIdentifier.METADATA_ANNOTATION -> new MetadataAnnotationProcessorScopePhase(),
      NameIdentifier.UNTRUSTED_CODE_ANNOTATION -> new UntrustedCodeAnnotationProcessor(),
      NameIdentifier.INTERCEPTOR_ANNOTATION -> new InterceptorValidatorAnnotationProcessor(),
      NameIdentifier.DATA_FORMAT_MODULE_ANNOTATION -> new DataFormatExtensionAnnotationProcessor())
  }
}

trait AnnotationContext extends DWAnnotationContext {
  def messageCollector: MessageCollector

  override def getMessageCollector: ApiMessageCollector = {
    messageCollector
  }
}

class DefaultAnnotationContext(val parsingContext: ParsingContext) extends AnnotationContext {

  override def messageCollector: MessageCollector = parsingContext.messageCollector
}

sealed trait ProcessingPhase[Context <: AnnotationContext] {}

object ParsingProcessingPhase extends ProcessingPhase[ParsingPhaseAnnotationContext]

object CanonicalProcessingPhase extends ProcessingPhase[CanonicalPhaseAnnotationContext]

object ScopeProcessingPhase extends ProcessingPhase[ScopePhaseAnnotationContext]

object TypeProcessingPhase extends ProcessingPhase[TypePhaseAnnotationContext]

case class ParsingPhaseAnnotationContext(astNavigator: AstNavigator, messageCollector: MessageCollector) extends AnnotationContext

case class CanonicalPhaseAnnotationContext(astNavigator: AstNavigator, dummyScopeNavigator: ScopesNavigator, messageCollector: MessageCollector) extends AnnotationContext

case class ScopePhaseAnnotationContext(astNavigator: AstNavigator, scopeNavigator: ScopesNavigator, messageCollector: MessageCollector) extends AnnotationContext

case class TypePhaseAnnotationContext(astNavigator: AstNavigator, scopeNavigator: ScopesNavigator, typeGraph: TypeGraph, messageCollector: MessageCollector) extends AnnotationContext

abstract class AbstractParsingAnnotationProcessor extends AnnotationProcessor[ParsingPhaseAnnotationContext] {
  final override def phase: ProcessingPhase[ParsingPhaseAnnotationContext] = ParsingProcessingPhase

  final override def getProcessingPhase: DWProcessingPhase = DWParsingProcessingPhase

  final override def annotationContext(astNavigator: AstNavigator, scopesNavigator: Option[ScopesNavigator], typeGraph: Option[TypeGraph], messageCollector: MessageCollector): ParsingPhaseAnnotationContext = ParsingPhaseAnnotationContext(astNavigator, messageCollector)
}

abstract class AbstractCanonicalAnnotationProcessor extends AnnotationProcessor[CanonicalPhaseAnnotationContext] {
  final override def phase: ProcessingPhase[CanonicalPhaseAnnotationContext] = CanonicalProcessingPhase

  final override def getProcessingPhase: DWProcessingPhase = DWCanonicalProcessingPhase

  final override def annotationContext(astNavigator: AstNavigator, scopesNavigator: Option[ScopesNavigator], typeGraph: Option[TypeGraph], messageCollector: MessageCollector): CanonicalPhaseAnnotationContext = CanonicalPhaseAnnotationContext(astNavigator, scopesNavigator.get, messageCollector)

}

abstract class AbstractScopeAnnotationProcessor extends AnnotationProcessor[ScopePhaseAnnotationContext] {
  final override def phase: ProcessingPhase[ScopePhaseAnnotationContext] = ScopeProcessingPhase

  final override def getProcessingPhase: DWProcessingPhase = DWScopeProcessingPhase

  final override def annotationContext(astNavigator: AstNavigator, scopesNavigator: Option[ScopesNavigator], typeGraph: Option[TypeGraph], messageCollector: MessageCollector): ScopePhaseAnnotationContext = ScopePhaseAnnotationContext(astNavigator, scopesNavigator.get, messageCollector)
}

abstract class AbstractTypeAnnotationProcessor extends AnnotationProcessor[TypePhaseAnnotationContext] {
  final override def phase: ProcessingPhase[TypePhaseAnnotationContext] = TypeProcessingPhase

  final override def getProcessingPhase: DWProcessingPhase = DWTypeProcessingPhase

  final override def annotationContext(astNavigator: AstNavigator, scopesNavigator: Option[ScopesNavigator], typeGraph: Option[TypeGraph], messageCollector: MessageCollector): TypePhaseAnnotationContext = TypePhaseAnnotationContext(astNavigator, scopesNavigator.get, typeGraph.get, messageCollector)
}
