package org.mule.weave.v2.parser.location

import org.mule.weave.v2.api.tooling.location.ResourceIdentifier
import org.mule.weave.v2.api.tooling.location.{ Location => ApiLocation }
import org.mule.weave.v2.api.tooling.location.{ Position => ApiPosition }
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

case class WeaveLocation(startPosition: Position, endPosition: Position, resourceName: NameIdentifier)
    extends ApiLocation
    with Location {

  def source: () => String = {
    () => sourceLines().mkString("...\n")
  }

  def sourceLines(): Seq[String] = {
    if (startPosition.line != endPosition.line) {
      Seq(startPosition.source(), endPosition.source())
    } else {
      Seq(startPosition.source())
    }
  }

  def resourceWithLocation(): String =
    resourceName.name + s" (line: ${startPosition.line}, column:${startPosition.column})"

  def contains(index: Int): Boolean = {
    startPosition.index <= index && endPosition.index >= index
  }

  def contains(location: WeaveLocation): Boolean = {
    contains(location.startPosition.index) && contains(location.endPosition.index)
  }

  def noErrorMarkerLocationString: String = {
    if (startPosition == null || startPosition.line <= 0) {
      "Unknown location"
    } else {
      val line = startPosition.line
      val endLine = endPosition.line

      val lineNumLength = Math.max(line.toString.length, endLine.toString.length)
      val paddedLine = leftPad(line, lineNumLength)
      var str = if (endLine > line + 1) {
        val paddedEndLine = leftPad(endLine, lineNumLength)
        s"$paddedLine| ${startPosition.source()}${" " * lineNumLength}|  ...\n$paddedEndLine| ${endPosition.source()}"
      } else {
        s"$paddedLine| ${this.source()}"
      }

      val endsWithNewLine = str.endsWith("\n") || str.endsWith(System.lineSeparator())
      str = if (endsWithNewLine) str else str + "\n"
      val startsWithNewLine = str.startsWith("\n") || str.startsWith(System.lineSeparator())
      str = if (startsWithNewLine) str else "\n" + str
      str
    }
  }

  override def locationString: String = {
    if (startPosition == null || startPosition.line <= 0) {
      "Unknown location"
    } else {
      val line = startPosition.line
      val endLine = endPosition.line
      val startCol = startPosition.column

      val lineNumLength = Math.max(line.toString.length, endLine.toString.length)
      val errorMarker = if (startCol > 0 && line == endLine) {
        " " * (lineNumLength + 1 + startCol) + ("^" * Math.max(endPosition.column - startCol, 1))
      } else {
        ""
      }
      s"$noErrorMarkerLocationString$errorMarker"
    }
  }

  def leftPad(str: Any, len: Int): String = {
    s"%${len}s".format(str.toString)
  }

  def length: Int = endPosition.index - startPosition.index

  override def getStartPosition: ApiPosition = {
    startPosition
  }

  override def getEndPosition: ApiPosition = {
    endPosition
  }

  override def getResourceIdentifier: ResourceIdentifier = {
    resourceName
  }

  override def toStringRepresentation: String = {
    locationString
  }
}
