package org.mule.weave.v2.sdk

import org.mule.weave.v2.annotations.WeaveApi

import java.io.{ File, FileInputStream, InputStream }
import java.net.URL
import org.mule.weave.v2.utils.WeaveConstants

import scala.io.Source

object WeaveResourceFactory {

  @WeaveApi(Seq("data-weave-agent"))
  def fromContent(content: String): WeaveResource = {
    fromContent("Unknown", content)
  }

  def fromContent(url: String, content: String): WeaveResource = {
    WeaveResource(url, content)
  }

  def fromBinaryInputStream(binaryContent: InputStream, sourceWeaveResource: WeaveResource): BinaryWeaveResource = {
    try {
      BinaryWeaveResource(binaryContent.readAllBytes(), sourceWeaveResource)
    } finally {
      if (binaryContent != null)
        binaryContent.close()
    }
  }

  def fromBinaryFile(binaryFile: File, sourceWeaveResource: WeaveResource): BinaryWeaveResource = {
    val inputStream = new FileInputStream(binaryFile)
    try {
      BinaryWeaveResource(inputStream.readAllBytes(), sourceWeaveResource)
    } finally {
      if (inputStream != null)
        inputStream.close()
    }
  }

  def fromFile(file: File): WeaveResource = {
    val source = Source.fromFile(file, WeaveConstants.default_encoding)
    try {
      WeaveResource(file.toURI.toURL.toExternalForm, source.mkString)
    } finally {
      source.close()
    }
  }

  def fromInputStream(url: String, content: InputStream): WeaveResource = {
    val source = Source.fromInputStream(content, WeaveConstants.default_encoding)
    try {
      WeaveResource(url, source.mkString)
    } finally {
      source.close()
    }
  }

  def fromUrl(url: URL): WeaveResource = {
    val source = Source.fromInputStream(url.openStream(), WeaveConstants.default_encoding)
    try {
      WeaveResource(url.toExternalForm, source.mkString)
    } finally {
      source.close()
    }
  }
}
