package org.mule.weave.v2.ts

import org.mule.weave.v2.grammar.literals.TypeLiteral
import org.mule.weave.v2.parser.ast.CommentNode
import org.mule.weave.v2.parser.ast.CommentType
import org.mule.weave.v2.parser.ast.structure.BooleanNode
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.structure.NumberNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.structure.schema.SchemaNode
import org.mule.weave.v2.parser.ast.structure.schema.SchemaPropertyNode
import org.mule.weave.v2.parser.ast.types.FunctionParameterTypeNode
import org.mule.weave.v2.parser.ast.types.FunctionTypeNode
import org.mule.weave.v2.parser.ast.types.IntersectionTypeNode
import org.mule.weave.v2.parser.ast.types.KeyTypeNode
import org.mule.weave.v2.parser.ast.types.KeyValueTypeNode
import org.mule.weave.v2.parser.ast.types.LiteralTypeNode
import org.mule.weave.v2.parser.ast.types.NameTypeNode
import org.mule.weave.v2.parser.ast.types.NameValueTypeNode
import org.mule.weave.v2.parser.ast.types.ObjectTypeNode
import org.mule.weave.v2.parser.ast.types.TypeParameterNode
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.parser.ast.types.TypeSelectorNode
import org.mule.weave.v2.parser.ast.types.UnionTypeNode
import org.mule.weave.v2.parser.ast.types.WeaveTypeNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.UnknownLocation
import org.mule.weave.v2.utils.StringEscapeHelper

class WeaveTypeNodesConverter {

  def toSchema(metadataConstraints: Seq[MetadataConstraint]): Option[SchemaNode] = {
    if (metadataConstraints.nonEmpty) {
      Some(SchemaNode(metadataConstraints.map(annotation => SchemaPropertyNode(StringNode(annotation.name), StringNode(StringEscapeHelper.escapeString(annotation.value.toString)), None))))
    } else {
      None
    }
  }

  def toWeaveTypeNode(weaveType: WeaveType): WeaveTypeNode = {
    val schema = toSchema(weaveType.metadataConstraints())
    val result = weaveType match {
      case ObjectType(properties, close, ordered) => {
        ObjectTypeNode(properties.map(toWeaveTypeNode(_)), schema, None, close, ordered) //TODO
      }
      case NameType(value) => {
        value match {
          case Some(value) => if (value.ns.isDefined) {
            NameTypeNode(Some(value.name), Some(NamespaceNode(NameIdentifier(value.ns.get))))
          } else {
            NameTypeNode(Some(value.name), None)
          }
          case None => NameTypeNode(None, None)
        }
      }
      case KeyValuePairType(key, value, optional, repeated) => {
        KeyValueTypeNode(toWeaveTypeNode(key), toWeaveTypeNode(value), repeated, optional)
      }
      case KeyType(name, attrs) => {
        KeyTypeNode(
          toWeaveTypeNode(name).asInstanceOf[NameTypeNode],
          attrs.map({
            case NameValuePairType(name, value, optional) => {
              NameValueTypeNode(toWeaveTypeNode(name).asInstanceOf[NameTypeNode], toWeaveTypeNode(value), optional)
            }
          }))
      }
      case TypeParameter(name: String, top, _, _, _) => TypeParameterNode(NameIdentifier(name), top.map(b => toWeaveTypeNode(b)))
      case FunctionType(_, args, returnType, _, _, _) => {
        val arguments: Seq[FunctionParameterTypeNode] = args.zipWithIndex.map((indexedType) => {
          val argName: String = indexedType._1.name
          val argType: WeaveTypeNode = toWeaveTypeNode(indexedType._1.wtype)
          FunctionParameterTypeNode(Some(NameIdentifier(argName)), argType, indexedType._1.optional)
        })
        FunctionTypeNode(arguments, toWeaveTypeNode(returnType), schema)
      }
      case UnionType(of) => {
        if (of.size <= 1) {
          val innerType = of.headOption.getOrElse(throw new RuntimeException(s"WeaveType `${weaveType}` is not supported. Please report this as a bug with the script."))
          weaveType.metadataConstraints().foreach(constraint => innerType.withMetadataConstraint(constraint))
          toWeaveTypeNode(innerType)
        } else {
          UnionTypeNode(of.map(toWeaveTypeNode), schema)
        }
      }
      case IntersectionType(of) => {
        if (of.size <= 1) {
          val innerType = of.headOption.getOrElse(throw new RuntimeException(s"WeaveType `${weaveType}` is not supported. Please report this as a bug with the script."))
          weaveType.metadataConstraints().foreach(constraint => innerType.withMetadataConstraint(constraint))
          toWeaveTypeNode(innerType)
        } else {
          IntersectionTypeNode(of.map(toWeaveTypeNode), schema)
        }
      }

      case StringType(None)         => TypeReferenceNode(NameIdentifier(TypeLiteral.STRING_TYPE_NAME), None, schema)
      case BooleanType(None, _)     => TypeReferenceNode(NameIdentifier(TypeLiteral.BOOLEAN_TYPE_NAME), None, schema)
      case NumberType(None)         => TypeReferenceNode(NameIdentifier(TypeLiteral.NUMBER_TYPE_NAME), None, schema)
      case RangeType()              => TypeReferenceNode(NameIdentifier(TypeLiteral.RANGE_TYPE_NAME), None, schema)
      case NamespaceType(_, _)      => TypeReferenceNode(NameIdentifier(TypeLiteral.NAME_SPACE_TYPE_NAME), None, schema)
      case UriType(_)               => TypeReferenceNode(NameIdentifier(TypeLiteral.URI_TYPE_NAME), None, schema)
      case DateTimeType()           => TypeReferenceNode(NameIdentifier(TypeLiteral.DATETIME_TYPE_NAME), None, schema)
      case LocalDateTimeType()      => TypeReferenceNode(NameIdentifier(TypeLiteral.LOCALDATETIME_TYPE_NAME), None, schema)
      case LocalDateType()          => TypeReferenceNode(NameIdentifier(TypeLiteral.DATE_TYPE_NAME), None, schema)
      case LocalTimeType()          => TypeReferenceNode(NameIdentifier(TypeLiteral.LOCALTIME_TYPE_NAME), None, schema)
      case TimeType()               => TypeReferenceNode(NameIdentifier(TypeLiteral.TIME_TYPE_NAME), None, schema)
      case TimeZoneType()           => TypeReferenceNode(NameIdentifier(TypeLiteral.TIMEZONE_TYPE_NAME), None, schema)
      case PeriodType()             => TypeReferenceNode(NameIdentifier(TypeLiteral.PERIOD_TYPE_NAME), None, schema)
      case BinaryType()             => TypeReferenceNode(NameIdentifier(TypeLiteral.BINARY_TYPE_NAME), None, schema)
      case NullType()               => TypeReferenceNode(NameIdentifier(TypeLiteral.NULL_TYPE_NAME), None, schema)
      case AnyType()                => TypeReferenceNode(NameIdentifier(TypeLiteral.ANY_TYPE_NAME), None, schema)
      case RegexType()              => TypeReferenceNode(NameIdentifier(TypeLiteral.REGEX_TYPE_NAME), None, schema)
      case NothingType()            => TypeReferenceNode(NameIdentifier(TypeLiteral.NOTHING_TYPE_NAME), None, schema)
      case ArrayType(NothingType()) => TypeReferenceNode(NameIdentifier(TypeLiteral.ARRAY_TYPE_NAME), None, schema)

      case ArrayType(of)            => TypeReferenceNode(NameIdentifier(TypeLiteral.ARRAY_TYPE_NAME), Some(Seq(toWeaveTypeNode(of))), schema)
      case TypeType(of)             => TypeReferenceNode(NameIdentifier(TypeLiteral.TYPE_TYPE_NAME), Some(Seq(toWeaveTypeNode(of))), schema)

      case SimpleReferenceType(name, typeParams, _) => {
        TypeReferenceNode(name, typeParams.map((wts) => wts.map((wt) => toWeaveTypeNode(wt))))
      }

      case TypeSelectorType(nsPrefix, selectorName, referencedType, _) => {
        TypeSelectorNode(NameNode(StringNode(selectorName), nsPrefix), toWeaveTypeNode(referencedType))
      }

      case NumberType(Some(value))     => LiteralTypeNode(NumberNode(value), schema)
      case StringType(Some(value))     => LiteralTypeNode(StringNode(value).withQuotation('"'), schema)
      case BooleanType(Some(value), _) => LiteralTypeNode(BooleanNode(value.toString), schema)

      case _                           => throw new RuntimeException(s"WeaveType `${weaveType}` is not supported. Please report this as a bug with the script.")
    }

    if (weaveType.location() == UnknownLocation)
      result._location = None
    else
      result._location = Some(weaveType.location())

    if (weaveType.getDocumentation().isDefined) {
      result.addComment(CommentNode(weaveType.getDocumentation().get, CommentType.DocComment))
      result
    } else
      result
  }
}

object WeaveTypeNodesConverter {
  def toWeaveTypeNode(weaveType: WeaveType): WeaveTypeNode = {
    new WeaveTypeNodesConverter().toWeaveTypeNode(weaveType)
  }
}
