package org.mule.weave.v2.parser

import org.mule.weave.v2.grammar.Grammar
import org.mule.weave.v2.imports.UnusedImportsPhase
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.phase._
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.visibility.AccessCheckPhase
import org.parboiled2.ParseError
import org.parboiled2.Parser.DeliveryScheme.Either

object ModuleParser {

  var counter: Iterator[Int] = Stream.from(0).iterator

  private val parsingValidations: Array[AstNodeVerifier] = MappingParser.parsingValidations :+ new ValidModuleDirectivesValidation()

  private val compilationTransformations: Array[AstNodeTransformer] = Array(new ImplicitImportsTransformer())

  def parsingPhase(): CompilationPhase[ParsingContentInput, ParsingResult[ModuleNode]] = {
    ParsingPhase[ModuleNode](parse)
      .chainWith(new AstNodeVerificationPhase[ModuleNode, ParsingResult[ModuleNode]](parsingValidations))
      .chainWith(new ParsingAnnotationProcessorPhase[ModuleNode, ParsingResult[ModuleNode]]())
  }

  def scopePhase(): CompilationPhase[ParsingContentInput, ScopeGraphResult[ModuleNode]] = {
    canonicalAstPhase().chainWith(scopePhasePhases())
  }

  def scopePhasePhases(): CompilationPhase[AstNodeResultAware[ModuleNode] with ParsingContentAware, ScopeGraphResult[ModuleNode]] = {
    ScopeGraphPhase[ModuleNode]()
      .chainWith(new ImportsValidation[ModuleNode, ScopeGraphResult[ModuleNode]]())
      .chainWith(new MaterializeVariableMarkerPhase[ModuleNode, ScopeGraphResult[ModuleNode]]())
      .chainWith(new MaterializeTypeMarkerPhase[ModuleNode, ScopeGraphResult[ModuleNode]]())
      .chainWith(new StreamingCapableVariableMarkerPhase[ModuleNode, ScopeGraphResult[ModuleNode]]())
      .chainWith(new TailRecursiveMarkerPhase[ModuleNode, ScopeGraphResult[ModuleNode]]())
      .chainWith(new AnnotationArgumentNodeDefaultValuePhase[ModuleNode, ScopeGraphResult[ModuleNode]]())
      .chainWith(new ScopeAnnotationProcessingPhase[ModuleNode, ScopeGraphResult[ModuleNode]]())
      .chainWith(new MetadataKeyNodeInjectorPhase[ModuleNode, ScopeGraphResult[ModuleNode]])
      .chainWith(new MetadataTypeNodeInjectorPhase[ModuleNode, ScopeGraphResult[ModuleNode]])
      .chainWith(new AccessCheckPhase[ModuleNode, ScopeGraphResult[ModuleNode]]())
      .chainWith(new UnusedImportsPhase[ModuleNode, ScopeGraphResult[ModuleNode]]())
      .chainWith(new TypeSelectorValidation[ModuleNode, ScopeGraphResult[ModuleNode]]())
  }

  def canonicalAstPhase(): CompilationPhase[ParsingContentInput, ParsingResult[ModuleNode]] = {
    parsingPhase()
      .chainWith(canonicalPhasePhases())
  }

  def canonicalPhasePhases(): CompilationPhase[ParsingResult[ModuleNode], ParsingResult[ModuleNode]] = {
    new VersionCheckerPhase[ModuleNode]()
      .chainWith(new FunctionAggregationPhase[ModuleNode]())
      .chainWith(new AstNodeTransformationPhase[ModuleNode, ParsingResult[ModuleNode]](compilationTransformations))
      .chainWith(new DummyScopesNavigatorPhases[ModuleNode])
      .chainWith(new CanonicalAnnotationProcessingPhase[ModuleNode, ParsingResultWithDummyScopeNavigator[ModuleNode]]())
      .chainWith(new ImplicitFunctionTransformer[ModuleNode, ParsingResultWithDummyScopeNavigator[ModuleNode]]())
  }

  def typeCheckPhase(): CompilationPhase[ParsingContentInput, TypeCheckingResult[ModuleNode]] = {
    scopePhase().chainWith(typeCheckPhasePhases())
  }

  def typeCheckPhasePhases(): CompilationPhase[ScopeGraphResult[ModuleNode], TypeCheckingResult[ModuleNode]] = {
    new TypeCheckingPhase[ModuleNode]()
      .chainWith(new EqualityTypeChecker[ModuleNode, TypeCheckingResult[ModuleNode]]())
      .chainWith(new AnnotationParameterValidation[ModuleNode, TypeCheckingResult[ModuleNode]]())
      .chainWith(new TypeReferenceGenericsChecker[ModuleNode, TypeCheckingResult[ModuleNode]]())
      .chainWith(new TypeAnnotationProcessingPhase[ModuleNode, TypeCheckingResult[ModuleNode]]())
  }

  def parse[T](phase: CompilationPhase[ParsingContentInput, T], input: WeaveResource, parsingContext: ParsingContext): PhaseResult[T] = {
    val parserInput = ParsingContentInput(input, parsingContext.nameIdentifier, SafeStringBasedParserInput(input.content()))
    phase.call(parserInput, parsingContext)
  }

  private def parse(parserInput: ParsingContentInput, context: ParsingContext): Either[ParseError, ModuleNode] = {
    val parser = new Grammar(parserInput.input, parserInput.nameIdentifier, context.errorTrace, context.attachDocumentation)
    parser.module.run()
  }
}
