package org.mule.weave.v2.ts

import org.mule.weave.v2.grammar.ValueSelectorOpId
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.QName
import org.mule.weave.v2.parser.ast.header.directives.NamespaceDirective
import org.mule.weave.v2.parser.ast.operators.BinaryOpNode
import org.mule.weave.v2.parser.ast.selectors.NullSafeNode
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.scope.Reference
import org.mule.weave.v2.scope.ScopesNavigator
import org.mule.weave.v2.sdk.selectors.ValueSelectorCustomTypeResolver
import org.mule.weave.v2.ts.TypeHelper.canBeAssignedTo

import scala.annotation.tailrec

trait SourceReference {
  val reference: Reference

  /**
    * Get the positive constraint for the referenced
    * variable, given the type of the referenced value.
    *
    * @param orig The type to constraint the referenced value with
    * @param node The typenode to typecheck on
    * @param ctx The TypeResolutionContect to typecheck on
    * @return The positive constraint on the referenced variable
    */
  def createPositiveConstraint(orig: WeaveType, node: TypeNode, ctx: WeaveTypeResolutionContext): VariableConstraint = {
    createConstraintsAndSelector(orig, node, ctx)._1
  }

  /**
    * Get the negative constraint for the referenced
    * variable, given the type of the referenced value.
    *
    * @param orig The type to constraint the referenced value with
    * @param node The typenode to typecheck on
    * @param ctx The TypeResolutionContect to typecheck on
    * @return The negative constraint on the referenced variable
    */
  def createNegativeConstraint(orig: WeaveType, node: TypeNode, ctx: WeaveTypeResolutionContext): VariableConstraint = {
    createConstraintsAndSelector(orig, node, ctx)._2
  }

  /**
    * Creates all constraints and the selector.
    *
    * This method optimizes the creation, since it performs a single branch of recursion.
    *
    * @param orig The type to constraint with
    * @param node The typenode to typecheck on
    * @param ctx The TypeResolutionContect to typecheck on
    * @return
    */
  def createConstraintsAndSelector(orig: WeaveType, node: TypeNode, ctx: WeaveTypeResolutionContext): (VariableConstraint, VariableConstraint, WeaveType => Option[WeaveType])
}

object SourceReference {
  def apply(node: AstNode, scope: ScopesNavigator): Option[SourceReference] = {
    node match {
      case VariableReferenceNode(variable, _) => scope.resolveVariable(variable).map(VariableReference)
      case BinaryOpNode(ValueSelectorOpId, upNode, NameNode(StringNode(field, _), ns, _), _) =>
        val uri = ns match {
          case Some(NamespaceNode(nsId: NameIdentifier)) =>
            scope
              .resolveVariable(nsId)
              .flatMap(ref => {
                ref.scope.astNavigator().parentWithType(ref.referencedNode, classOf[NamespaceDirective])
              })
              .map(nsDir => {
                nsDir.uri.literalValue
              })
          case _ => None
        }
        val upSource = SourceReference(upNode, scope)
        upSource.map(SelectorReference(_, field, uri))
      case NullSafeNode(selector, _) => SourceReference(selector, scope)
      case _                         => None
    }
  }
}

case class VariableReference(varRef: Reference) extends SourceReference {
  override val reference: Reference = varRef

  override def createConstraintsAndSelector(orig: WeaveType, node: TypeNode, ctx: WeaveTypeResolutionContext): (VariableConstraint, VariableConstraint, WeaveType => Option[WeaveType]) = {
    (IsConstraint(orig), NotConstraint(orig), (a) => Some(a))
  }
}

case class SelectorReference(sourceReference: SourceReference, name: String, namespace: Option[String]) extends SourceReference {
  override lazy val reference: Reference = sourceReference.reference

  private def wrapConditionalConstraint(innerConstraint: VariableConstraint, selector: WeaveType => Option[WeaveType]) = {
    ConditionalConstraint(
      innerConstraint,
      (_, incomingType, ctx) => conditionalConstraintCondition(selector, innerConstraint, incomingType, ctx))
  }

  @tailrec
  private def conditionalConstraintCondition(selector: WeaveType => Option[WeaveType], const: VariableConstraint, incomingType: WeaveType, ctx: WeaveTypeResolutionContext): Boolean = {
    val maybeInnerType = selector(incomingType)
    maybeInnerType match {
      case Some(innerType) if canBeAssignedTo(innerType, ObjectType(), ctx) =>
        namespace match {
          case Some(_) => true
          case None =>
            innerType match {
              case rt: ReferenceType => conditionalConstraintCondition(selector, const, rt.resolveType(), ctx)
              case UnionType(of) => {
                val selectedPropsByPart = of.map(TypeHelper.selectPropertyPairs(_, name))
                selectedPropsByPart.forall(_.size <= 1)
              }
              // Only validate the constraint if it's an object with one matching key
              case _ => TypeHelper.selectPropertyPairs(innerType, name).size == 1
            }
        }
      case _ => false
    }
  }

  override def createConstraintsAndSelector(orig: WeaveType, node: TypeNode, ctx: WeaveTypeResolutionContext): (VariableConstraint, VariableConstraint, WeaveType => Option[WeaveType]) = {
    val qname = namespace match {
      case Some(_) => QName(name, namespace)
      case None    => QName.matchingAllNs(name)
    }
    val innerType = ObjectType(Seq(KeyValuePairType(KeyType(NameType(Some(qname))), orig)))

    val (innerPosConstraint, innerNegConstraint, innerSelector) = sourceReference.createConstraintsAndSelector(innerType, node, ctx)

    val posConstraint = wrapConditionalConstraint(innerPosConstraint, innerSelector)
    val negConstraint = wrapConditionalConstraint(innerNegConstraint, innerSelector)
    val selector = (a: WeaveType) => {
      val innerType: Option[WeaveType] = innerSelector(a)
      innerType
        .map(ValueSelectorCustomTypeResolver.resolve(_, NameType(Some(QName(name))), ctx, node, insideArray = false))
        .flatMap((r) => r.maybeResult())
    }
    (posConstraint, negConstraint, selector)
  }
}
