package org.mule.weave.v2.editor.composer

import org.mule.weave.v2.editor.ValidationMessage
import org.mule.weave.v2.parser.DeniedFunctionUsage
import org.mule.weave.v2.parser.Message
import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.functions.FunctionCallNode
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.ParsingResult
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.scope.ScopesNavigator

trait ComposerExpressionValidator {

  def validateComposerExpression(value: PhaseResult[ParsingResult[DocumentNode]], parsingContext: ParsingContext, allowedFunctions: Seq[String], allowedModules: Seq[String] = Seq.empty): ComposerValidationMessages = {
    if (value.hasResult()) {
      val theResult = value.getResult()
      val documentNode: DocumentNode = theResult.astNode
      validateAllowedFunctions(documentNode, parsingContext, allowedFunctions, allowedModules)
    }
    toComposerValidationMessages(value, parsingContext.nameIdentifier.name)
  }

  private def validateAllowedFunctions(documentNode: DocumentNode, parsingContext: ParsingContext, allowedFunctions: Seq[String], allowedModules: Seq[String]): Unit = {
    val scopesNavigator = ScopesNavigator(documentNode, parsingContext)

    def checkAccess(ref: NameIdentifier, location: WeaveLocation): Unit = {
      val maybeReference = scopesNavigator.resolveVariable(ref)
      if (maybeReference.isDefined) {
        val fqn = maybeReference.get.fqnReferenceName.name
        val maybeModuleSource = maybeReference.get.moduleSource
        if (!allowedFunctions.contains(fqn)) {
          if (maybeModuleSource.isDefined) {
            val moduleSource = maybeModuleSource.get.name
            if (!allowedModules.contains(moduleSource)) {
              parsingContext.messageCollector.error(DeniedFunctionUsage(ref.name), location)
            }
          } else {
            parsingContext.messageCollector.error(DeniedFunctionUsage(ref.name), location)
          }
        }
      }
    }

    if (allowedFunctions.nonEmpty || allowedModules.nonEmpty) {
      AstNodeHelper.traverse(
        documentNode.root, {
          case fcn @ FunctionCallNode(vrn: VariableReferenceNode, _, _, _) =>
            checkAccess(vrn.variable, fcn.location())
            true
          case _ =>
            true
        })
    }
  }

  private def toComposerValidationMessages(value: PhaseResult[ParsingResult[DocumentNode]], name: String): ComposerValidationMessages = {
    val messageCollector: MessageCollector = value.messages()
    val errorMessages: Seq[(WeaveLocation, Message)] = messageCollector.errorMessages
    val warningMessages: Seq[(WeaveLocation, Message)] = messageCollector.warningMessages
    val warnings: Seq[ValidationMessage] = warningMessages.filter(wm => wm._1.resourceName.name == name).map(pair => {
      ValidationMessage(pair._1, pair._2)
    })

    val errors: Seq[ValidationMessage] = errorMessages.filter(wm => wm._1.resourceName.name == name).map(pair => {
      ValidationMessage(pair._1, pair._2)
    })

    ComposerValidationMessages(warnings, errors)
  }
}
