package org.mule.weave.v2.sdk

import org.mule.weave.v2.parser.ast.variables.NameIdentifier

/**
  * Module resolver that tries to resolve the module with any of its inner resolves but in order
  */
class ChainedWeaveResourceResolver(chain: Seq[WeaveResourceResolver]) extends WeaveResourceResolver {

  override def resolve(name: NameIdentifier): Option[WeaveResource] = {
    val resolvers: Iterator[WeaveResourceResolver] = chain.iterator
    while (resolvers.hasNext) {
      val resolver = resolvers.next()
      val resolved: Option[WeaveResource] = resolver.resolve(name)
      if (resolved.isDefined) {
        return resolved
      }
    }
    None
  }

  override def resolvePath(path: String): Option[WeaveResource] = {
    val resolvers: Iterator[WeaveResourceResolver] = chain.iterator
    while (resolvers.hasNext) {
      val resolver = resolvers.next()
      val resolved: Option[WeaveResource] = resolver.resolvePath(path)
      if (resolved.isDefined) {
        return resolved
      }
    }
    None
  }

  override def resolveBinary(name: NameIdentifier): Option[BinaryWeaveResource] = {
    val resolvers: Iterator[WeaveResourceResolver] = chain.iterator
    while (resolvers.hasNext) {
      val resolver = resolvers.next()
      val resolved: Option[BinaryWeaveResource] = resolver.resolveBinary(name)
      if (resolved.isDefined) {
        return resolved
      }
    }
    None
  }

  override def resolveBinaryPath(path: String): Option[BinaryWeaveResource] = {
    val resolvers: Iterator[WeaveResourceResolver] = chain.iterator
    while (resolvers.hasNext) {
      val resolver = resolvers.next()
      val resolved: Option[BinaryWeaveResource] = resolver.resolveBinaryPath(path)
      if (resolved.isDefined) {
        return resolved
      }
    }
    None
  }

  override def resolveAll(name: NameIdentifier): Seq[WeaveResource] = {
    chain.flatMap(_.resolveAll(name))
  }

  override def canResolveResource(name: NameIdentifier): Boolean = chain.exists(_.canResolveResource(name))
}

object CompositeWeaveResourceResolver {
  def apply(parent: WeaveResourceResolver, child: WeaveResourceResolver): WeaveResourceResolver = {
    new ChainedWeaveResourceResolver(Seq(parent, child))
  }
}