package org.mule.weave.v2.scope

import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.functions.{ DoBlockNode, FunctionNode, OverloadedFunctionNode, UsingNode }
import org.mule.weave.v2.parser.ast.header.directives._
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.patterns._
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.parser.ast.updates.UpdateExpressionNode
import org.mule.weave.v2.parser.ast.variables.{ NameIdentifier, VariableReferenceNode }
import org.mule.weave.v2.parser.phase.{ ParsingContext, PhaseResult, ScopeGraphResult }
import org.mule.weave.v2.utils.LazyValRef

class VariableScopeFactory(context: ParsingContext) {
  def create(astNode: AstNode): VariableScope = {
    val factory = astNode match {
      case mn: ModuleNode =>
        val scope = VariableScopeBuilder(context, mn.name.name, astNode)
        mn.elements.foreach(x => visitNode(x, scope))
        scope
      case _ =>
        val scope = VariableScopeBuilder(context, astNode)
        visitNode(astNode, scope)
        scope
    }
    factory.build()
  }

  private def visitChildren(astNode: AstNode, scope: VariableScopeBuilder): Unit = {
    val children = astNode.children()
    var i = 0
    while (i < children.length) {
      visitNode(children(i), scope)
      i = i + 1
    }
  }

  private def visitNode(astNode: AstNode, scope: VariableScopeBuilder): Unit = {
    astNode match {
      case _: VersionDirective =>
      case vd @ VarDirective(varName, _, _, _) =>
        scope.addDeclaration(varName)
        visitChildren(vd, scope)
      case id @ ImportDirective(moduleName, _, codeAnnotations) =>
        val importedNameIdentifier = moduleName.elementName
        if (!importedNameIdentifier.equals(context.nameIdentifier)) { //If we are importing same we don't do anything
          val lazyImportedModule = LazyValRef(() => {
            val module: PhaseResult[ScopeGraphResult[ModuleNode]] = context.getScopeGraphForModule(importedNameIdentifier)
            if (module.hasResult()) {
              Option.apply(module.getResult().scope.rootScope)
            } else {
              Option.empty
            }
          })
          scope.addLazyImportedModule(id, lazyImportedModule)

          id.subElements.elements.foreach(element => {
            if (!element.elementName.equals(NameIdentifier.$star)) {
              scope.addReference(element.elementName)
            }
          })
        }
        visitNodes(codeAnnotations, scope)
      case TypeDirective(varName, typeParametersListNode, typeExpression, codeAnnotations) =>
        scope.addDeclaration(varName)

        val typeScope = scope.createChild(typeExpression)
        if (typeParametersListNode.isDefined) {
          typeParametersListNode.get.typeParameters.foreach(node => {
            typeScope.addDeclaration(node.name)
            visitNode(node, scope)
          })
        }
        visitNode(typeExpression, typeScope)
        visitNodes(codeAnnotations, scope)
      case trn @ TypeReferenceNode(_, typeParameters, schema, typeSchema, codeAnnotations) =>
        scope.addReference(trn.variable)

        if (typeParameters.isDefined) {
          visitNodes(typeParameters.get, scope)
        }
        visitOptionNode(schema, scope)
        visitOptionNode(typeSchema, scope)
        visitNodes(codeAnnotations, scope)
      case id: InputDirective =>
        scope.addDeclaration(id.variable)
        visitChildren(id, scope)
      case ad: AnnotationDirectiveNode =>
        scope.addDeclaration(ad.nameIdentifier)
        visitChildren(ad, scope)
      case AnnotationNode(name, args) =>
        scope.addReference(name)
        visitOptionNode(args, scope)
      case NamespaceDirective(prefix, _, codeAnnotations) =>
        scope.addDeclaration(prefix)
        visitNodes(codeAnnotations, scope)
      case nn: NamespaceNode =>
        scope.addReference(nn.prefix)
      case uen: UpdateExpressionNode =>
        val variableScope = scope.createChild(uen)
        variableScope.addDeclaration(uen.name)
        variableScope.addDeclaration(uen.indexId)
        visitNode(uen.selector, variableScope)
        visitNode(uen.updateExpression, variableScope)
        visitOptionNode(uen.condition, variableScope)
      case pn: PatternExpressionNode =>
        val expressionPatternScope: VariableScopeBuilder = scope.createChild(pn)
        pn match {
          case RegexPatternNode(pattern, name, onMatch) =>
            expressionPatternScope.addDeclaration(name)
            visitNode(pattern, expressionPatternScope)
            visitNode(onMatch, expressionPatternScope)
          case TypePatternNode(pattern, name, onMatch) =>
            visitNode(pattern, expressionPatternScope)
            expressionPatternScope.addDeclaration(name)
            visitNode(onMatch, expressionPatternScope)
          case LiteralPatternNode(pattern, name, onMatch) =>
            visitNode(pattern, expressionPatternScope)
            expressionPatternScope.addDeclaration(name)
            visitNode(onMatch, expressionPatternScope)
          case ExpressionPatternNode(pattern, name, onMatch) =>
            expressionPatternScope.addDeclaration(name)
            visitNode(pattern, expressionPatternScope)
            visitNode(onMatch, expressionPatternScope)
          case EmptyArrayPatternNode(onMatch) =>
            visitNode(onMatch, expressionPatternScope)
          case DeconstructArrayPatternNode(headNameIdentifier, tailNameIdentifier, onMatch) =>
            expressionPatternScope.addDeclaration(headNameIdentifier)
            expressionPatternScope.addDeclaration(tailNameIdentifier)
            visitNode(onMatch, expressionPatternScope)
          case DeconstructObjectPatternNode(headKeyNameIdentifier, headValueNameIdentifier, tailNameIdentifier, onMatch) =>
            expressionPatternScope.addDeclaration(headKeyNameIdentifier)
            expressionPatternScope.addDeclaration(headValueNameIdentifier)
            expressionPatternScope.addDeclaration(tailNameIdentifier)
            visitNode(onMatch, expressionPatternScope)
          case DefaultPatternNode(value, name) =>
            expressionPatternScope.addDeclaration(name)
            visitNode(value, expressionPatternScope)
          case _ =>
        }
      case FunctionDirectiveNode(varName, fn, codeAnnotation) =>
        scope.addDeclaration(varName)
        codeAnnotation.foreach(co => visitNode(co, scope))
        visitNode(fn, scope)
      case ModuleNode(name, elements) =>
        val child: VariableScopeBuilder = scope.createChild(name.name, astNode)
        elements.foreach(x => visitNode(x, child))
      case overloadedFunctionNode: OverloadedFunctionNode =>
        val children = overloadedFunctionNode.functions
        var i = 0
        while (i < children.length) {
          visitNode(children(i), scope)
          i = i + 1
        }
      case fn @ FunctionNode(arguments, body, returnType, typeParameterList) =>
        //Function has a scope with the function definition. All type parameters and function params and return type
        val functionScope = scope.createChild(fn)
        typeParameterList match {
          case None =>
          case Some(typeParametersList) =>
            functionScope.addDeclarations(typeParametersList.typeParameters.map(_.name))
        }
        arguments.paramList.foreach(argument => {
          argument.wtype match {
            case None =>
            case Some(argType) =>
              visitNode(argType, functionScope)
          }
        })

        arguments.paramList.foreach(argument => {
          argument.codeAnnotations.foreach(ann => visitNode(ann, functionScope))
        })

        returnType match {
          case None                 =>
          case Some(returnTypeNode) => visitNode(returnTypeNode, functionScope)
        }
        for (maybeDefaultValue <- arguments.paramList.map(_.defaultValue); defaultValue <- maybeDefaultValue) {
          visitNode(defaultValue, functionScope)
        }
        //End of function scope.
        //Function body is child of the function definition scope
        val funBodyScope: VariableScopeBuilder = functionScope.createChild(body)
        val argNames: Seq[NameIdentifier] = arguments.paramList.map(_.variable)
        funBodyScope.addDeclarations(argNames)
        visitNode(body, funBodyScope)
      case UsingNode(assignments, expr, codeAnnotations) =>
        var currentScope = scope
        assignments.assignmentSeq.foreach(assignment => {
          currentScope = currentScope.createChild(assignment)
          visitNode(assignment.value, currentScope)
          currentScope.addDeclaration(assignment.name)
        })
        val usingScope = currentScope.createChild(expr)
        visitNode(expr, usingScope)
        visitNodes(codeAnnotations, scope)
      case doBlock: DoBlockNode =>
        val doBlockScope = scope.createChild(doBlock)
        doBlock.header.children().foreach(x => visitNode(x, doBlockScope))
        visitNode(doBlock.body, doBlockScope)
        visitNodes(doBlock.codeAnnotations, scope)
      case ref: VariableReferenceNode =>
        scope.addReference(ref.variable)
        visitNodes(ref.codeAnnotations, scope)
      case node =>
        visitChildren(node, scope)
    }
  }

  private def visitNodes(nodes: Seq[AstNode], scope: VariableScopeBuilder): Unit = {
    nodes.foreach(co => {
      visitNode(co, scope)
    })
  }

  private def visitOptionNode(nodes: Option[AstNode], scope: VariableScopeBuilder): Unit = {
    nodes.foreach(co => {
      visitNode(co, scope)
    })
  }
}
