package org.mule.weave.v2.sdk

import java.net.URL
import java.util
import java.util.Collections
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.sdk.ClassLoaderWeaveResourceResolver.PATH_SEPARATOR
import org.mule.weave.v2.utils.{ WeaveConstants, WeaveFile }

import scala.collection.JavaConverters._
import scala.collection.mutable
import scala.io.Source

class ClassLoaderWeaveResourceResolver(classloaders: Seq[() => ClassLoader]) extends WeaveResourceResolver {

  override def resolve(name: NameIdentifier): Option[WeaveResource] = {
    val path = NameIdentifierHelper.toWeaveFilePath(name, ClassLoaderWeaveResourceResolver.PATH_SEPARATOR)
    resolvePath(path)
  }

  override def resolveBinary(name: NameIdentifier): Option[BinaryWeaveResource] = {
    val path = NameIdentifierHelper.toWeaveBinaryFilePath(name, ClassLoaderWeaveResourceResolver.PATH_SEPARATOR)
    resolveBinaryPath(path)
  }

  override def resolvePath(path: String): Option[WeaveResource] = {
    resolveUrl(path).map(url => toWeaveSourceResource(url))
  }

  override def resolveBinaryPath(path: String): Option[BinaryWeaveResource] = {
    val maybeSourceWeaveResource = resolveUrl(path.replace(WeaveFile.binaryFileExtension, WeaveFile.fileExtension))
      .map(url => toWeaveSourceResource(url))
    maybeSourceWeaveResource.flatMap(sourceWeaveResource =>
      resolveUrl(path).map(url => toWeaveBinaryResource(url, sourceWeaveResource)))
  }

  private def toWeaveSourceResource(element: URL) = {
    val source = Source.fromInputStream(element.openStream(), WeaveConstants.default_encoding)
    try {
      WeaveResource(element.toExternalForm, source.mkString)
    } finally {
      source.close()
    }
  }

  private def toWeaveBinaryResource(element: URL, sourceWeaveResource: WeaveResource) = {
    WeaveResourceFactory.fromBinaryInputStream(element.openStream(), sourceWeaveResource)
  }

  def resolveUrl(path: String): Option[URL] = {
    val cannonicalPath = if (NameIdentifierHelper.fileSeparator != ClassLoaderWeaveResourceResolver.PATH_SEPARATOR) {
      path.replace(NameIdentifierHelper.fileSeparator, ClassLoaderWeaveResourceResolver.PATH_SEPARATOR)
    } else path

    val weaveFilePath: String = if (cannonicalPath.startsWith(ClassLoaderWeaveResourceResolver.PATH_SEPARATOR)) cannonicalPath.substring(1) else cannonicalPath
    var resources: util.Enumeration[URL] = Collections.emptyEnumeration()
    val classLoadersIt = classloaders.toIterator
    while (classLoadersIt.hasNext && !resources.hasMoreElements) {
      val classLoader = classLoadersIt.next()()
      resources = classLoader.getResources(weaveFilePath)
    }
    if (resources.hasMoreElements) {
      Option.apply(resources.nextElement())
    } else {
      None
    }
  }

  override def resolveAll(name: NameIdentifier): Seq[WeaveResource] = {
    val path = NameIdentifierHelper.toWeaveFilePath(name, ClassLoaderWeaveResourceResolver.PATH_SEPARATOR)
    lookupResources(path)
  }

  def lookupResources(path: String): Seq[WeaveResource] = {
    val weaveFilePath: String = if (path.startsWith(ClassLoaderWeaveResourceResolver.PATH_SEPARATOR)) path.substring(1) else path
    var resources: util.Enumeration[URL] = Collections.emptyEnumeration();
    val classLoadersIt = classloaders.toIterator
    val result = mutable.ArrayBuffer[WeaveResource]()
    while (classLoadersIt.hasNext) {
      val classLoader = classLoadersIt.next()()
      resources = classLoader.getResources(weaveFilePath)
      result.++=(resources.asScala.map((resource) => {
        val source = Source.fromInputStream(resource.openStream(), "UTF-8")
        try {
          WeaveResource(resource.toExternalForm, source.mkString)
        } finally {
          source.close()
        }
      }))
    }
    result
  }

  override def canResolveResource(name: NameIdentifier): Boolean = {
    val path = NameIdentifierHelper.toWeaveFilePath(name, PATH_SEPARATOR)
    resolveUrl(path).isDefined
  }
}

object ClassLoaderWeaveResourceResolver {

  val PATH_SEPARATOR = "/"

  def apply(): ClassLoaderWeaveResourceResolver = {
    val mainClassloader = getClass.getClassLoader
    new ClassLoaderWeaveResourceResolver(Seq(() => mainClassloader, () => Thread.currentThread().getContextClassLoader))
  }

  def noContextClassloader(): ClassLoaderWeaveResourceResolver = {
    val mainClassloader = getClass.getClassLoader
    new ClassLoaderWeaveResourceResolver(Seq(() => mainClassloader))
  }

  def contextClassloaderOnly(): ClassLoaderWeaveResourceResolver = {
    new ClassLoaderWeaveResourceResolver(Seq(() => Thread.currentThread().getContextClassLoader))
  }

  def providedClassLoader(classLoaders: Seq[ClassLoader]): ClassLoaderWeaveResourceResolver = {
    val mainClassloader = getClass.getClassLoader
    val all = Seq(mainClassloader) ++ classLoaders
    new ClassLoaderWeaveResourceResolver(all.map(cl => () => cl))
  }

}
