package org.mule.weave.v2.ts

import org.mule.weave.v2.annotations.WeaveApi
import org.mule.weave.v2.parser.CloseDoesNotAllowExtraProperties
import org.mule.weave.v2.parser.CloseDoesNotAllowOpen
import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.MissingRequiredProperty
import org.mule.weave.v2.parser.NotEnoughArgumentMessage
import org.mule.weave.v2.parser.RepeatedFieldNotSupported
import org.mule.weave.v2.parser.TooManyArgumentMessage
import org.mule.weave.v2.parser.TypeMessage
import org.mule.weave.v2.parser.TypeMismatch
import org.mule.weave.v2.parser.ast.QName
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.listener.ParsingNotificationManager
import org.mule.weave.v2.utils.SeqUtils

import scala.annotation.tailrec
import scala.collection.mutable.ArrayBuffer

class TypeHelper(notificationManager: ParsingNotificationManager) {

  /**
    * Transforms a bottom constrain into a Top constrain.
    *
    * Literal types   -> Base Types "Shoki" -> String
    * Close Objects   -> Open Object
    * Array<Z>        -> Array<Top(Z)>
    * Union<X>        -> Union<Top(X)
    * Intersection<X> -> Intersection<Top(X)
    *
    * @param bottom The bottom type
    * @return
    */
  def toTopType(bottom: WeaveType): WeaveType = {
    bottom match {
      case obt: ObjectType      => obt.copy(close = false)
      case at: ArrayType        => at.copy(of = toTopType(at.of))
      case ut: UnionType        => ut.copy(of = ut.of.map(wt => toTopType(wt)))
      case it: IntersectionType => it.copy(of = it.of.map(wt => toTopType(wt)))
      case t                    => t.baseType()
    }
  }

  /**
    * Returns true iff the given types don't have any common values
    *
    * @param typeA The first type
    * @param typeB The second type
    * @return true iff typeA and typeB don't have any values in common
    */
  def areDisjointTypes(typeA: WeaveType, typeB: WeaveType): Boolean = {
    areEqualStructurally(NothingType(), resolveAlgebraicIntersection(typeA, typeB))
  }

  /**
    * Returns true if this type needs to be materialize
    *
    * @param weaveType The Type
    * @return True if it requires materialize
    */
  def requiredMaterialize(weaveType: WeaveType): Boolean = {
    WeaveTypeTraverse.treeExists(
      weaveType, {
      case _: ArrayType        => false
      case ot: ObjectType      => ot.properties.nonEmpty
      case StringType(_)       => false
      case AnyType()           => false
      case BooleanType(_, _)   => false
      case NumberType(_)       => false
      case RangeType()         => false
      case UriType(_)          => false
      case DateTimeType()      => false
      case LocalDateTimeType() => false
      case LocalDateType()     => false
      case LocalTimeType()     => false
      case TimeType()          => false
      case TimeZoneType()      => false
      case PeriodType()        => false
      case BinaryType()        => false
      case TypeType(_)         => false
      case RegexType()         => false
      case NullType()          => false
      case NothingType()       => false
      case _: FunctionType     => false
    },
      RecursionDetector[Boolean]((_, _) => true))
  }

  def selectProperty(weaveType: WeaveType, name: String): Option[WeaveType] = {
    selectPropertyPairs(weaveType, name).headOption.map(_.value)
  }

  def selectPropertyPairs(weaveType: WeaveType, qname: QName): Seq[KeyValuePairType] = {
    weaveType match {
      case ot: ObjectType =>
        ot.properties
          .filter(prop => {
            propQName(prop).exists(_.matchesPattern(qname))
          })
      case IntersectionType(of) =>
        resolveIntersection(of) match {
          case IntersectionType(of) =>
            val props = of.map(selectPropertyPairs(_, qname))
            props.reduce(intersectProperties)
          case otherType =>
            selectPropertyPairs(otherType, qname)
        }
      case ut: UnionType =>
        ut.of.flatMap(selectPropertyPairs(_, qname))
      case rt: ReferenceType => selectPropertyPairs(rt.resolveType(), qname)
      case TypeParameter(_, topType, bottomType, _, _) =>
        selectPropertyPairs(topType.orElse(bottomType).getOrElse(AnyType()), qname)
      case _ => Seq()
    }
  }

  def selectPropertyPairs(weaveType: WeaveType, name: String): Seq[KeyValuePairType] = {
    selectPropertyPairs(weaveType, QName.matchingAllNs(name))
  }

  def propQName(prop: KeyValuePairType): Option[QName] = {
    prop.key match {
      case KeyType(NameType(Some(propName)), _) =>
        val nameName = propName.copy()
        nameName.matchesAllNs = false
        Some(nameName)
      case _ => None
    }
  }

  def propName(prop: KeyValuePairType): Option[String] = {
    propQName(prop).map(propQName => propQName.name)
  }

  def isNullType(wtype: WeaveType): Boolean = {
    wtype match {
      case _: NullType       => true
      case rt: ReferenceType => isNullType(rt.resolveType())
      case _                 => false
    }
  }

  def isArithmeticType(expected: WeaveType): Boolean = {
    expected match {
      case _: UnionType | _: IntersectionType => true
      case _                                  => false
    }
  }

  @tailrec
  final def asFunctionType(weaveType: WeaveType): Option[FunctionType] = {
    weaveType match {
      case ft: FunctionType  => Some(ft)
      case rt: ReferenceType => asFunctionType(rt.resolveType())
      case it: IntersectionType =>
        resolveIntersection(it.of) match {
          case IntersectionType(_) => None
          case otherType           => asFunctionType(otherType)
        }
      case _ => None
    }
  }

  def getArrayType(weaveType: WeaveType): Option[WeaveType] = {
    weaveType match {
      case at: ArrayType     => Some(at.of)
      case rt: ReferenceType => getArrayType(rt.resolveType())
      case ut: UnionType =>
        val weaveTypes = ut.of.flatMap(getArrayType)
        if (weaveTypes.isEmpty) {
          None
        } else {
          Some(unify(weaveTypes))
        }
      case it: IntersectionType =>
        resolveIntersection(it.of) match {
          case IntersectionType(_) => None
          case otherType           => getArrayType(otherType)
        }
      case _ => None

    }
  }

  def isObjectType(wtype: WeaveType): Boolean = {
    wtype match {
      case _: ObjectType     => true
      case tr: ReferenceType => isObjectType(tr.resolveType())
      case _                 => false
    }
  }

  def isArrayType(wtype: WeaveType): Boolean = {
    wtype match {
      case _: ArrayType      => true
      case tr: ReferenceType => isArrayType(tr.resolveType())
      case _                 => false
    }
  }

  def isArrayType(wtype: WeaveType, of: Class[_ <: WeaveType]): Boolean = {
    wtype match {
      case obt: ArrayType    => isOfType(obt.of, of)
      case tr: ReferenceType => isArrayType(tr.resolveType(), of)
      case _                 => false
    }
  }

  def isOfType(wtype: WeaveType, typeType: Class[_ <: WeaveType]): Boolean = {
    wtype match {
      case tr: ReferenceType =>
        isOfType(tr.resolveType(), typeType)
      case tr: UnionType =>
        tr.of.forall(ut => isOfType(ut, typeType))
      case _ => typeType.isInstance(wtype)
    }
  }

  def isEmptyObject(wtype: WeaveType): Boolean = {
    wtype match {
      case obt: ObjectType   => obt.properties.isEmpty
      case tr: ReferenceType => isEmptyObject(tr.resolveType())
      case _                 => false
    }
  }

  /**
    * Collects all the type parameters
    *
    * @return
    */
  def collectTypeParameters(weaveType: WeaveType): Seq[TypeParameter] = {
    WeaveTypeTraverse.shallowCollectAll(weaveType, {
      case tp: TypeParameter => Seq(tp)
      case _                 => Seq()
    })
  }

  def collectAbstractTypeParameters(weaveType: WeaveType): Seq[TypeParameter] = {
    WeaveTypeTraverse.shallowCollectAll(weaveType, {
      case tp: TypeParameter if tp.isAbstract() => Seq(tp)
      case _                                    => Seq()
    })
  }

  /**
    * Cleanups the Type parameters
    *
    * @param wt The type to be cleaned
    * @return The cleaned type
    */
  def cleanupUnionTypeWithParameters(wt: WeaveType, typesToCheck: Seq[WeaveType]): WeaveType = {
    def exists(wt: WeaveType): Boolean = {
      wt match {
        case tp: TypeParameter =>
          if (typesToCheck.exists(typeToCheck => typeToCheck eq tp))
            tp.top.nonEmpty
          else
            true
        case rt: ReferenceType =>
          exists(rt.resolveType())
        case _ => true
      }
    }

    def doCleanupUnionTypeWithParameters(wt: WeaveType, typesToCheck: Seq[WeaveType], recursionDetector: RecursionDetector[WeaveType]): WeaveType = {
      WeaveTypeTraverse.treeMap(
        wt, {
        case UnionType(of) =>
          val filteredTypes = of.filter(wt => exists(wt))
          if (filteredTypes.nonEmpty) {
            unionWithoutSimplification(filteredTypes.map(doCleanupUnionTypeWithParameters(_, typesToCheck, recursionDetector)))
          } else {
            unionWithoutSimplification(of.map(doCleanupUnionTypeWithParameters(_, typesToCheck, recursionDetector)))
          }
        case ft: FunctionType =>
          ft // Function Types are the only ones that don't need to be cleaned up
      },

        recursionDetector)
    }

    doCleanupUnionTypeWithParameters(wt, typesToCheck, RecursionDetector((id, refResolver) => SimpleReferenceType(id, None, () => refResolver())))

  }

  def resolveIntersection(types: Seq[WeaveType]): WeaveType = {
    types.reduce((l, r) => {
      resolveIntersection(l, r, createDoubleRecursionIntersectionDetector())
    })
  }

  def resolveIntersection(left: WeaveType, right: WeaveType): WeaveType = {
    resolveIntersection(simplify(left), simplify(right), createDoubleRecursionIntersectionDetector())
  }

  def resolveAlgebraicIntersection(left: WeaveType, right: WeaveType): WeaveType = {
    resolveIntersection(simplify(left), simplify(right), createDoubleRecursionIntersectionDetector(), algebraic = true)
  }

  def resolveAlgebraicIntersection(types: Seq[WeaveType]): WeaveType = {
    types.reduce((l, r) => {
      val weaveType = resolveAlgebraicIntersection(l, r)
      weaveType
    })
  }

  def resolveIntersection(types: Seq[WeaveType], recursionDetector: DoubleRecursionDetector[WeaveType], algebraic: Boolean): WeaveType = {
    if (types.isEmpty) {
      NothingType()
    } else {
      types.reduce((l, r) => resolveIntersection(l, r, recursionDetector, algebraic))
    }
  }

  def removeTypeParameters(weaveType: WeaveType): WeaveType = {
    WeaveTypeTraverse.treeMap(weaveType, {
      case tp: TypeParameter =>
        removeTypeParameters(tp.top.getOrElse(AnyType()))
    })
  }

  /**
    * Resolves the intersection of two types and return the new type. If the intersection is empty then a NothingType is returns.
    *
    * @param left  The left type
    * @param right The right type
    * @return The result of intersecting the two types.
    */
  private def resolveIntersection(left: WeaveType, right: WeaveType, recursionDetector: DoubleRecursionDetector[WeaveType], algebraic: Boolean = false): WeaveType = {
    notificationManager.progress()
    right match {
      case IntersectionType(of) =>
        (left +: of).reduce((l, r) => resolveIntersection(l, r, recursionDetector, algebraic))
      case UnionType(rof) =>
        val lTypes: Seq[WeaveType] = left match {
          case UnionType(lof) => lof
          case _              => Seq(left)
        }
        val combinedTypes = SeqUtils.combine(Seq(lTypes, rof)).toArray
        val weaveTypes = combinedTypes.map(types => {
          val weaveType = resolveIntersection(types, recursionDetector, algebraic)
          weaveType
        })
        unify(weaveTypes)
      case _: ReferenceType =>
        recursionDetector.resolve(left, right, wt => resolveIntersection(wt.left, wt.right, recursionDetector, algebraic))
      case _: NothingType => NothingType()
      case _: AnyType     => left
      case _ =>
        left match {
          case ObjectType(leftProps, lClose, lOrder) =>
            right match {
              case ObjectType(rightProps, rClose, rOrder) =>
                if (!algebraic) {
                  ObjectType(intersectProperties(leftProps, rightProps), lClose && rClose, lOrder && rOrder)
                } else {
                  val resolvedProperties: Seq[WeaveType] = intersectPropertiesAlgebraically(leftProps, rightProps, lClose, rClose)
                  if (resolvedProperties.exists(!_.isInstanceOf[KeyValuePairType])) {
                    NothingType()
                  } else {
                    val result = ObjectType(resolvedProperties.asInstanceOf[Seq[KeyValuePairType]], lClose || rClose)
                    if (isEmptyType(result)) NothingType() else result
                  }
                }
              case _ => NothingType()
            }
          case KeyType(name, attrs) =>
            right match {
              case KeyType(rName, rAttrs) if compatibleNames(name, rName) =>
                val attributesByName = rAttrs
                  .++(attrs)
                  .groupBy(_.name match {
                    case NameType(value) => value
                    case key             => key
                  })
                val resolvedAttributes = attributesByName.values.map(resolveIntersection(_, recursionDetector, algebraic))
                if (resolvedAttributes.exists(!_.isInstanceOf[NameValuePairType])) {
                  NothingType()
                } else {
                  val newName = (name, rName) match {
                    case (NameType(maybeLQName), NameType(maybeRQName)) => NameType(maybeLQName.orElse(maybeRQName))
                    case _ => name
                  }

                  // Sorting attributes by key name
                  val sortedAttributes = resolvedAttributes.asInstanceOf[Seq[NameValuePairType]].sortBy(_.name.toString.toLowerCase)
                  KeyType(newName, sortedAttributes)
                }
              case _ => NothingType()
            }
          case NameValuePairType(name, value, optional) =>
            right match {
              case NameValuePairType(_, rvalue, roptional) => NameValuePairType(name, resolveIntersection(value, rvalue, recursionDetector, algebraic), optional && roptional)
              case _                                       => NothingType()
            }
          case KeyValuePairType(key, value, optional, repeated) =>
            right match {
              case KeyValuePairType(rKey, rValue, rOptional, rRepeated) =>
                val keyResult: WeaveType = resolveIntersection(key, rKey, recursionDetector, algebraic)
                val valueResult: WeaveType = resolveIntersection(value, rValue, recursionDetector, algebraic)
                val isNothing: Boolean = valueResult.isInstanceOf[NothingType] && !(optional && rOptional)
                // || !keyResult.isInstanceOf[KeyType]
                if (isNothing) {
                  NothingType()
                } else {
                  KeyValuePairType(keyResult, valueResult, rOptional && optional, rRepeated && repeated)
                }
              case _ => NothingType()
            }
          case _: ReferenceType =>
            recursionDetector.resolve(left, right, wt => resolveIntersection(wt.left, wt.right, recursionDetector, algebraic))
          case ArrayType(of) =>
            right match {
              case ArrayType(rof) => ArrayType(resolveIntersection(of, rof, recursionDetector, algebraic))
              case _              => NothingType()
            }
          case UnionType(of) =>
            val rTypes: Seq[WeaveType] = right match {
              case UnionType(rof) => rof
              case _              => Seq(right)
            }
            val combinedTypes: Array[Seq[WeaveType]] = SeqUtils.combine(Seq(of, rTypes)).toArray
            unify(combinedTypes.map(types => resolveIntersection(types, recursionDetector, algebraic)))
          case IntersectionType(of) =>
            resolveIntersection(of, recursionDetector, algebraic) match {
              case a: IntersectionType => intersec(a, right)
              case otherType           => resolveIntersection(otherType, right, recursionDetector, algebraic)
            }
          case NamespaceType(_, _) =>
            right match {
              case bt: NamespaceType => bt
              case _                 => NothingType()
            }
          case AnyType() => right
          case StringType(of) =>
            right match {
              case bt: StringType =>
                if (of.isDefined) {
                  if (bt.value.isDefined) {
                    if (of.get == bt.value.get) bt else NothingType()
                  } else left
                } else bt
              case _ => NothingType()
            }
          case BooleanType(of, _) =>
            // TODO merge constraints
            right match {
              case bt: BooleanType =>
                if (of.isDefined) {
                  if (bt.value.isDefined) {
                    if (of.get == bt.value.get) bt else NothingType()
                  } else left
                } else bt
              case _ => NothingType()
            }
          case NumberType(of) =>
            right match {
              case bt: NumberType =>
                if (of.isDefined) {
                  if (bt.value.isDefined) {
                    if (BigDecimal(of.get) == BigDecimal(bt.value.get)) bt else NothingType()
                  } else left
                } else bt
              case _ => NothingType()
            }
          case RangeType() =>
            right match {
              case rt: RangeType => rt
              case _             => NothingType()
            }
          case UriType(_) =>
            right match {
              case ut: UriType => ut
              case _           => NothingType()
            }
          case DateTimeType() =>
            right match {
              case bt: DateTimeType => bt
              case _                => NothingType()
            }
          case LocalDateTimeType() =>
            right match {
              case bt: LocalDateTimeType => bt
              case _                     => NothingType()
            }
          case LocalDateType() =>
            right match {
              case bt: LocalDateType => bt
              case _                 => NothingType()
            }
          case LocalTimeType() =>
            right match {
              case bt: LocalTimeType => bt
              case _                 => NothingType()
            }
          case TimeType() =>
            right match {
              case bt: TimeType => bt
              case _            => NothingType()
            }
          case TimeZoneType() =>
            right match {
              case bt: TimeZoneType => bt
              case _                => NothingType()
            }
          case PeriodType() =>
            right match {
              case bt: PeriodType => bt
              case _              => NothingType()
            }
          case BinaryType() =>
            right match {
              case bt: BinaryType => bt
              case _              => NothingType()
            }
          case TypeType(_) =>
            right match {
              case bt: TypeType => bt
              case _            => NothingType()
            }
          case RegexType() =>
            right match {
              case bt: RegexType => bt
              case _             => NothingType()
            }
          case NullType() =>
            right match {
              case bt: NullType => bt
              case _            => NothingType()
            }
          case NothingType() => NothingType()
          case leftTP: TypeParameter =>
            right match {
              case rightTP: TypeParameter =>
                rightTP.top match {
                  case Some(rightTopType) =>
                    val resolved = resolveIntersection(leftTP.top.getOrElse(AnyType()), rightTopType, recursionDetector, algebraic)
                    resolved match {
                      case nt: NothingType => nt //This case is when top bound of TP doesn't match with the other type
                      case _               => IntersectionType(Seq(left, right))
                    }
                  case None => IntersectionType(Seq(left, right))
                }
              case _ =>
                leftTP.top match {
                  case Some(leftTopType) =>
                    val resolved = resolveIntersection(leftTopType, right, recursionDetector, algebraic)
                    resolved match {
                      case nt: NothingType => nt //This case is when top bound of TP doesn't match with the other type
                      case _               => IntersectionType(Seq(left, right))
                    }
                  case None => IntersectionType(Seq(left, right))
                }
            }
          case FunctionType(_, lParams, lReturnType, lOverloads, _, _) =>
            right match {
              case FunctionType(_, rParams, rReturnType, rOverloads, _, _) =>
                val resultReturnType = unify(Seq(lReturnType, rReturnType))
                if (lParams.length != rParams.length) {
                  NothingType()
                } else {
                  val resultParams = ArrayBuffer[FunctionTypeParameter]()
                  var i = 0
                  while (i < lParams.length) {
                    val name: String = lParams(i).name
                    val optional: Boolean = lParams(i).optional && rParams(i).optional
                    val resolvedParamType: WeaveType = resolveIntersection(lParams(i).wtype, rParams(i).wtype)
                    resolvedParamType match {
                      case _: NothingType => return NothingType()
                      case _ =>
                        resultParams.+=(FunctionTypeParameter(name, resolvedParamType, optional))
                    }
                    i = i + 1
                  }
                  FunctionType(Seq(), resultParams, resultReturnType, lOverloads.++(rOverloads))
                }
              case _ => NothingType()

            }
          case _: DynamicReturnType => NothingType()
        }
    }
  }

  /**
    * Return true if two types are compatible when used as names.
    *
    * The interesting case is that a NameType(None) matches with
    * every other NameType(_)
    *
    * @param name  The first type to consider
    * @param rName The second type to consider
    * @return Whether both names are compatible
    */
  def compatibleNames(name: WeaveType, rName: WeaveType): Boolean = {
    (name, rName) match {
      case (NameType(maybeQName), NameType(maybeRQName)) =>
        maybeQName.isEmpty || maybeRQName.isEmpty || maybeQName.equals(maybeRQName)
      case _ => rName.equals(name)
    }
  }

  /**
    * Check if two types  A and B can be merge with A
    */
  private def isMergeableWith(topType: WeaveType, bottomType: WeaveType, recursionDetector: DoubleRecursionDetector[Boolean] = DoubleRecursionDetector((_, _) => true)): Boolean = {
    areEqualStructurally(bottomType, topType, recursionDetector, compareMetadataConstraints = false, unificationMode = true)
  }

  /**
    * Check if two types are structurally equals
    *
    * @param actualType   The type of the assignment expression
    * @param expectedType The type expected type of the assignment
    * @return True they are equals
    */
  def areEqualStructurally(
    actualType: WeaveType,
    expectedType: WeaveType,
    recursionDetector: DoubleRecursionDetector[Boolean] = DoubleRecursionDetector((_, _) => true),
    compareMetadataConstraints: Boolean = false,
    unificationMode: Boolean = false): Boolean = {

    val actualResolved = actualType match {
      case UnionType(_)         => simplify(actualType)
      case IntersectionType(of) => resolveIntersection(of)
      case _                    => actualType
    }

    val expectedResolved = expectedType match {
      case UnionType(_)         => simplify(expectedType)
      case IntersectionType(of) => resolveIntersection(of)
      case _                    => expectedType
    }

    val result = actualResolved match {
      case _: NothingType if unificationMode => true
      case _: ReferenceType =>
        recursionDetector.resolve(actualResolved, expectedResolved, t => areEqualStructurally(t.left, t.right, recursionDetector, compareMetadataConstraints, unificationMode))
      case _: UnionType if !expectedResolved.isInstanceOf[UnionType] =>
        //We flip it if they right is not a union
        areEqualStructurally(expectedResolved, actualResolved, recursionDetector, compareMetadataConstraints, unificationMode)
      case _ =>
        expectedResolved match {
          case IntersectionType(expectedTypes) => //Intersection type was already resolve so this means that it should match the same types of interesection
            actualResolved match {
              case IntersectionType(assignedTypes) =>
                //First we check that all types in assignedTypes have an element in expectedTypes that is equalStructurally. After that, we check from expectedTypes to assignedTypes but only the unchecked types.
                expectedTypes.size == assignedTypes.size || {
                  val missingAssigned = ArrayBuffer[WeaveType](assignedTypes: _*)
                  expectedTypes.forall(expectedType => {
                    missingAssigned.exists(actualType => {
                      val exists = areEqualStructurally(actualType, expectedType, recursionDetector, compareMetadataConstraints, unificationMode)
                      if (exists) {
                        missingAssigned.-=(actualType)
                      }
                      exists
                    })
                  })
                }
              case _ => false
            }
          case _: ReferenceType =>
            recursionDetector.resolve(actualResolved, expectedResolved, t => {
              areEqualStructurally(t.left, t.right, recursionDetector, compareMetadataConstraints, unificationMode)
            })
          case ObjectType(expectedProperties, rClose, _) =>
            actualResolved match {
              case ObjectType(assignedProperties, lClose, _) =>
                if (expectedProperties.size == assignedProperties.size && rClose == lClose) {
                  if (expectedProperties.isEmpty) {
                    true
                  } else {
                    var equal = true
                    val iterator = expectedProperties.iterator
                    while (iterator.hasNext && equal) {
                      val expectedProp = iterator.next()
                      val assignedPropertiesIterator = assignedProperties.iterator
                      var exists = false
                      while (assignedPropertiesIterator.hasNext && !exists) {
                        val assignedProperty = assignedPropertiesIterator.next()
                        exists = areEqualStructurally(assignedProperty, expectedProp, recursionDetector, compareMetadataConstraints, unificationMode)
                      }
                      equal = exists
                    }
                    equal
                  }
                } else {
                  false
                }
              case _ =>
                false
            }
          case KeyValuePairType(rKey, rValue, rOptional, rRepeated) =>
            actualResolved match {
              case KeyValuePairType(lKey, lValue, lOptional, lRepeated) =>
                rOptional == lOptional &&
                  rRepeated == lRepeated &&
                  areEqualStructurally(lKey, rKey, recursionDetector, compareMetadataConstraints, unificationMode) &&
                  areEqualStructurally(lValue, rValue, recursionDetector, compareMetadataConstraints, unificationMode)
              case _ =>
                false
            }
          case KeyType(rName, rAttributes) =>
            actualResolved match {
              case assignedKey: KeyType =>
                areEqualStructurally(assignedKey.name, rName, recursionDetector, compareMetadataConstraints, unificationMode) &&
                  rAttributes.size == assignedKey.attrs.size &&
                  rAttributes.forall(expectedAttr =>
                    assignedKey.attrs.exists(assignedAttr => {
                      areEqualStructurally(assignedAttr, expectedAttr, recursionDetector, compareMetadataConstraints, unificationMode)
                    }))
              case _ =>
                false
            }
          case NameValuePairType(expectedName, expectedValue, lOptional) =>
            actualResolved match {
              case NameValuePairType(assignedName, assignedValue, rOptional) =>
                (rOptional == lOptional) &&
                  areEqualStructurally(assignedName, expectedName, recursionDetector, compareMetadataConstraints, unificationMode) &&
                  areEqualStructurally(assignedValue, expectedValue, recursionDetector, compareMetadataConstraints, unificationMode)
              case _ =>
                false
            }
          case NameType(rName) =>
            actualResolved match {
              case NameType(lName) =>
                rName.isDefined == lName.isDefined &&
                  (rName.isEmpty || rName.get.equals(lName.get))
              case _ =>
                false
            }
          case ArrayType(expectedItemType) =>
            actualResolved match {
              case ArrayType(assignmentItemType) => areEqualStructurally(assignmentItemType, expectedItemType, recursionDetector, compareMetadataConstraints, unificationMode)
              case _ =>
                false
            }
          case UnionType(expectedTypes) =>
            //The types of the assigned union should be all contained in the types of the expected union
            actualResolved match {
              case UnionType(assignedTypes) =>
                //First we check that all types in assignedTypes have an element in expectedTypes that is equalStructurally. After that, we check from expectedTypes to assignedTypes but only the unchecked types.
                val allAssignedArInExpected = areUnionEqual(assignedTypes, expectedTypes, recursionDetector, compareMetadataConstraints, unificationMode)
                if (allAssignedArInExpected && assignedTypes.size == expectedTypes.size) {
                  true
                } else if (allAssignedArInExpected) {
                  areUnionEqual(expectedTypes, assignedTypes, recursionDetector, compareMetadataConstraints, unificationMode)
                } else {
                  false
                }
              case _ =>
                if (unificationMode) {
                  expectedTypes.exists(wtype => {
                    areEqualStructurally(actualResolved, wtype, recursionDetector, compareMetadataConstraints, unificationMode)
                  })
                } else {
                  expectedTypes.forall(wtype => {
                    areEqualStructurally(actualResolved, wtype, recursionDetector, compareMetadataConstraints, unificationMode)
                  })
                }
            }
          case TypeType(expectedTypeType) =>
            actualResolved match {
              case TypeType(assignmentTypeType) => areEqualStructurally(assignmentTypeType, expectedTypeType, recursionDetector, compareMetadataConstraints, unificationMode)
              case _ =>
                false
            }
          case _: TypeParameter =>
            //Type parameters are only equals if they are the same instance
            expectedResolved eq actualResolved
          case FunctionType(_, expectedArguments, expectedReturnType, _, _, _) =>
            actualResolved match {
              case FunctionType(_, assignedArguments, assignedReturnType, _, _, _) =>
                if (areEqualStructurally(expectedReturnType, assignedReturnType, recursionDetector, compareMetadataConstraints, unificationMode) && expectedArguments.size == assignedArguments.size) {
                  expectedArguments.zip(assignedArguments).forall(t => areEqualStructurally(t._1.wtype, t._2.wtype, recursionDetector, compareMetadataConstraints, unificationMode))
                } else {
                  false
                }
              case _ =>
                false
            }
          case _: DynamicReturnType =>
            false
          case StringType(expectedValue) =>
            actualResolved match {
              case StringType(actualValue) =>
                if (unificationMode) {
                  expectedValue.isEmpty || actualValue == expectedValue
                } else {
                  actualValue == expectedValue
                }
              case _ =>
                false
            }
          case BooleanType(expectedValue, _) =>
            actualResolved match {
              case BooleanType(actualValue, _) =>
                if (unificationMode) {
                  expectedValue.isEmpty || actualValue == expectedValue
                } else {
                  actualValue == expectedValue
                }
              case _ =>
                false
            }
          case NumberType(expectedValue) =>
            actualResolved match {
              case NumberType(actualValue) =>
                if (unificationMode) {
                  expectedValue.isEmpty || actualValue.map(BigDecimal(_)) == expectedValue.map(BigDecimal(_))
                } else {
                  actualValue.map(BigDecimal(_)) == expectedValue.map(BigDecimal(_))
                }
              case _ =>
                false
            }
          case AnyType() if unificationMode => true
          case _                            => expectedResolved.getClass.isInstance(actualResolved)
        }
    }

    if (result && compareMetadataConstraints && !(actualResolved.isInstanceOf[ReferenceType] || expectedResolved.isInstanceOf[ReferenceType])) {
      val leftMetadataConstrains = actualResolved.metadataConstraints().sortBy(_.name)
      val rightMetadataConstrains = expectedResolved.metadataConstraints().sortBy(_.name)
      leftMetadataConstrains == rightMetadataConstrains
    } else {
      result
    }
  }

  private def areUnionEqual(assignedTypes: Seq[WeaveType], expectedTypes: Seq[WeaveType], recursionDetector: DoubleRecursionDetector[Boolean], compareMetadataConstraints: Boolean, mergeMode: Boolean) = {
    val missingAssigned = ArrayBuffer[WeaveType](assignedTypes: _*)
    val result: Boolean = expectedTypes.forall(expectedType => {
      var exists = false
      val iterator = missingAssigned.iterator
      while (iterator.hasNext && !exists) {
        val actualType = iterator.next()
        exists = areEqualStructurally(actualType, expectedType, recursionDetector, compareMetadataConstraints, mergeMode)
        if (exists && assignedTypes.size == expectedTypes.size) {
          //This optimization can only be done if the union type has the same amount of types
          missingAssigned.-=(actualType)
        }
      }
      exists
    })
    result
  }

  /**
    * Returns if the assignmentType can be assigned to the expectedType
    *
    * @param expectedType   The type of the assignment expression
    * @param assignmentType The type expected type of the assignment
    * @return True if is a sub type
    */
  def canBeAssignedTo(assignmentType: WeaveType, expectedType: WeaveType, ctx: WeaveTypeResolutionContext, strict: Boolean = false, messageCollector: MessageCollector = MessageCollector()): Boolean = {
    checkAssignment(assignmentType, expectedType, ctx, strict, substitutionMode = false, typeErasure = false, DoubleRecursionDetector((_, _) => true), SelectionPath(), messageCollector)
  }

  /**
    * Returns if the assignmentType can be assigned to the expectedType
    *
    * @param expectedType   The type of the assignment expression
    * @param assignmentType The type expected type of the assignment
    * @return True if is a sub type
    */
  def canBeSubstituted(assignmentType: WeaveType, expectedType: WeaveType, ctx: WeaveTypeResolutionContext, messageCollector: MessageCollector = MessageCollector()): Boolean = {
    checkAssignment(assignmentType, expectedType, ctx, strict = false, substitutionMode = true, typeErasure = false, DoubleRecursionDetector((_, _) => true), SelectionPath(), messageCollector)
  }

  /**
    * Returns if the assignmentType can be assigned to the expectedType having in mind type erasure to match runtime behaviour.
    *
    * @param expectedType   The type of the assignment expression
    * @param assignmentType The type expected type of the assignment
    * @return True if is a sub type
    */
  def canBeSubstitutedWithErasure(assignmentType: WeaveType, expectedType: WeaveType, ctx: WeaveTypeResolutionContext, messageCollector: MessageCollector = MessageCollector()): Boolean = {
    checkAssignment(assignmentType, expectedType, ctx, strict = false, substitutionMode = true, typeErasure = true, DoubleRecursionDetector((_, _) => true), SelectionPath(), messageCollector)
  }

  private def checkAssignment(assignmentType: WeaveType, expectedType: WeaveType, ctx: WeaveTypeResolutionContext, strict: Boolean, substitutionMode: Boolean, typeErasure: Boolean, recursionDetector: DoubleRecursionDetector[Boolean], path: SelectionPath, messageCollector: MessageCollector): Boolean = {
    if (assignmentType eq expectedType) {
      return true
    }

    if (expectedType.isInstanceOf[ReferenceType] || assignmentType.isInstanceOf[ReferenceType]) {
      return recursionDetector.resolve(
        assignmentType,
        expectedType,
        r => checkAssignment(r.left, r.right, ctx, strict, substitutionMode, typeErasure, recursionDetector, path, messageCollector))
    }

    val location: WeaveLocation = assignmentType.location()
    val assignmentTypeCriteria: Option[Boolean] = assignmentType match {
      case UnionType(of) =>
        Some(if (typeErasure) {
          of.exists(t => {
            checkAssignment(t, expectedType, ctx, strict, substitutionMode, typeErasure, recursionDetector, path, messageCollector)
          })
        } else {
          of.forall(t => {
            checkAssignment(t, expectedType, ctx, strict, substitutionMode, typeErasure, recursionDetector, path, messageCollector)
          })
        })
      case IntersectionType(of) =>
        val resolvedIntersection = resolveIntersection(of)
        resolvedIntersection match {
          case IntersectionType(assignmentOf) =>
            expectedType match {
              case IntersectionType(expectedOf) =>
                val assignable = if (assignmentOf.size == expectedOf.size) {
                  Some(
                    assignmentOf.forall(assignmentType => {
                      expectedOf.exists(expectedType => {
                        checkAssignment(assignmentType, expectedType, ctx, strict, substitutionMode, typeErasure, recursionDetector, path, messageCollector)
                      })
                    }))
                } else {
                  Some(false)
                }
                assignable
              case _ =>
                Some(assignmentOf.forall(wt => {
                  checkAssignment(wt, expectedType, ctx, strict, substitutionMode, typeErasure, recursionDetector, path, messageCollector)
                }))
            }
          case _ =>
            Some(checkAssignment(resolvedIntersection, expectedType, ctx, strict, substitutionMode, typeErasure, recursionDetector, path, messageCollector))
        }
      case TypeParameter(_, topType, baseType, _, noImplicitBounds) if !isTypeParameter(expectedType) =>
        if (topType.isDefined) {
          val valid = if (substitutionMode) {
            !TypeHelper.resolveAlgebraicIntersection(topType.get, expectedType).isInstanceOf[NothingType]
          } else {
            checkAssignment(topType.get, expectedType, ctx, strict, substitutionMode, typeErasure, recursionDetector, path, messageCollector)
          }
          if (valid) {
            Some(true)
          } else {
            messageCollector.error(TypeMismatch(expectedType, assignmentType, path = path), location)
            Some(false)
          }
        } else if (baseType.isDefined) {
          Some(checkAssignment(baseType.get, expectedType, ctx, strict, substitutionMode, typeErasure, recursionDetector, path, messageCollector))
        } else {
          Some(noImplicitBounds || {
            //If implicit bounds set upper bound to Any
            val notAssignable = checkAssignment(AnyType(), expectedType, ctx, strict, substitutionMode, typeErasure, recursionDetector, path, new MessageCollector)
            if (!notAssignable) {
              messageCollector.error(TypeMismatch(expectedType, assignmentType, path = path), location)
            }
            notAssignable
          })
        }
      case _: DynamicReturnType => Some(true)
      case _: NothingType       => Some(true) //Is a bottom type so it should be able to assign to anything
      case _                    => None
    }

    if (assignmentTypeCriteria.isDefined) { //Check if we can define assignability by looking at assignment type
      if (!assignmentTypeCriteria.get && !messageCollector.hasErrors()) {
        messageCollector.error(TypeMismatch(expectedType, assignmentType, path = path), location)
      }
      return assignmentTypeCriteria.get
    }

    val matches = expectedType match {
      case IntersectionType(of) =>
        val resolvedExpectedIntersection = resolveIntersection(of)
        resolvedExpectedIntersection match {
          case IntersectionType(of) =>
            of.forall(wt => { //We know assignment type is not Intersection (it would have matched on assignmentTypeCriteria)
              checkAssignment(assignmentType, wt, ctx, strict, substitutionMode, typeErasure, recursionDetector, path, messageCollector)
            })
          case _: NothingType =>
            messageCollector.error(TypeMismatch(expectedType, assignmentType, path = path), assignmentType.location())
            false
          case _ =>
            val collector = new MessageCollector
            val matches = checkAssignment(assignmentType, resolvedExpectedIntersection, ctx, strict, substitutionMode, typeErasure, recursionDetector, path, collector)
            if (!matches) {
              mergeCollectorsAndTrace(messageCollector, collector, expectedType)
            }
            matches
        }
      case UnionType(expectedOf) =>
        val typesWithMessage = expectedOf.map((_, new MessageCollector()))
        val matches = typesWithMessage.exists(wtypeCollectorPair => {
          checkAssignment(assignmentType, wtypeCollectorPair._1, ctx, strict = false, substitutionMode, typeErasure, recursionDetector, path, wtypeCollectorPair._2)
        })
        if (!matches) {
          if (isPrimitiveType(assignmentType)) {
            messageCollector.error(TypeMismatch(expectedType, assignmentType, path), assignmentType.location())
          } else {
            //If it doesn't match then we collect all the messages from all the options
            typesWithMessage.foreach(pair => {
              mergeCollectorsAndTrace(messageCollector, pair._2, expectedType)
            })
          }
        }
        matches
      case et: ObjectType =>
        assignmentType match {
          case at: ObjectType =>
            (et, at) match {
              case (ObjectType(_, true, _), ObjectType(_, false, _)) =>
                messageCollector.error(CloseDoesNotAllowOpen(et, assignmentType), assignmentType.location())
                false
              case (ObjectType(eprop, eClose, _), ObjectType(aprop, _, _)) =>
                val matchedProperties: Seq[(Option[KeyValuePairType], Option[KeyValuePairType])] = ObjectTypeHelper.matchLeftProperties(eprop, aprop, substitutionMode)
                matchedProperties.forall({
                  case (Some(eType), Some(aType)) =>
                    val propertyPath: SelectionPath = eType.key match {
                      case KeyType(NameType(maybeName), _) =>
                        maybeName match {
                          case Some(expectedName) => path.field(expectedName.name)
                          case None               => path.field("_")
                        }
                      case a =>
                        path.field("(" + a.toString(prettyPrint = false, namesOnly = true) + ")")
                    }
                    if (aType.repeated && !eType.repeated) {
                      // TODO Do we want this case?
                      messageCollector.error(RepeatedFieldNotSupported(propertyPath, et, at), location)
                      false
                    } else {
                      // TODO shouldnt path here be specified?
                      checkAssignment(aType, eType, ctx, strict, substitutionMode = substitutionMode, typeErasure, recursionDetector, path, messageCollector)
                    }
                  case (Some(eType), None) =>
                    if (eType.optional || !eType.key.isInstanceOf[KeyType]) { //If it is a type parameter then it works as optional
                      true
                    } else {
                      val propertyPath: SelectionPath = eType.key match {
                        case KeyType(NameType(maybeName), _) =>
                          maybeName match {
                            case Some(expectedName) => path.field(expectedName.name)
                            case None               => path.field("_")
                          }
                        case a =>
                          path.field("(" + a.toString(prettyPrint = false, namesOnly = true) + ")")
                      }
                      messageCollector.error(MissingRequiredProperty(propertyPath, et, at), location)
                      false
                    }
                  case (None, Some(aType)) =>
                    if (eClose) {
                      // TODO shouldnt path here be specified?
                      messageCollector.error(CloseDoesNotAllowExtraProperties(path, aType, et, at), aType.location())
                      false
                    } else {
                      true
                    }
                })
            }
          case _ => false
        }
      case kvt @ KeyValuePairType(expectedKey, expectedValue, _, _) =>
        assignmentType match {
          case assignedKVP: KeyValuePairType =>
            val propertyPath: SelectionPath = path.field(kvt.getKeyName)
            checkAssignment(assignedKVP.key, expectedKey, ctx, strict, substitutionMode, typeErasure, recursionDetector, propertyPath, messageCollector) &&
              checkAssignment(assignedKVP.value, expectedValue, ctx, strict, substitutionMode, typeErasure, recursionDetector, propertyPath, messageCollector)
          case _ =>
            false
        }
      case et @ KeyType(expectedAttrName, expectedAttributes) =>
        assignmentType match {
          case assignedKey: KeyType =>
            val nameAssignment: Boolean = checkAssignment(assignedKey.name, expectedAttrName, ctx, strict, substitutionMode, typeErasure, recursionDetector, path, messageCollector)
            if (nameAssignment) {
              val missingProperties = assignedKey.attrs
              expectedAttributes.forall({
                case expectedAttribute @ NameValuePairType(NameType(Some(expectedName)), _, optional) =>
                  val attributePath = path.field("@" + expectedName.toString)
                  val matchedAttributes: Option[NameValuePairType] = ObjectTypeHelper.selectAttribute(expectedName, missingProperties)
                  matchedAttributes match {
                    case Some(assignedAttribute) =>
                      checkAssignment(assignedAttribute, expectedAttribute, ctx, strict, substitutionMode, typeErasure, recursionDetector, path, messageCollector)
                    case None if !optional =>
                      messageCollector.error(MissingRequiredProperty(attributePath, et, assignmentType), assignedKey.location())
                      false
                    case None =>
                      //Optional property good to ignore
                      true
                  }
                case _ => true
              })

            } else {
              nameAssignment
            }
          case _ =>
            false
        }
      case NameValuePairType(expectedName, expectedValue, _) =>
        assignmentType match {
          case NameValuePairType(assignedName, assignedValue, _) =>
            val propertyPath = path.field("@" + expectedName.toString)
            checkAssignment(assignedName, expectedName, ctx, strict, substitutionMode, typeErasure, recursionDetector, propertyPath, messageCollector) &&
              checkAssignment(assignedValue, expectedValue, ctx, strict, substitutionMode, typeErasure, recursionDetector, propertyPath, messageCollector)
          case _ =>
            false
        }
      case NameType(expectedName) =>
        assignmentType match {
          case NameType(assignedName) =>
            if (expectedName.isDefined) {
              if (assignedName.isDefined) {
                expectedName.get.selectedBy(assignedName.get)
              } else {
                false
              }
            } else {
              true
            }
          case _ =>
            false
        }
      case ArrayType(expectedItemType) =>
        assignmentType match {
          case ArrayType(assignmentItemType) =>
            val collector = new MessageCollector
            val matches = typeErasure || checkAssignment(assignmentItemType, expectedItemType, ctx, strict, substitutionMode, typeErasure, recursionDetector, path.itemArray(), collector)
            if (!matches) {
              mergeCollectorsAndTrace(messageCollector, collector, expectedType)
            }
            matches
          case _ =>
            false
        }
      case TypeType(expectedTypeType) =>
        assignmentType match {
          case TypeType(assignmentTypeType) =>
            val collector = new MessageCollector
            val matches = checkAssignment(assignmentTypeType, expectedTypeType, ctx, strict, substitutionMode, typeErasure, recursionDetector, path, messageCollector)
            if (!matches) {
              mergeCollectorsAndTrace(messageCollector, collector, expectedType)
            }
            matches
          case _ => false
        }
      case etp @ TypeParameter(_, baseType, topType, _, _) =>
        if (substitutionMode) {
          baseType.forall((expectedType: WeaveType) => {
            canBeSubstituted(assignmentType, expectedType, ctx)
          }) && topType.forall(topType => {
            canBeSubstituted(topType, assignmentType, ctx)
          })
        } else {
          assignmentType match {
            case atp: TypeParameter =>
              if ((etp eq atp) || atp.top.exists(top => canBeSubstituted(atp, top, ctx))) {
                true
              } else {
                messageCollector.error(TypeMismatch(expectedType, assignmentType, path = path), location)
                false
              }
            case _ =>
              messageCollector.error(TypeMismatch(expectedType, assignmentType, path = path), location)
              false
          }
        }
      case eft @ FunctionType(_, expectedArguments, expectedReturnType, _, _, _) =>
        assignmentType match {
          case FunctionType(_, assignedArguments, assignedReturnType, overloads, _, _) =>
            if (overloads.nonEmpty) {
              overloads.exists(aft => {
                checkAssignment(aft, eft, ctx, strict, substitutionMode, typeErasure, recursionDetector, path, messageCollector)
              })
            } else {
              var matches = true
              if (expectedArguments.size == assignedArguments.size) {
                matches = canFunctionParameterByAssigned(assignedArguments, expectedArguments, ctx, strict, substitutionMode, typeErasure, path, messageCollector, recursionDetector)
              } else if (expectedArguments.size < assignedArguments.size) {
                //If head has optional
                if (assignedArguments.headOption.exists(_.optional)) {
                  val delta: Int = assignedArguments.length - expectedArguments.size
                  //Make sure all non validated params are optional
                  val paramsSplit = assignedArguments.splitAt(delta)
                  if (paramsSplit._1.forall(_.optional)) {
                    matches = matches && canFunctionParameterByAssigned(paramsSplit._2, expectedArguments, ctx, strict, substitutionMode, typeErasure, path, messageCollector, recursionDetector)
                  } else {
                    //Missing optional head parameters
                    messageCollector.error(TooManyArgumentMessage(expectedArguments.map(_.wtype), assignedArguments.map(_.wtype), eft, assignmentType), location)
                    matches = false
                  }
                } else {
                  //Make sure all non validated params are optional
                  val assignedArgsSplitByDefault = assignedArguments.splitAt(expectedArguments.size)
                  if (assignedArgsSplitByDefault._2.forall(_.optional)) {
                    matches = matches && canFunctionParameterByAssigned(assignedArgsSplitByDefault._1, expectedArguments, ctx, strict, substitutionMode, typeErasure, path, messageCollector, recursionDetector)
                  } else {
                    messageCollector.error(TooManyArgumentMessage(expectedArguments.map(_.wtype), assignedArguments.map(_.wtype), eft, assignmentType), location)
                    matches = false
                  }
                }
              } else {
                messageCollector.error(NotEnoughArgumentMessage(expectedArguments.map(_.wtype), assignedArguments.map(_.wtype), eft, assignmentType), location)
                matches = false
              }
              if (matches) {
                matches = assignedReturnType match {
                  case drt: DynamicReturnType =>
                    //We need to verify context as in some cases it is null and so ignore it
                    if (!FunctionTypeHelper.isDynamicFunction(eft) && ctx != null) {
                      val mayBeReturnType: Option[WeaveType] = FunctionTypeHelper.resolveReturnType(expectedArguments.map(_.wtype), None, ctx, drt, strict, messageCollector)
                      mayBeReturnType match {
                        case Some(realReturnType) =>
                          checkAssignment(realReturnType, expectedReturnType, ctx, strict, substitutionMode, typeErasure, recursionDetector, path.returnType(assignedArguments.length, realReturnType), messageCollector)
                        case None =>
                          //If nothing was inferred we accept the dynamic return type only if not error were present
                          !messageCollector.hasErrors()
                      }
                    } else {
                      true
                    }
                  case _ =>
                    checkAssignment(assignedReturnType, expectedReturnType, ctx, strict, substitutionMode, typeErasure, recursionDetector, path.returnType(assignedArguments.length, assignedReturnType), messageCollector)
                }
              }
              matches
            }
          case _ =>
            false
        }
      case AnyType()            => true
      case _: DynamicReturnType => true
      case StringType(Some(expectedValue)) =>
        assignmentType match {
          case StringType(Some(actualValue)) =>
            actualValue == expectedValue
          case _ =>
            false
        }
      case BooleanType(Some(expectedValue), _) =>
        assignmentType match {
          case BooleanType(Some(actualValue), _) =>
            actualValue == expectedValue
          case _ =>
            false
        }
      case NumberType(Some(expectedValue)) =>
        assignmentType match {
          case NumberType(Some(actualValue)) =>
            actualValue.map(BigDecimal(_)) == expectedValue.map(BigDecimal(_))
          case _ =>
            false
        }
      case _ =>
        if (expectedType.getClass.isInstance(assignmentType)) {
          true
        } else {
          false
        }
    }

    if (!matches && messageCollector.errorMessages.isEmpty) {
      //Add default message
      messageCollector.error(TypeMismatch(expectedType, assignmentType, path = path), location)
    }
    matches
  }

  private def mergeCollectorsAndTrace(targetCollector: MessageCollector, sourceCollector: MessageCollector, expectedType: WeaveType): MessageCollector = {
    sourceCollector.foreachErrorMessage({
      case tm: TypeMessage => tm.addTrace(expectedType)
      case m               => m
    })
    targetCollector.mergeWith(sourceCollector)
  }

  /**
    * Returns true if it is a type parameter or a union of type parameters
    *
    * @param theType The type to be checked
    * @return
    */
  def isTypeParameter(theType: WeaveType, recursionDetector: RecursionDetector[Boolean] = RecursionDetector((_, _) => false)): Boolean = {
    theType match {
      case _: TypeParameter => true
      case UnionType(of)    => of.exists(t => isTypeParameter(t, recursionDetector))
      case rt: ReferenceType =>
        recursionDetector.resolve(rt, wt => isTypeParameter(wt, recursionDetector))
      case _ => false
    }
  }

  /**
    * Returns true if it is a type given type is a type parameter
    *
    * @param theType The type to be checked
    * @return
    */
  def isJustTypeParameter(theType: WeaveType): Boolean = {
    theType match {
      case _: TypeParameter => true
      case _                => false
    }
  }

  private def canFunctionParameterByAssigned(
    assignedArguments: Seq[FunctionTypeParameter], //
    expectedArguments: Seq[FunctionTypeParameter], //
    ctx: WeaveTypeResolutionContext, //
    strict: Boolean, //
    substitutionMode: Boolean, //
    typeErasure: Boolean, //
    path: SelectionPath, //
    messageCollector: MessageCollector, //
    recursionDetector: DoubleRecursionDetector[Boolean]): Boolean = {
    //Pair of expected actual params
    val expectedActualArgs: Seq[(FunctionTypeParameter, FunctionTypeParameter)] = expectedArguments.zip(assignedArguments)
    // If we are trying to replicate runtime behaviour then function parameters will be assignable.
    if (typeErasure) {
      true
    } else {

      !expectedActualArgs
        .zipWithIndex
        .exists(expectedActualIndexPair => {
          val index: Int = expectedActualIndexPair._2
          val (expectedParam: FunctionTypeParameter, actualParam: FunctionTypeParameter) = expectedActualIndexPair._1
          val expectedType: WeaveType = expectedParam.wtype

          val actualArgName: String = actualParam.name
          val actualType: WeaveType = actualParam.wtype

          //For function parameters we flip it
          val isDynamicTypeParameters = FunctionTypeHelper.isDynamicTypeParameter(actualType)
          if (isDynamicTypeParameters) {
            false
          } else {
            //This collector is used to gather all messages and then fix them as we revert the types
            val tmpCollector = new MessageCollector
            val canBeAssigned = checkAssignment(expectedType, actualType, ctx, strict, substitutionMode, typeErasure, recursionDetector, path.arg(actualArgName, actualType, index), tmpCollector)
            tmpCollector.errorMessages.foreach(errorMessage => {
              errorMessage._2 match {
                case tm: TypeMismatch =>
                  //We flip the type mismatch
                  val mismatch = TypeMismatch(tm.actualType, tm.expectedType, tm.path).addDetail("Function parameters are contravariant.")
                  //The expected is the actual
                  val actualLocation = tm.expectedType.location()
                  messageCollector.error(mismatch, actualLocation)
                case tm: TypeMessage =>
                  tm.addDetail("Function parameters are contravariant.")
                  messageCollector.error(tm, actualType.location())
                case message =>
                  messageCollector.error(message, actualType.location())
              }
            })

            tmpCollector.warningMessages.foreach(warningMessage => {
              messageCollector.warning(warningMessage._2, actualType.location())
            })
            !canBeAssigned
          }
        })
    }
  }

  /**
    * Tries to simplify the specified Type into another that is equivalent but declared in a simple way
    *
    * @param wtype The type to simplify
    * @return The simplified type
    */
  def simplify(wtype: WeaveType): WeaveType = {
    WeaveTypeTraverse.treeMap(wtype, {
      case UnionType(of) =>
        UnionType(inlineUnionTypes(of))
      case rt: ReferenceType => rt
    })
  }

  def collectAllTypeFormUnionType(weaveType: WeaveType, stack: RecursionDetector[Seq[WeaveType]] = RecursionDetector[Seq[WeaveType]]((_, _) => Seq[WeaveType]())): Seq[WeaveType] = {
    weaveType match {
      case UnionType(ut) =>
        ut.flatMap(wt => {
          val typeToTypes = collectAllTypeFormUnionType(wt, stack)
          if (typeToTypes.isEmpty) {
            Seq(wt)
          } else {
            typeToTypes
          }
        })
      case rt: ReferenceType => stack.resolve(rt, wt => collectAllTypeFormUnionType(wt, stack))
      case _                 => Seq()
    }
  }

  def simplifyIntersections(wtype: WeaveType): WeaveType = {
    WeaveTypeTraverse.treeMap(wtype, {
      case IntersectionType(of) =>
        resolveIntersection(of)
      case rt: ReferenceType => rt
    })
  }

  def simplifyUnions(wtype: WeaveType): WeaveType = {
    WeaveTypeTraverse.treeMap(wtype, {
      case UnionType(of) =>
        unify(of)
      case rt: ReferenceType => rt
    })
  }

  def intersec(wt: Seq[WeaveType]): WeaveType = {
    wt.reduce(intersec)
  }

  def intersec(a: WeaveType, b: WeaveType): IntersectionType = {
    val types = (a, b) match {
      case (IntersectionType(aOf), IntersectionType(bOf)) => aOf ++ bOf
      case (a, IntersectionType(bOf))                     => a +: bOf
      case (IntersectionType(aOf), b)                     => aOf :+ b
      case (a, b)                                         => Seq(a, b)
    }
    IntersectionType(types.distinct)
  }

  /**
    * Replace union types of Object into one Object, doing the corresponding type changes so that is still compliant
    *
    * @param wtype The type where to do the replacements
    * @return The simplified type
    */
  @WeaveApi(Seq("data-weave-agent"))
  def resolveUnion(wtype: WeaveType): WeaveType = {
    resolveUnion(wtype, createRecursionDetector())
  }

  def createRecursionDetector(): RecursionDetector[WeaveType] = {
    RecursionDetector[WeaveType]((id, refResolver) => SimpleReferenceType(id, None, () => refResolver()))
  }

  def createDoubleRecursionDetector(): DoubleRecursionDetector[WeaveType] = {
    DoubleRecursionDetector[WeaveType]((wtp, refResolver) => SimpleReferenceType(NameIdentifier("LazyRecursive"), None, () => refResolver()))
  }

  def createDoubleRecursionIntersectionDetector(): DoubleRecursionDetector[WeaveType] = {
    DoubleRecursionDetector[WeaveType]((wtp, _) => {
      TypeHelper.intersec(wtp.left, wtp.right)
    })
  }

  private def resolveUnion(wtype: WeaveType, recursionDetector: RecursionDetector[WeaveType]): WeaveType = {
    notificationManager.progress()
    WeaveTypeTraverse.treeMap(
      wtype, {
      case ut @ UnionType(of) =>
        val propertiesWithIndex = of.zipWithIndex
          .map({
            case (rt: ReferenceType, index) =>
              val resolvedType: WeaveType = recursionDetector.resolve(rt, refType => refType)
              (index, resolvedType)
            case (t, index) =>
              (index, t)
          })
        val objectTypes: Seq[(Int, ObjectType)] = propertiesWithIndex
          .collect({
            case (index, ot: ObjectType) => (index, ot)
          })
        if (objectTypes.size <= 1) {
          ut
        } else {
          //We only simplify things if there are more than one object
          val objectIndexes = objectTypes.map(_._1)
          val objectTypeProperties = objectTypes.map(_._2).iterator
          var result: ObjectType = objectTypeProperties.next()
          while (objectTypeProperties.hasNext) {
            result = unifyProperties(objectTypeProperties.next(), result, recursionDetector)
          }
          val nonObjectTypes = of.zipWithIndex
            .filter(pair => !objectIndexes.contains(pair._2))
            .map(_._1)
          unify(nonObjectTypes :+ result)
        }
    },
      recursionDetector)
  }

  private def unifyProperties(leftObject: ObjectType, rightObject: ObjectType, recursionDetector: RecursionDetector[WeaveType]): ObjectType = {
    val leftProps = leftObject.properties
    val rightProps = rightObject.properties
    val allProperties = leftProps.++(rightProps)
    val weaveTypeByKey: Map[WeaveType, Seq[KeyValuePairType]] = allProperties.groupBy(_.key match {
      case KeyType(name, _) =>
        name
      case key => key
    })
    val properties = new ArrayBuffer[KeyValuePairType]()
    val allValuesOfKey = weaveTypeByKey.values.iterator

    while (allValuesOfKey.hasNext) {
      val duplicatedKVP = allValuesOfKey.next()
      val repeated: Boolean = duplicatedKVP.exists(_.repeated)
      val optional: Boolean = duplicatedKVP.size == 1 || duplicatedKVP.exists(_.optional)
      val valueType = if (duplicatedKVP.size == 1) {
        duplicatedKVP.head.value
      } else {
        val weaveTypes: Seq[WeaveType] = duplicatedKVP.map(_.value)
        val uniqueTypes = dedupTypes(weaveTypes)
        if (uniqueTypes.size == 1) {
          uniqueTypes.head
        } else {
          resolveUnion(unionWithoutSimplification(uniqueTypes), recursionDetector)
        }
      }
      val keys = duplicatedKVP.map(_.key).collect({ case kt: KeyType => kt })
      val attributes = keys.map(_.attrs).reduce(_.++(_)).groupBy(_.name)
      val attributeResult: Seq[NameValuePairType] = attributes.values
        .map(attrs => {
          NameValuePairType(attrs.head.name, unify(attrs.map(_.value)), attrs.size == 1 || attrs.exists(_.optional))
        })
        .toSeq
      // Sorting attributes by key name
      val sortedAttributes = attributeResult.sortBy(_.name.toString.toLowerCase)
      val resultKey = keys.headOption.map(key => KeyType(key.name, sortedAttributes)).getOrElse(duplicatedKVP.head.key)
      val keyValuePairType = KeyValuePairType(resultKey, valueType, optional, repeated)
      copyMetadataTo(duplicatedKVP, keyValuePairType)

      properties.+=(keyValuePairType)
    }

    // Sorting properties by key name
    val sortedProperties = properties.sortBy(_.getKeyName.toLowerCase)
    ObjectType(sortedProperties, !(leftObject.isOpen || rightObject.isOpen))
  }

  /**
    * This implementation of intersectProperties inspires the one at org.mule.weave.v2.model.types.Types,
    * used at the execution phase
    * KEEP IN SYNC
    *
    * @param leftProps
    * @param rightProps
    * @return
    */
  private def intersectProperties(leftProps: Seq[KeyValuePairType], rightProps: Seq[KeyValuePairType]): Seq[KeyValuePairType] = {
    val allProperties = leftProps.++(rightProps)
    val weaveTypeToTypes = allProperties.groupBy(_.key match {
      case KeyType(name, _) =>
        name
      case key => key
    })
    val properties = weaveTypeToTypes.values
      .map(duplicatedKVP => {
        if (duplicatedKVP.size == 1) {
          duplicatedKVP.head
        } else {
          val repeated: Boolean = duplicatedKVP.size > 1
          val optional: Boolean = duplicatedKVP.forall(_.optional)
          val valueType: WeaveType = unify(duplicatedKVP.map(_.value))
          val keys: Seq[KeyType] = duplicatedKVP.map(_.key).collect({ case kt: KeyType => kt })
          val attributes: Map[WeaveType, Seq[NameValuePairType]] = keys.flatMap(_.attrs).groupBy(_.name)
          val attributeResult: Seq[NameValuePairType] = attributes.values
            .map(attrs => {
              NameValuePairType(attrs.head.name, unify(attrs.map(_.value)))
            })
            .toSeq

          // Sorting attributes by key name
          val sortedAttributes = attributeResult.sortBy(_.name.toString.toLowerCase)
          val resultKey = keys.headOption.map(key => KeyType(key.name, sortedAttributes)).getOrElse(duplicatedKVP.head.key)
          val pairType = KeyValuePairType(resultKey, valueType, optional, repeated)

          //Copy all metadata
          copyMetadataTo(duplicatedKVP, pairType)
          pairType
        }
      })
      .toSeq
    // Sorting properties by key name
    val sortedProperties = properties.sortBy(_.getKeyName.toLowerCase)
    sortedProperties
  }

  private def intersectPropertiesAlgebraically(leftProps: Seq[KeyValuePairType], rightProps: Seq[KeyValuePairType], lClose: Boolean, rClose: Boolean): Seq[WeaveType] = {
    val matchedProps: Seq[(WeaveType, WeaveType)] = ObjectTypeHelper.matchAllProperties(leftProps, rightProps, lClose, rClose)

    matchedProps.map({
      case (lkvp, rkvp) =>
        resolveAlgebraicIntersection(lkvp, rkvp)
    })
  }

  private def copyMetadataTo(duplicatedKVP: Seq[KeyValuePairType], pairType: KeyValuePairType): Unit = {
    duplicatedKVP.foreach(dkp => {
      WeaveTypeCloneHelper.copyMetadataTo(dkp, pairType)
    })
  }

  /**
    * Resolves the intersection of two types
    *
    * @param left  The left type
    * @param right The right type
    * @return The type that results of the intersection of this two types
    */
  def append(left: WeaveType, right: WeaveType): WeaveType = {
    resolveIntersection(left, right)
  }

  /**
    * Unifies the specified types into a single type
    *
    * @param types The types to unify
    * @return The unified type
    */
  def unify(types: Seq[WeaveType]): WeaveType = {
    val unifiedTypes: Seq[WeaveType] = simplify(UnionType(types)).asInstanceOf[UnionType].of
    val distinct: Seq[WeaveType] = dedupTypes(unifiedTypes)
    unionWithoutSimplification(distinct)
  }

  private def unionWithoutSimplification(distinct: Seq[WeaveType]) = {
    if (distinct.isEmpty) {
      //If no type then is an error
      AnyType()
    } else if (distinct.size == 1) {
      distinct.head
    } else {
      UnionType(distinct)
    }
  }

  /**
    * Removes duplicated types in a given set of types and removes Unknown Types
    *
    * @param unifiedTypes The types where to remove
    * @return The list of unique types
    */
  def dedupTypes(unifiedTypes: Seq[WeaveType]): Seq[WeaveType] = {
    if (unifiedTypes.isEmpty) {
      unifiedTypes
    } else {
      //We first look for any top element if it existes then we just return it
      val maybeAnyType = unifiedTypes.find(_.isInstanceOf[AnyType])
      if (maybeAnyType.isDefined) {
        Seq(maybeAnyType.get)
      } else {
        //Filter bottom types
        val filteredTypes = unifiedTypes.filterNot(_.isInstanceOf[NothingType])
        //If they are all Nothing types then return Nothing
        if (filteredTypes.isEmpty) {
          Seq(unifiedTypes.head)
        } else if (filteredTypes.size == 1) {
          Seq(filteredTypes.head)
        } else {
          val distinct: ArrayBuffer[WeaveType] = ArrayBuffer()
          filteredTypes.zipWithIndex.foreach(typeIndex => {
            val missingTypes: Seq[WeaveType] = filteredTypes.slice(typeIndex._2 + 1, filteredTypes.size)
            // We search if any other type is more generic. This mean that this type can be assigned to the other
            // If a = b this mean that type a is equal to b or a super type (less specific)
            // So we need to search in the already added types and in the missing
            if (!missingTypes.exists((expectedType: WeaveType) => isMergeableWith(expectedType, typeIndex._1)) &&
              !distinct.exists((expectedType: WeaveType) => isMergeableWith(expectedType, typeIndex._1))) {
              distinct.+=(typeIndex._1)
            }
          })
          distinct
        }
      }
    }
  }

  /**
    * Returns true if it is a primitive type (String, Number, Boolean, Date,Regex...)
    *
    * @param wtype The type to verify
    * @return True if it is primitive or not
    */
  def isPrimitiveType(wtype: WeaveType): Boolean = {
    wtype match {
      case StringType(_)       => true
      case AnyType()           => true
      case BooleanType(_, _)   => true
      case NumberType(_)       => true
      case RangeType()         => true
      case UriType(_)          => true
      case DateTimeType()      => true
      case LocalDateTimeType() => true
      case LocalDateType()     => true
      case LocalTimeType()     => true
      case TimeType()          => true
      case TimeZoneType()      => true
      case PeriodType()        => true
      case BinaryType()        => true
      case TypeType(_)         => true
      case RegexType()         => true
      case NullType()          => true
      case NothingType()       => true
      case _                   => false
    }
  }

  def inlineUnionTypes(of: Seq[WeaveType]): Seq[WeaveType] = {
    of.flatMap {
      case UnionType(unionOf) => inlineUnionTypes(unionOf)
      case t: WeaveType       => Seq(t)
    }
  }

  def hasPatternProperty(properties: Seq[KeyValuePairType]): Boolean = {
    properties.exists(keyValuePair =>
      keyValuePair.key match {
        case KeyType(NameType(maybeName), _) =>
          maybeName match {
            case Some(_) => false
            case None    => true
          }
        case _ => false
      })
  }

  def isLiteralType(wtype: WeaveType): Boolean = {
    wtype match {
      case StringType(Some(_))     => true
      case NumberType(Some(_))     => true
      case BooleanType(Some(_), _) => true
      case _                       => false
    }
  }

  def isSingletonType(wtype: WeaveType): Boolean = {
    wtype match {
      case StringType(Some(_))     => true
      case NumberType(Some(_))     => true
      case BooleanType(Some(_), _) => true
      case NullType()              => true
      case ObjectType(prop, true, _) =>
        prop.forall(p => {
          isSingletonType(p.value) || (p.optional && isEmptyType(p.value))
        })
      case ArrayType(of) => isEmptyType(of)
      case _             => false
    }
  }

  def isEmptyType(a: WeaveType): Boolean = {
    a match {
      case NothingType() => true
      case ObjectType(props, _, _) =>
        props.exists({
          case KeyValuePairType(KeyType(_, attrs), value, false, _) =>
            isEmptyType(value) || attrs.exists({
              case NameValuePairType(_, value, false) => isEmptyType(value)
              case _                                  => false
            })
          case _ => false
        })
      case other => areEqualStructurally(NothingType(), other)
    }
  }

  /**
    * Subtract toRemove from orig
    *
    * Keep in mind that the returned type is not exactly the type given
    * by removing toRemove from orig, it's a best effort algorithm that may be improved.
    *
    * @param orig     the original type
    * @param toRemove the type to remove
    * @return a type given by removing toRemove from orig
    */
  def subtractType(orig: WeaveType, toRemove: WeaveType, ctx: WeaveTypeResolutionContext, recursionDetector: DoubleRecursionDetector[WeaveType] = createDoubleRecursionDetector()): WeaveType = {
    orig match {
      case _: ReferenceType =>
        recursionDetector.resolve(orig, toRemove, t => subtractType(t.left, t.right, ctx, recursionDetector))
      case UnionType(of)        => unify(of.map(subtractType(_, toRemove, ctx, recursionDetector)))
      case IntersectionType(of) => resolveIntersection(of.map(subtractType(_, toRemove, ctx, recursionDetector)))
      case _ =>
        toRemove match {
          case _: ReferenceType =>
            recursionDetector.resolve(orig, toRemove, t => subtractType(t.left, t.right, ctx, recursionDetector))
          case UnionType(of)        => of.foldLeft(orig)((acc, toRemove) => subtractType(acc, toRemove, ctx, recursionDetector))
          case IntersectionType(of) => unify(of.map(subtractType(orig, _, ctx, recursionDetector)))
          case BooleanType(Some(value), _) =>
            orig match {
              case BooleanType(None, constraints) => BooleanType(Some(!value), constraints)
              case BooleanType(Some(origValue), _) if value == origValue => NothingType()
              case _ => orig
            }
          case objToRemove: ObjectType =>
            orig match {
              case objOrig: ObjectType => ObjectTypeHelper.subtractObjects(objOrig, objToRemove, ctx, this)
              case _                   => orig
            }
          case _ =>
            if (canBeAssignedTo(orig, toRemove, ctx))
              NothingType()
            else
              orig
        }
    }
  }
}

object TypeHelper extends TypeHelper(new ParsingNotificationManager) {

  def apply(ctx: WeaveTypeResolutionContext): TypeHelper = TypeHelper(ctx.currentParsingContext)

  def apply(ctx: ParsingContext): TypeHelper = ctx.typeHelper
}

object Kind extends Enumeration {
  type Kind = Value
  val RETURN_TYPE, FIELD, ARRAY = Value
}
