package org.mule.weave.v2.parser

import org.mule.weave.v2.grammar.Grammar
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.phase._
import org.mule.weave.v2.sdk.WeaveResource
import org.parboiled2.ParseError
import org.parboiled2.Parser.DeliveryScheme.Either

object CatalogParser {

  private val compilationTransformations: Array[AstNodeTransformer] = Array(new ImplicitImportsTransformer())

  def parsingPhase(): CompilationPhase[ParsingContentInput, ParsingResult[ModuleNode]] = {
    ParsingPhase[ModuleNode](parse)
      .chainWith(new ParsingAnnotationProcessorPhase[ModuleNode, ParsingResult[ModuleNode]]())
  }

  def scopePhase(): CompilationPhase[ParsingContentInput, ScopeGraphResult[ModuleNode]] = {
    canonicalAstPhase().chainWith(scopePhasePhases())
  }

  def canonicalAstPhase(): CompilationPhase[ParsingContentInput, ParsingResult[ModuleNode]] = {
    parsingPhase()
      .chainWith(canonicalPhasePhases())
  }

  def canonicalPhasePhases(): CompilationPhase[ParsingResult[ModuleNode], ParsingResult[ModuleNode]] = {
    new AstNodeTransformationPhase[ModuleNode, ParsingResult[ModuleNode]](compilationTransformations)
  }

  def scopePhasePhases(): CompilationPhase[AstNodeResultAware[ModuleNode] with ParsingContentAware, ScopeGraphResult[ModuleNode]] = {
    new NoValidationScopeGraphPhase[ModuleNode]()
      .chainWith(new MetadataKeyNodeInjectorPhase[ModuleNode, ScopeGraphResult[ModuleNode]])
      .chainWith(new MetadataTypeNodeInjectorPhase[ModuleNode, ScopeGraphResult[ModuleNode]])
  }

  def parse[T](phase: CompilationPhase[ParsingContentInput, T], input: WeaveResource, parsingContext: ParsingContext): PhaseResult[T] = {
    val parserInput = ParsingContentInput(input, parsingContext.nameIdentifier, SafeStringBasedParserInput(input.content()))
    phase.call(parserInput, parsingContext)
  }

  private def parse(parserInput: ParsingContentInput, context: ParsingContext): Either[ParseError, ModuleNode] = {
    val parser = new Grammar(parserInput.input, parserInput.nameIdentifier, context.errorTrace, context.attachDocumentation)
    parser.module.run()
  }
}

object CatalogAnnotationProcessorRegistry extends AnnotationProcessorRegistry {

  override def annotations(): Map[NameIdentifier, AnnotationProcessor[_]] = {
    Map(
      NameIdentifier.METADATA_ANNOTATION.localName() -> new MetadataAnnotationProcessorParsingPhase())
  }
}
