package org.mule.weave.v2.parser.phase

import org.mule.weave.v2.parser.annotation._
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.header.directives.VarDirective
import org.mule.weave.v2.parser.phase.DataFormatSettingsHelper.{ collectReaderSettings, collectWriterSettings, toDataFormatSetting }
import org.mule.weave.v2.ts
import org.mule.weave.v2.ts._

import scala.annotation.tailrec

class DataFormatExtensionAnnotationProcessor extends AbstractTypeAnnotationProcessor {

  override def run(annotatedNode: AstNode, annotation: AnnotationNode, context: TypePhaseAnnotationContext): Unit = {
    annotatedNode match {
      case vd: VarDirective =>
        val maybeNode: Option[TypeNode] = context.typeGraph.findNode(vd.variable)
        val readerOptions: Seq[DataFormatSetting[_]] = maybeNode.map(collectReaderSettings(_, toDataFormatSetting)).getOrElse(Seq.empty)
        val writerOptions: Seq[DataFormatSetting[_]] = maybeNode.map(collectWriterSettings(_, toDataFormatSetting)).getOrElse(Seq.empty)

        annotatedNode.annotate(DataFormatExtensionAnnotation(DataFormatSettings(readerOptions, writerOptions)))
      case _ =>
    }
  }

}

object DataFormatSettingsHelper {
  def collectReaderSettings[T](value: TypeNode, toSetting: (String, WeaveType, Option[String]) => T): Seq[T] = {
    value
      .resultType()
      .flatMap((weaveType) => {
        val maybeReaderType = TypeHelper.selectProperty(weaveType, "reader")

        val maybeParameter = maybeReaderType
          .collect({
            case ft: FunctionType => ft
          })
          .flatMap((ft) => if (ft.params.size == 3) Some(ft.params(2).wtype) else None)

        maybeParameter.map(toSettings(_, toSetting))
      })
      .getOrElse(Seq.empty)
  }

  def collectWriterSettings[T](value: TypeNode, toSetting: (String, WeaveType, Option[String]) => T): Seq[T] = {
    value
      .resultType()
      .flatMap((weaveType) => {
        val maybeReaderType = TypeHelper.selectProperty(weaveType, "writer")

        val maybeParameter = maybeReaderType
          .collect({
            case ft: FunctionType => ft
          })
          .flatMap((ft) => if (ft.params.size == 2) Some(ft.params(1).wtype) else None)

        maybeParameter.map(toSettings(_, toSetting))
      })
      .getOrElse(Seq.empty)
  }

  private def toSettings[T](maybeParameter: WeaveType, toSetting: (String, WeaveType, Option[String]) => T): Seq[T] = {
    maybeParameter match {
      case ot: ts.ObjectType => {
        ot.properties
          .flatMap(prop => {
            TypeHelper.propName(prop).map((propName) => toSetting(propName, prop.value, prop.getDocumentation()))
          })
      }
      case rt: ts.ReferenceType => toSettings(rt.resolveType(), toSetting)
      case it: ts.IntersectionType =>
        TypeHelper.resolveIntersection(it.of) match {
          case otherType if !otherType.isInstanceOf[ts.IntersectionType] => toSettings(otherType, toSetting)
        }
      case ut: ts.UnionType => {
        ut.of.map(toSettings(_, toSetting)).reduce(_ ++ _)
      }
    }
  }

  @tailrec
  def toDataFormatSetting(name: String, value: WeaveType, kvpDocumentation: Option[String]): DataFormatSetting[_] = {
    val defaultValue = value.getMetadataConstraint("defaultValue").map(_.value)
    val options = value.getMetadataConstraint("possibleValues").map(_.value.toString.split(",").map(_.trim).toSeq)
    value match {
      case _: ts.StringType  => StringSetting(name, defaultValue = defaultValue.map(_.toString).orNull, possibleValues = options.getOrElse(Seq()))
      case _: ts.BooleanType => BooleanSetting(name, defaultValue = defaultValue.forall(_.toString.toBoolean))
      case _: ts.NumberType  => IntSetting(name, defaultValue = defaultValue.map(_.toString.toInt).getOrElse(0))
      case rt: ReferenceType => toDataFormatSetting(name, rt.resolveType(), kvpDocumentation)
      case _                 => StringSetting(name, defaultValue = null)
    }
  }
}
