package org.mule.weave.v2.completion

import scala.collection.mutable.ArrayBuffer

class Template {

  val elements: ArrayBuffer[Element] = ArrayBuffer()

  def add(literal: String): Template = {
    elements.+=(LiteralElement(literal))
    this
  }

  def space() = add(" ")

  def prepend(literal: String) = {
    elements.prepend(LiteralElement(literal))
    this
  }

  def addnl(literal: String): Template = {
    add(literal + "\n")
  }

  def newLine(): Template = add("\n")

  def placeHolder(defaultValue: String): Template = {
    elements.+=(PlaceHolderElement(Some(defaultValue)))
    this
  }

  def placeHolder(): Template = {
    elements.+=(PlaceHolderElement())
    this
  }

  def end(): Template = {
    elements.+=(EndPlaceHolderElement())
    this
  }

  def choice(options: Seq[String]): Template = {
    elements.+=(ChoicePlaceHolderElement(options))
    this
  }

  /**
    * Create template definition according to https://code.visualstudio.com/docs/editor/userdefinedsnippets
    *
    * @return
    */
  def toVSCodeString: String = {
    var placeHolderIndex: Int = 1
    val result = new StringBuilder
    elements.foreach {
      case LiteralElement(content) => {
        result.append(content)
      }
      case PlaceHolderElement(defaultValue) => {
        result.append("${" + placeHolderIndex + (if (defaultValue.isDefined) ":" + defaultValue.get else "") + "}")
        placeHolderIndex = placeHolderIndex + 1
      }
      case ChoicePlaceHolderElement(options) => {
        result.append("${" + placeHolderIndex + "|" + options.mkString(",") + "|}")
        placeHolderIndex = placeHolderIndex + 1
      }
      case EndPlaceHolderElement() => {
        result.append("${0}")
      }
    }
    result.toString()
  }

  /**
    * Returns the eclipse representation of this template
    * @return
    */
  def toEclipseTemplate: EclipseTemplate = {
    var placeHolderIndex: Int = 1
    val variables = ArrayBuffer[EclipseTemplateVariable]()
    val result = new StringBuilder
    elements.foreach {
      case LiteralElement(content) => {
        result.append(content)
      }
      case PlaceHolderElement(defaultValue) => {
        if (defaultValue.isEmpty) {
          result.append("${}")
        } else {
          val varName = s"var${placeHolderIndex}"
          variables.+=(EclipseTemplateVariable(varName, defaultValue.toArray))
          result.append("${" + varName + "}")
        }
        placeHolderIndex = placeHolderIndex + 1
      }
      case ChoicePlaceHolderElement(options) => {
        val varName = s"var${placeHolderIndex}"
        variables.+=(EclipseTemplateVariable(varName, options.toArray))
        result.append("${" + varName + "}")
        placeHolderIndex = placeHolderIndex + 1
      }
      case EndPlaceHolderElement() => {

      }
    }
    EclipseTemplate(result.toString(), variables.toArray)
  }

  /**
    * Returns a literal string
    * @return
    */
  def toLiteralString: String = {
    val result = new StringBuilder
    elements.foreach {
      case LiteralElement(content) => {
        result.append(content)
      }
      case PlaceHolderElement(defaultValue) => {
        result.append(if (defaultValue.isDefined) defaultValue.get else "")
      }
      case ChoicePlaceHolderElement(options) => {
        result.append(options.head)
      }
      case EndPlaceHolderElement() => {
        result.append("")
      }
    }
    result.toString()
  }

  /**
    * Create the template definition according to https://www.jetbrains.com/help/idea/edit-template-variables-dialog.html
    *
    * @return
    */
  def toIntellijTemplate: IntellijTemplate = {
    var placeHolderIndex: Int = 1
    val variables = ArrayBuffer[IntellijTemplateVariable]()
    val result = new StringBuilder
    elements.foreach {
      case LiteralElement(content) => {
        result.append(content)
      }
      case PlaceHolderElement(defaultValue) => {
        val varName = s"var${placeHolderIndex}"
        variables.+=(IntellijTemplateVariable(varName, defaultValue))
        result.append("$" + varName + "$")
        placeHolderIndex = placeHolderIndex + 1
      }
      case ChoicePlaceHolderElement(options) => {
        val varName = s"var${placeHolderIndex}"
        variables.+=(IntellijTemplateVariable(varName, Some("enum(" + options.map(value => "\"" + value + "\"").mkString(", ") + ")")))
        result.append("$" + varName + "$")
        placeHolderIndex = placeHolderIndex + 1
      }
      case EndPlaceHolderElement() => {
        result.append("$END$")
      }
    }
    IntellijTemplate(result.toString(), variables.toArray)
  }

}

case class IntellijTemplate(text: String, variables: Array[IntellijTemplateVariable])

case class IntellijTemplateVariable(name: String, defaultValue: Option[String])

case class EclipseTemplate(text: String, variables: Array[EclipseTemplateVariable])

case class EclipseTemplateVariable(name: String, values: Array[String])

object Template {

  def apply(): Template = new Template()

  def apply(text: String): Template = new Template().add(text)
}

sealed trait Element {}

case class LiteralElement(content: String) extends Element

case class PlaceHolderElement(defaultValue: Option[String] = None) extends Element

case class ChoicePlaceHolderElement(options: Seq[String]) extends Element

case class EndPlaceHolderElement() extends Element