package org.mule.weave.v2.editor.indexing

import org.mule.weave.v2.editor.VirtualFile
import org.mule.weave.v2.parser.DocumentParser
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.header.directives.AnnotationDirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.FunctionDirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.ImportDirective
import org.mule.weave.v2.parser.ast.header.directives.NamespaceDirective
import org.mule.weave.v2.parser.ast.header.directives.TypeDirective
import org.mule.weave.v2.parser.ast.header.directives.VarDirective
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.ParsingResult
import org.mule.weave.v2.parser.phase.PhaseResult

import scala.collection.mutable.ArrayBuffer

/**
  * Returns the list of global available symbols.
  */
class DefaultWeaveIndexer extends WeaveIndexer {
  var result: PhaseResult[ParsingResult[AstNode]] = _

  override def parse(vf: VirtualFile, context: ParsingContext): Boolean = {
    result = DocumentParser().parse(vf.asResource(), context)
    result.hasResult()
  }

  override def identifiers(): Array[WeaveIdentifier] = {
    references() ++ declarations()
  }

  private def references(): Array[WeaveIdentifier] = {
    val collector = ArrayBuffer[WeaveIdentifier]()
    AstNodeHelper.traverse(
      result.getResult().astNode,
      (node) => {
        node match {
          case vrn: VariableReferenceNode => {
            val variable = vrn.variable
            collector.+=(asReference(variable, IdentifierType.VARIABLE))
          }
          case trn: TypeReferenceNode => {
            collector.+=(asReference(trn.variable, IdentifierType.TYPE))
          }
          case trn: AnnotationNode => {
            collector.+=(asReference(trn.name, IdentifierType.ANNOTATION))
          }
          case trn: NamespaceNode => {
            collector.+=(asReference(trn.prefix, IdentifierType.NAMESPACE))
          }
          case id: ImportDirective => {
            id.subElements.elements.foreach((element) => {
              if (element.elementName != NameIdentifier.$star) {
                collector.+=(asReference(element.elementName, IdentifierType.IMPORT))
              }
            })
          }
          case _ => {}
        }
        true
      })
    collector.toArray
  }

  private def asReference(variable: NameIdentifier, annotation: Int): WeaveIdentifier = {
    val location = variable.location()
    WeaveIdentifier(location.startPosition.index, location.endPosition.index, variable.name, annotation, IdentifierKind.REFERENCE)
  }

  private def declarations(): Array[WeaveIdentifier] = {

    if (result.hasResult()) {
      result.getResult().astNode match {
        case module: ModuleNode => {
          val directives = module.directives
          val symbols = directives.flatMap({
            case annotation: AnnotationDirectiveNode => {
              val location = annotation.location()
              Some(WeaveIdentifier(location.startPosition.index, location.endPosition.index, annotation.nameIdentifier.name, IdentifierType.ANNOTATION, IdentifierKind.DEFINITION))
            }
            case fnc: FunctionDirectiveNode => {
              val location = fnc.location()
              Some(WeaveIdentifier(location.startPosition.index, location.endPosition.index, fnc.variable.name, IdentifierType.FUNCTION, IdentifierKind.DEFINITION))
            }
            case namespace: NamespaceDirective => {
              val location = namespace.location()
              Some(WeaveIdentifier(location.startPosition.index, location.endPosition.index, namespace.prefix.name, IdentifierType.NAMESPACE, IdentifierKind.DEFINITION))
            }
            case typeD: TypeDirective => {
              val location = typeD.location()
              Some(WeaveIdentifier(location.startPosition.index, location.endPosition.index, typeD.variable.name, IdentifierType.TYPE, IdentifierKind.DEFINITION))
            }
            case varD: VarDirective => {
              val location = varD.location()
              Some(WeaveIdentifier(location.startPosition.index, location.endPosition.index, varD.variable.name, IdentifierType.VARIABLE, IdentifierKind.DEFINITION))
            }
            case _ => None
          })
          symbols.toArray
        }
        case _ => Array.empty
      }
    } else {
      Array.empty
    }
  }

  override def document(): WeaveDocument = {
    result.getResult().astNode match {
      case dn: DocumentNode => {
        WeaveDocument(dn.weaveDoc.map(_.literalValue).getOrElse(""), DocumentKind.MAPPING)
      }
      case m => {
        WeaveDocument(m.weaveDoc.map(_.literalValue).getOrElse(""), DocumentKind.MODULE)
      }
    }
  }
}
