package org.mule.weave.v2.weavedoc

import org.mule.weave.v2.api.tooling.ast.DWAstNodeKind
import org.mule.weave.v2.grammar.location.PositionTracking
import org.mule.weave.v2.parser.SafeStringBasedParserInput
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.Children
import org.mule.weave.v2.parser.ast.LiteralValueAstNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.parboiled2.CharPredicate
import org.parboiled2.Parser
import org.parboiled2.Rule0
import org.parboiled2.Rule1
import org.parboiled2.StringBuilding

class WeaveDocGrammar(content: String) extends Parser with StringBuilding with PositionTracking {

  def resourceName: NameIdentifier = NameIdentifier.ANONYMOUS_NAME

  val input = new SafeStringBasedParserInput(content)

  private def createDescriptionNode = (content: String) => DescriptionNode(content)

  private def createContentNode = (content: String) => {
    TextContentNode(content)
  }

  private def createParametersSectionNode = (style: TextContentNode, params: Seq[ParameterNode]) => {
    ParametersSectionNode(style, params)
  }

  private def createParameterNode = (name: TextContentNode, description: TextContentNode) => {
    ParameterNode(name, None, description)
  }

  private def createParameterNodeWithType = (name: TextContentNode, weaveType: TextContentNode, description: TextContentNode) => {
    ParameterNode(name, Some(weaveType), description)
  }

  private def createMoreExampleSectionNode = (content: Option[TextContentNode]) => MoreExamplesSectionNode(content.getOrElse(TextContentNode("")))

  private def createExampleSectionNode = (content: Option[TextContentNode], sections: Seq[ExampleInternalSection]) => ExampleSectionNode(content, sections)

  private def createCodeBlockDescription = (langType: String) => CodeBlockInfoNode(langType)

  private def createCodeBlock = (info: Option[CodeBlockInfoNode], content: String) => CodeBlockNode(content, info)

  private def createInputSection = (content: Seq[CodeBlockWithDescriptionNode]) => InputSectionNode(content)

  private def createSourceSection = (content: CodeBlockNode) => SourceSectionNode(content)

  private def createOutputSection = (content: Seq[CodeBlockWithDescriptionNode]) => OutputSectionNode(content)

  private def createCodeBlockWithDescriptionNode = (description: Option[DescriptionNode], code: CodeBlockNode) => CodeBlockWithDescriptionNode(description, code)

  private def createWeaveDoc = (description: Option[DescriptionNode], content: Option[TextContentNode], sections: Seq[WeaveDocSection]) => WeaveDocNode(description, content, sections)

  private val whiteSpaceChar = CharPredicate(" \f\t")

  private val newLineChar = CharPredicate("\r\n")

  def newLine: Rule0 = rule {
    quiet(str("\r\n") | newLineChar)
  }

  def ws: Rule0 = rule {
    quiet(zeroOrMore(whiteSpaceChar | newLine))
  }

  def fws: Rule0 = rule {
    quiet(oneOrMore(whiteSpaceChar | newLine))
  }

  def wsnoeol: Rule0 = rule {
    quiet(zeroOrMore(whiteSpaceChar) ~ optional(eol))
  }

  def eol: Rule0 = rule {
    quiet(oneOrMore(whiteSpaceChar) ~ newLine)
  }

  def weaveDoc: Rule1[WeaveDocNode] = rule {
    pushPosition ~ (optional(description) ~ optional(content) ~ zeroOrMore(sections) ~ EOI) ~> createWeaveDoc ~ injectPosition
  }

  def sections: Rule1[WeaveDocSection] = rule {
    parametersSection | exampleSection | moreExamplesSection
  }

  def description = namedRule("Description") {
    pushPosition ~ ((clearSB() ~ ws ~ oneOrMore(!(newLine ~ newLine) ~ !anyKeyword ~ ANY ~ appendSB()) ~ push(sb.toString)) ~> createDescriptionNode) ~ injectPosition
  }

  def content: Rule1[TextContentNode] = rule {
    pushPosition ~ ((clearSB() ~ ws ~ zeroOrMore(!anyKeyword ~ ANY ~ appendSB()) ~ push(sb.toString)) ~> createContentNode) ~ injectPosition
  }

  def exampleSection: Rule1[ExampleSectionNode] = namedRule("=== Example") {
    pushPosition ~ ((exampleKeyword ~!~ ws ~ optional(content) ~ zeroOrMore(exampleInternalSection)) ~> createExampleSectionNode) ~ injectPosition
  }

  def exampleInternalSection: Rule1[ExampleInternalSection] = namedRule("=== Source or ==== Input or ==== Output") {
    sourceSection | inputSection | outputSection
  }

  def moreExamplesSection: Rule1[ExampleSectionNode] = namedRule("=== More Examples") {
    pushPosition ~ ((moreExamplesKeyword ~!~ ws ~ optional(content)) ~> createMoreExampleSectionNode) ~ injectPosition
  }

  def sourceSection: Rule1[SourceSectionNode] = namedRule("=== Source") {
    pushPosition ~ (sourceKeyword ~!~ ws ~ codeBlock ~> createSourceSection) ~ injectPosition
  }

  def inputSection: Rule1[InputSectionNode] = namedRule("==== Input") {
    pushPosition ~ ((inputKeyword ~!~ ws ~ oneOrMore(codeBlockWithDescription)) ~> createInputSection) ~ injectPosition
  }

  def outputSection: Rule1[OutputSectionNode] = namedRule("==== Output") {
    pushPosition ~ ((outputKeyword ~!~ ws ~ oneOrMore(codeBlockWithDescription)) ~> createOutputSection) ~ injectPosition
  }

  def codeBlockWithDescription: Rule1[CodeBlockWithDescriptionNode] = rule {
    pushPosition ~ ((push(None) ~ codeBlock | optional(description) ~ codeBlock) ~> createCodeBlockWithDescriptionNode) ~ injectPosition
  }

  def parametersSection: Rule1[ParametersSectionNode] = namedRule("=== Parameters") {
    pushPosition ~ ((parametersKeyword ~!~ ws ~ parametersStyle ~ ws ~ tableBodyKeyword ~ ws ~!~ (parameters | parametersWithTypes)) ~> createParametersSectionNode ~ ws ~ tableBodyKeyword) ~ injectPosition
  }

  def parameters: Rule1[Seq[ParameterNode]] = namedRule("Parameters") {
    ws ~ pipeKeyword ~ ws ~ str("Name") ~ ws ~ pipeKeyword ~ ws ~ str("Description") ~ ws ~!~ oneOrMore(parameter)
  }

  def parametersWithTypes: Rule1[Seq[ParameterNode]] = namedRule("Parameters with types") {
    ws ~ pipeKeyword ~ ws ~ str("Name") ~ ws ~ pipeKeyword ~ ws ~ str("Type") ~ ws ~ pipeKeyword ~ ws ~ str("Description") ~ ws ~!~ oneOrMore(parameterWithType)
  }

  def parameter: Rule1[ParameterNode] = namedRule("Parameter") {
    pushPosition ~ (ws ~ pipeKeyword ~ ws ~!~ parameterName ~ ws ~ pipeKeyword ~ ws ~!~ parameterContent ~ ws ~> createParameterNode) ~ injectPosition
  }

  def parameterWithType: Rule1[ParameterNode] = namedRule("Parameter with type") {
    pushPosition ~ (ws ~ pipeKeyword ~ ws ~!~ parameterName ~ ws ~ pipeKeyword ~ ws ~!~ parameterContent ~ ws ~ pipeKeyword ~ ws ~!~ parameterContent ~ ws ~> createParameterNodeWithType) ~ injectPosition
  }

  def parameterName: Rule1[TextContentNode] = namedRule("Parameter name") {
    pushPosition ~ ((clearSB() ~ ws ~ oneOrMore(!(ws ~ pipeKeyword ~ ws) ~ !anyKeyword ~ ANY ~ appendSB()) ~ push(sb.toString)) ~> createContentNode) ~ injectPosition
  }

  def parameterContent: Rule1[TextContentNode] = rule {
    pushPosition ~ ((clearSB() ~ zeroOrMore(!(ws ~ pipeKeyword ~ ws) ~ !anyKeyword ~ ANY ~ appendSB()) ~ push(sb.toString)) ~> createContentNode) ~ injectPosition
  }

  def parametersStyle: Rule1[TextContentNode] = namedRule("Parameters style") {
    pushPosition ~ ((clearSB() ~ zeroOrMore(!(ws ~ tableBodyKeyword) ~ ANY ~ appendSB()) ~ push(sb.toString)) ~> createContentNode) ~ injectPosition
  }

  //KEYWORDS
  def anyKeyword = namedRule("=== Example or === Parameters or === More Examples or ==== Source or ==== Input or ==== Output") {
    exampleKeyword | parametersKeyword | moreExamplesKeyword | sourceKeyword | inputKeyword | outputKeyword
  }

  def toplevelSectionKeyword = rule {
    exampleKeyword | parametersKeyword | moreExamplesKeyword
  }

  def codeBlockInfo = namedRule("Code Block") {
    pushPosition ~ (ws ~ ch('[') ~!~ ws ~ str("source") ~!~ ws ~ ch(',') ~ ws ~!~ languageType ~ ws ~!~ optional(ch(',') ~ ws ~ atomic(str("linenums")) ~ ws) ~ ch(']') ~> createCodeBlockDescription) ~ injectPosition
  }

  def languageType = namedRule("Language Type") {
    quiet(clearSB() ~ oneOrMore(CharPredicate.Alpha ~ appendSB()) ~ push(sb.toString.trim))
  }

  def codeBlock: Rule1[CodeBlockNode] = rule {
    pushPosition ~ (ws ~ optional(codeBlockInfo) ~ ws ~ codeBlockSeparatorKeyword ~!~ codeText ~ ws ~!~ codeBlockSeparatorKeyword ~ ws ~> createCodeBlock) ~ injectPosition
  }

  def codeText = namedRule("Code") {
    (clearSB() ~ oneOrMore(!codeBlockSeparatorKeyword ~ ANY ~ appendSB()) ~ push(sb.toString))
  }

  def codeBlockSeparatorKeyword: Rule0 = namedRule("----") {
    atomic(str("----") ~ (fws | EOI))
  }

  def parametersKeyword: Rule0 = namedRule("Parameters") {
    atomic(str("=== Parameters"))
  }

  def exampleKeyword: Rule0 = namedRule("Example") {
    atomic(str("=== Example"))
  }

  def moreExamplesKeyword: Rule0 = namedRule("More Examples") {
    atomic(str("=== More Examples"))
  }

  def tableBodyKeyword: Rule0 = namedRule("|===") {
    atomic(str("|===") ~ (fws | EOI))
  }

  def sourceKeyword: Rule0 = namedRule("Source") {
    atomic(str("==== Source"))
  }

  def inputKeyword: Rule0 = namedRule("Input") {
    atomic(str("==== Input"))
  }

  def outputKeyword: Rule0 = namedRule("Output") {
    atomic(str("==== Output"))
  }

  def pipeKeyword: Rule0 = namedRule("|") {
    atomic(ch('|'))
  }
}

case class WeaveDocNode(description: Option[DescriptionNode], content: Option[TextContentNode], sections: Seq[WeaveDocSection]) extends AstNode {
  override def children(): Seq[AstNode] = Children().++=(sections).++=(description).+=(content).result()

  override protected def doClone(): AstNode = {
    copy(
      description.map(_.cloneAst().asInstanceOf[DescriptionNode]),
      content.map(_.cloneAst().asInstanceOf[TextContentNode]),
      sections.map(_.cloneAst().asInstanceOf[WeaveDocSection]))
  }

  override def getKind(): String = DWAstNodeKind.DOCUMENT_NODE
}

case class CodeBlockInfoNode(langType: String) extends AstNode with LiteralValueAstNode {
  override def literalValue: String = langType

  override protected def doClone(): AstNode = {
    copy()
  }

  override def getKind(): String = DWAstNodeKind.CODE_BLOCK_INFO_NODE
}

case class CodeBlockNode(content: String, info: Option[CodeBlockInfoNode]) extends AstNode with LiteralValueAstNode {

  override def children(): Seq[AstNode] = Seq() ++ info

  override def literalValue: String = content

  override def doClone(): AstNode = {
    copy(content, info.map(_.cloneAst().asInstanceOf[CodeBlockInfoNode]))
  }

  override def getKind(): String = DWAstNodeKind.CODE_BLOCK_NODE
}

case class TextContentNode(content: String) extends AstNode with LiteralValueAstNode {
  override def literalValue: String = content

  override protected def doClone(): AstNode = {
    copy()
  }

  override def getKind(): String = DWAstNodeKind.TEXT_CONTENT_NODE
}

case class DescriptionNode(content: String) extends AstNode with LiteralValueAstNode {
  override def literalValue: String = content

  override protected def doClone(): AstNode = {
    copy()
  }

  override def getKind(): String = DWAstNodeKind.DESCRIPTION_NODE
}

sealed trait WeaveDocSection extends AstNode

case class ParametersSectionNode(style: TextContentNode, params: Seq[ParameterNode] = Seq()) extends WeaveDocSection {

  override def children(): Seq[AstNode] = Children().+=(style).++=(params).result()

  override protected def doClone(): AstNode = {
    copy(style.cloneAst().asInstanceOf[TextContentNode], params.map(_.cloneAst().asInstanceOf[ParameterNode]))
  }

  override def getKind(): String = DWAstNodeKind.PARAMETERS_DOC_NODE
}

case class ParameterNode(name: TextContentNode, weaveType: Option[TextContentNode], description: TextContentNode) extends AstNode {

  override def children(): Seq[AstNode] = Children().+=(name).+=(weaveType).+=(description).result()

  override protected def doClone(): AstNode = {
    copy(name.cloneAst().asInstanceOf[TextContentNode], weaveType.map(_.cloneAst().asInstanceOf[TextContentNode]), description.cloneAst().asInstanceOf[TextContentNode])
  }

  override def getKind(): String = DWAstNodeKind.PARAMETER_NODE
}

case class ExampleSectionNode(descriptionNode: Option[TextContentNode], sections: Seq[ExampleInternalSection]) extends WeaveDocSection {
  override def children(): Seq[AstNode] = Children().++=(sections).++=(descriptionNode).result()

  override protected def doClone(): AstNode =
    copy(descriptionNode.map(_.cloneAst().asInstanceOf[TextContentNode]), sections.map(_.cloneAst().asInstanceOf[ExampleInternalSection]))

  override def getKind(): String = DWAstNodeKind.EXAMPLE_SECTION_DOC_NODE
}

sealed trait ExampleInternalSection extends AstNode

case class InputSectionNode(content: Seq[CodeBlockWithDescriptionNode]) extends ExampleInternalSection {
  override def children(): Seq[AstNode] = content

  override protected def doClone(): AstNode = {
    copy(content.map(_.cloneAst().asInstanceOf[CodeBlockWithDescriptionNode]))
  }

  override def getKind(): String = DWAstNodeKind.INPUT_EXAMPLE_SECTION_NODE
}

case class SourceSectionNode(content: CodeBlockNode) extends ExampleInternalSection {
  override def children(): Seq[AstNode] = Seq(content)

  override protected def doClone(): AstNode = {
    copy(content.cloneAst().asInstanceOf[CodeBlockNode])
  }

  override def getKind(): String = DWAstNodeKind.SOURCE_EXAMPLE_SECTION_NODE
}

case class OutputSectionNode(content: Seq[CodeBlockWithDescriptionNode]) extends ExampleInternalSection {

  override def children(): Seq[AstNode] = content

  override protected def doClone(): AstNode = {
    copy(content.map(_.cloneAst().asInstanceOf[CodeBlockWithDescriptionNode]))
  }

  override def getKind(): String = DWAstNodeKind.OUTPUT_EXAMPLE_SECTION_NODE
}

case class CodeBlockWithDescriptionNode(description: Option[DescriptionNode], code: CodeBlockNode) extends AstNode {
  override def children(): Seq[AstNode] = Children().++=(description).+=(code).result()

  override protected def doClone(): AstNode = {
    copy(description.map(_.cloneAst().asInstanceOf[DescriptionNode]), code.cloneAst().asInstanceOf[CodeBlockNode])
  }

  override def getKind(): String = DWAstNodeKind.CODE_BLOCK_INFO_NODE
}

case class MoreExamplesSectionNode(descriptionNode: TextContentNode) extends WeaveDocSection {
  override def children(): Seq[AstNode] = Children().+=(descriptionNode).result()

  override protected def doClone(): AstNode = {
    copy(descriptionNode.cloneAst().asInstanceOf[TextContentNode])
  }

  override def getKind(): String = DWAstNodeKind.MORE_EXAMPLES_DOC_NODE
}