package org.mule.weave.v2.parser.ast.header.directives

import org.mule.weave.v2.api.tooling.ast.DWAstNodeKind
import org.mule.weave.v2.grammar.Tokens
import org.mule.weave.v2.parser.ErrorAstNode
import org.mule.weave.v2.parser.Message
import org.mule.weave.v2.parser.MessageKind
import org.mule.weave.v2.parser.ParsingPhaseCategory
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.Child
import org.mule.weave.v2.parser.ast.Children
import org.mule.weave.v2.parser.ast.ContainerAstNode
import org.mule.weave.v2.parser.ast.MutableAstNode
import org.mule.weave.v2.parser.ast.NamedAstNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationCapableNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.structure.UriNode
import org.mule.weave.v2.parser.ast.types.TypeParametersListNode
import org.mule.weave.v2.parser.ast.types.WeaveTypeNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

/**
  * DataWeave Directives such as out or in
  */
sealed trait DirectiveNode extends AstNode with AnnotationCapableNode {
  def name: String

  override def cloneAst(): DirectiveNode = super.cloneAst().asInstanceOf[DirectiveNode]

}

case class AnnotationDirectiveNode(nameIdentifier: NameIdentifier, params: AnnotationParametersNode, var codeAnnotations: Seq[AnnotationNode] = Seq()) extends DirectiveNode with NamedAstNode {
  override def children(): Seq[AstNode] = Children(nameIdentifier, params).++=(codeAnnotations).result()

  override def name: String = "annotation"

  override protected def doClone(): DirectiveNode = copy(nameIdentifier.cloneAst(), params.cloneAst().asInstanceOf[AnnotationParametersNode], codeAnnotations.map(_.cloneAst()))

  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = {
    this.codeAnnotations = annotations
  }

  override def getKind(): String = DWAstNodeKind.ANNOTATION_DIRECTIVE_NODE
}

case class AnnotationParametersNode(paramList: Seq[AnnotationParameterNode]) extends AstNode {
  override def children(): Seq[AstNode] = paramList

  override protected def doClone(): AstNode = copy(paramList.map(_.cloneAst().asInstanceOf[AnnotationParameterNode]))

  override def getKind(): String = DWAstNodeKind.ANNOTATION_PARAMETERS_NODE
}

case class AnnotationParameterNode(nameIdentifier: NameIdentifier, weaveType: WeaveTypeNode, var defaultValue: Option[AstNode]) extends AstNode with MutableAstNode {
  override def children(): Seq[AstNode] = Child(nameIdentifier, weaveType, defaultValue)

  override protected def doClone(): AstNode = copy(nameIdentifier.cloneAst(), weaveType.cloneAst(), defaultValue.map(_.cloneAst()))

  override def getKind(): String = DWAstNodeKind.ANNOTATION_PARAMETER_NODE

  override def update(toBeReplaced: AstNode, withNode: AstNode): Unit = {
    defaultValue.foreach(value => if (value eq toBeReplaced) defaultValue = Some(withNode))
  }
}

case class FunctionDirectiveNode(var variable: NameIdentifier, var literal: AstNode, var codeAnnotations: Seq[AnnotationNode] = Seq()) extends DirectiveNode with AstNode with AnnotationCapableNode with NamedAstNode {

  override def name: String = Tokens.FUNCTION

  override def children(): Seq[AstNode] = Children(variable, literal).++=(codeAnnotations).result()

  override protected def doClone(): FunctionDirectiveNode = copy(variable.cloneAst(), literal.cloneAst(), codeAnnotations.map(_.cloneAst()))

  override def cloneAst(): FunctionDirectiveNode = super.cloneAst().asInstanceOf[FunctionDirectiveNode]

  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = {
    this.codeAnnotations = annotations
  }

  override def nameIdentifier: NameIdentifier = variable

  override def getKind(): String = DWAstNodeKind.FUNCTION_DIRECTIVE_NODE
}

/**
  * Trait to mark both InputDirective and OutputDirective
  */
trait InputOutputDirective extends DirectiveNode {
  var dataFormat: Option[DataFormatId]
  var mime: Option[ContentType]
  var options: Option[Seq[DirectiveOption]]
  val wtype: Option[WeaveTypeNode]
}

case class DirectiveOption(var name: DirectiveOptionName, var value: AstNode) extends AstNode {
  override def children(): Seq[AstNode] = Child(name, value)

  override protected def doClone(): AstNode = copy(name.cloneAst().asInstanceOf[DirectiveOptionName], value.cloneAst())

  override def getKind(): String = DWAstNodeKind.DIRECTIVE_OPTION_NODE
}

case class DirectiveOptionName(var name: String) extends AstNode {
  override protected def doClone(): AstNode = copy()

  override def getKind(): String = DWAstNodeKind.DIRECTIVE_OPTION_NAME_NODE
}

case class InputDirective(var variable: NameIdentifier, var dataFormat: Option[DataFormatId], var mime: Option[ContentType], var options: Option[Seq[DirectiveOption]], wtype: Option[WeaveTypeNode], var codeAnnotations: Seq[AnnotationNode] = Seq()) extends InputOutputDirective with NamedAstNode {

  override def name: String = Tokens.INPUT

  override def children(): Seq[AstNode] = {
    Children().++=(mime).+=(variable).++=(dataFormat).++=(options).+=(wtype).++=(codeAnnotations).result()
  }

  override protected def doClone(): DirectiveNode =
    copy(
      variable.cloneAst(),
      dataFormat.map(w => w.cloneAst().asInstanceOf[DataFormatId]),
      mime.map(w => w.cloneAst().asInstanceOf[ContentType]),
      options.map((option) => option.map(_.cloneAst().asInstanceOf[DirectiveOption])),
      wtype.map(_.cloneAst()),
      codeAnnotations.map(_.cloneAst()))

  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = {
    this.codeAnnotations = annotations
  }

  override def nameIdentifier: NameIdentifier = variable

  override def getKind(): String = DWAstNodeKind.INPUT_DIRECTIVE_NODE
}

object InputDirective {
  def apply(variable: NameIdentifier, mime: ContentType, options: Option[Seq[DirectiveOption]], wtype: Option[WeaveTypeNode]): InputDirective = {
    new InputDirective(variable, None, Some(mime), options, wtype)
  }

  def apply(variable: NameIdentifier, dataFormatId: DataFormatId, options: Option[Seq[DirectiveOption]], wtype: Option[WeaveTypeNode]): InputDirective = {
    new InputDirective(variable, Some(dataFormatId), None, options, wtype)
  }

}

case class ImportDirective(var importedModule: ImportedElement, var subElements: ImportedElements = ImportedElements(Seq()), var codeAnnotations: Seq[AnnotationNode] = Seq()) extends DirectiveNode with AnnotationCapableNode {

  override def name: String = Tokens.IMPORT

  override def children(): Seq[AstNode] = Children(importedModule, subElements).++=(codeAnnotations).result()

  override protected def doClone(): DirectiveNode = {
    copy(importedModule.cloneAst().asInstanceOf[ImportedElement], subElements.cloneAst().asInstanceOf[ImportedElements], codeAnnotations.map(_.cloneAst()))
  }

  def isImportStart(): Boolean = {
    subElements.elements.headOption.exists((n) => NameIdentifier.isStarImport(n.elementName))
  }

  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = {
    this.codeAnnotations = annotations
  }

  override def getKind(): String = DWAstNodeKind.IMPORT_DIRECTIVE_NODE
}

case class ImportedElements(var elements: Seq[ImportedElement]) extends AstNode {
  override def children(): Seq[AstNode] = elements

  override protected def doClone(): AstNode = {
    copy(elements.map(_.cloneAst().asInstanceOf[ImportedElement]))
  }

  override def getKind(): String = DWAstNodeKind.IMPORTED_ELEMENTS_NODE
}

/**
  * It can be a module, function, variable or any importable element.
  */
case class ImportedElement(elementName: NameIdentifier, alias: Option[NameIdentifier] = None) extends ContainerAstNode {
  override def children(): Seq[AstNode] = Children(elementName).++=(alias).result()

  def aliasOrLocalName: NameIdentifier = alias.getOrElse(elementName.localName())

  override protected def doClone(): AstNode = {
    copy(elementName.cloneAst(), alias.map(_.cloneAst()))
  }

  override def getKind(): String = DWAstNodeKind.IMPORTED_ELEMENT_NODE
}

case class NamespaceDirective(var prefix: NameIdentifier, var uri: UriNode, var codeAnnotations: Seq[AnnotationNode] = Seq()) extends DirectiveNode {

  override def name: String = Tokens.NS

  override def children(): Seq[AstNode] = {
    Children(prefix, uri).++=(codeAnnotations).result()
  }

  override protected def doClone(): DirectiveNode = {
    copy(prefix.cloneAst(), uri.cloneAst().asInstanceOf[UriNode], codeAnnotations.map(_.cloneAst()))
  }

  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = {
    this.codeAnnotations = annotations
  }

  override def getKind(): String = DWAstNodeKind.NAMESPACE_DIRECTIVE_NODE
}

case class OutputDirective(var dataFormat: Option[DataFormatId], var mime: Option[ContentType], var options: Option[Seq[DirectiveOption]], wtype: Option[WeaveTypeNode], var codeAnnotations: Seq[AnnotationNode] = Seq()) extends InputOutputDirective {

  override def name: String = Tokens.OUTPUT

  override def children(): Seq[AstNode] = Children().++=(mime).++=(dataFormat).++=(options).+=(wtype).++=(codeAnnotations).result()

  override protected def doClone(): DirectiveNode = {
    copy(
      dataFormat.map(w => w.cloneAst().asInstanceOf[DataFormatId]),
      mime.map(m => m.cloneAst().asInstanceOf[ContentType]),
      options.map((option) => option.map(_.cloneAst().asInstanceOf[DirectiveOption])),
      wtype.map(_.cloneAst()),
      codeAnnotations.map(_.cloneAst()))
  }

  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = {
    this.codeAnnotations = annotations
  }

  override def getKind(): String = DWAstNodeKind.OUTPUT_DIRECTIVE_NODE
}

object OutputDirective {
  def apply(mime: ContentType, options: Option[Seq[DirectiveOption]], wtype: Option[WeaveTypeNode]): OutputDirective = {
    new OutputDirective(None, Some(mime), options, wtype)
  }

  def apply(id: Option[DataFormatId], mime: ContentType, options: Option[Seq[DirectiveOption]], wtype: Option[WeaveTypeNode]): OutputDirective = {
    new OutputDirective(id, Some(mime), options, wtype)
  }

  def apply(id: DataFormatId, options: Option[Seq[DirectiveOption]], wtype: Option[WeaveTypeNode]): OutputDirective = {
    new OutputDirective(Some(id), None, options, wtype)
  }
}

case class TypeDirective(var variable: NameIdentifier, typeParametersListNode: Option[TypeParametersListNode], var typeExpression: WeaveTypeNode, var codeAnnotations: Seq[AnnotationNode] = Seq()) extends DirectiveNode with NamedAstNode {

  override def name: String = Tokens.TYPE

  override def children(): Seq[AstNode] = Children(variable).+=(typeParametersListNode).+=(typeExpression).++=(codeAnnotations).result()

  override protected def doClone(): DirectiveNode = {
    copy(variable.cloneAst(), typeParametersListNode.map(_.cloneAst()), typeExpression.cloneAst(), codeAnnotations.map(_.cloneAst()))
  }

  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = {
    this.codeAnnotations = annotations
  }

  override def nameIdentifier: NameIdentifier = variable

  override def getKind(): String = DWAstNodeKind.TYPE_DIRECTIVE_NODE
}

case class VarDirective(var variable: NameIdentifier, var value: AstNode, var wtype: Option[WeaveTypeNode] = None, var codeAnnotations: Seq[AnnotationNode] = Seq()) extends DirectiveNode with MutableAstNode with NamedAstNode {

  override def name: String = Tokens.VAR

  override def children(): Seq[AstNode] = Children(variable, value).+=(wtype).++=(codeAnnotations).result()

  override protected def doClone(): DirectiveNode = {
    copy(variable.cloneAst(), value.cloneAst(), wtype.map(_.cloneAst()), codeAnnotations.map(_.cloneAst()))
  }

  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = {
    this.codeAnnotations = annotations
  }

  override def update(toBeReplaced: AstNode, withNode: AstNode): Unit = {
    if (toBeReplaced eq value) {
      this.value = withNode
    }
  }

  override def nameIdentifier: NameIdentifier = {
    variable
  }

  override def getKind(): String = DWAstNodeKind.VAR_DIRECTIVE_NODE
}

case class VersionMajor(var v: String) extends AstNode {
  override protected def doClone(): AstNode = copy()

  override def getKind(): String = DWAstNodeKind.VERSION_MAJOR_NODE
}

case class VersionMinor(var v: String) extends AstNode {
  override protected def doClone(): AstNode = copy()

  override def getKind(): String = DWAstNodeKind.VERSION_MINOR_NODE
}

case class VersionDirective(major: VersionMajor, minor: VersionMinor, var codeAnnotations: Seq[AnnotationNode] = Seq()) extends DirectiveNode {
  override def name: String = Tokens.VERSION

  def this(major: String, minor: String) = {
    this(VersionMajor(major), VersionMinor(minor))
  }

  def this() = {
    this(VersionMajor("2"), VersionMinor("0"))
  }

  override def children(): Seq[AstNode] = Children(major, minor).++=(codeAnnotations).result()

  override protected def doClone(): DirectiveNode = copy(major.cloneAst().asInstanceOf[VersionMajor], minor.cloneAst().asInstanceOf[VersionMinor], codeAnnotations.map(_.cloneAst()))

  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = {
    this.codeAnnotations = annotations
  }

  override def getKind(): String = DWAstNodeKind.VERSION_DIRECTIVE_NODE
}

class ErrorDirectiveNode(errors: Seq[ErrorAstNode], var codeAnnotations: Seq[AnnotationNode] = Seq()) extends DirectiveNode with ErrorAstNode {
  override def name: String = "Error directive node"

  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = {
    this.codeAnnotations = annotations
  }

  override val literalValue: String = errors(0).literalValue

  override def children(): Seq[AstNode] = Children().++=(errors).++=(codeAnnotations).result()

  override protected def doClone(): DirectiveNode = new ErrorDirectiveNode(errors.map(_.cloneAst().asInstanceOf[ErrorAstNode]), codeAnnotations.map(_.cloneAst()))

  override def message(): Message = Message(MessageKind.ERROR_DIRECTIVE_MESSAGE_KIND, literalValue, ParsingPhaseCategory)

  override def getKind(): String = DWAstNodeKind.ERROR_DIRECTIVE_NODE
}

object ErrorDirectiveNode {
  def apply(error: ErrorAstNode) =
    new ErrorDirectiveNode(Seq(error))

  def apply(errors: Seq[ErrorAstNode]) =
    new ErrorDirectiveNode(errors)
}