package org.mule.weave.v2.sdk

import org.mule.weave.v2.grammar.Identifiers
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.utils.WeaveFile

import java.util.regex.Pattern

/**
  * Utility class for mapping name identifier and file system resources
  */
object NameIdentifierHelper {

  /**
    * Returns the extension with the dot of the NameIdentifier
    * @param name the NameIdentifier
    * @return The extension of the file
    */
  def extensionOf(name: NameIdentifier): String = {
    if (name.loader.isEmpty) {
      WeaveFile.fileExtension
    } else {
      "." + name.loader.get.toLowerCase
    }
  }

  val fileSeparator: String = System.getProperty("file.separator")

  def toWeaveFilePath(name: NameIdentifier): String = {
    toWeaveFilePath(name, fileSeparator)
  }

  def toWeaveFilePath(name: NameIdentifier, fileSeparator: String): String = {
    toWeaveFilePath(name, fileSeparator, extensionOf(name))
  }

  def toWeaveBinaryFilePath(name: NameIdentifier): String = {
    toWeaveFilePath(name, fileSeparator, WeaveFile.binaryFileExtension)
  }

  def toWeaveBinaryFilePath(name: NameIdentifier, fileSeparator: String): String = {
    toWeaveFilePath(name, fileSeparator, WeaveFile.binaryFileExtension)
  }

  def toWeaveFilePath(name: NameIdentifier, fileSeparator: String, extension: String): String = {
    s"$fileSeparator${name.name.replace("::", fileSeparator)}${`extension`}"
  }

  def fromWeaveFilePath(name: String): NameIdentifier = {
    fromWeaveFilePath(name, fileSeparator)
  }

  def fromWeaveFilePath(name: String, filePath: String): NameIdentifier = {
    var path: String = name
    val mayBeExtension = getExtension(name)
    if (mayBeExtension.isDefined) {
      path = path.substring(0, name.length - mayBeExtension.get.length)
    }
    if (path.startsWith(filePath)) {
      path = path.substring(1)
    }
    if (mayBeExtension.isDefined && mayBeExtension.get == WeaveFile.fileExtension) {
      NameIdentifier(path.replace(filePath, "::"))
    } else {
      //We remove the dot
      NameIdentifier(path.replace(filePath, "::"), mayBeExtension.map(_.substring(1)))
    }
  }

  private def getExtension(name: String): Option[String] = {
    if (name.lastIndexOf('.') >= 0) {
      Some(name.drop(name lastIndexOf '.'))
    } else {
      None
    }
  }

  val VALID_IDENTIFIER_PATTERN: Pattern = Pattern.compile("^[a-zA-Z][a-zA-Z0-9_]*$")

  def isValidIdentifier(name: String): Boolean = {
    VALID_IDENTIFIER_PATTERN.matcher(name).matches() && !Identifiers.keywords.contains(name)
  }

}
