package org.mule.weave.v2.ts

import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.functions._
import org.mule.weave.v2.parser.ast.header.directives._
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.scope.ScopesNavigator
import org.mule.weave.v2.ts.WeaveTypeTraverse.equalsWith
import org.mule.weave.v2.utils.ConcurrentIdentityHashMap

import scala.collection._
import scala.collection.mutable.ArrayBuffer

/**
  * Represents a type propagation graph
  *
  * @param nodes         The nodes in the graph
  * @param parentGraph   The parent graph
  * @param syntaxVersion The version of the syntax defined by the document
  */
class TypeGraph(val nodes: mutable.Seq[TypeNode], val parentGraph: Option[TypeGraph], val syntaxVersion: VersionDirective) {

  private val _subGraphs: ArrayBuffer[(FunctionNode, ArrayBuffer[(Seq[WeaveType], TypeGraph, MessageCollector)])] = ArrayBuffer()

  private lazy val nodesIndex: ConcurrentIdentityHashMap[AstNode, TypeNode] = {
    val identityHashMap = new ConcurrentIdentityHashMap[AstNode, TypeNode]()
    var i = 0;
    while (i < nodes.length) {
      identityHashMap.put(nodes(i).astNode, nodes(i))
      i = i + 1;

    }
    identityHashMap
  }

  //Set Parent to the children
  nodes.foreach(_.withParent(this))

  def rootGraph(): TypeGraph = {
    parentGraph.map(_.rootGraph()).getOrElse(this)
  }

  private def subGraphs(): Seq[TypeGraph] = {
    _subGraphs.flatMap((functionNodePair) => {
      functionNodePair._2.map((typeGraphPair) => {
        typeGraphPair._2
      })
    })
  }

  def subGraphsWithLabel(): Seq[(TypeGraph, String)] = {
    _subGraphs.flatMap((functionNodePair) => {
      functionNodePair._2.map((typeGraphPair) => {
        (typeGraphPair._2, typeGraphPair._1.mkString(","))
      })
    })
  }

  def removeFunctionSubGraph(functionNode: FunctionNode, parameterTypes: Seq[WeaveType]): Unit = {
    getFunctionSubGraphs(functionNode) match {
      case Some(x) => {
        x.find((graph) => matches(graph._1, parameterTypes)).map((function) => x.-=(function))
      }
      case None =>
    }
  }

  def getFunctionSubGraph(functionNode: FunctionNode, parameterTypes: Seq[WeaveType]): Option[(TypeGraph, MessageCollector)] = {
    getFunctionSubGraphs(functionNode) match {
      case Some(x) => {
        x.find((graph) => matches(graph._1, parameterTypes)).map(g => (g._2, g._3))
      }
      case None => None
    }
  }

  def getFunctionSubGraphs(functionNode: FunctionNode): Option[ArrayBuffer[(Seq[WeaveType], TypeGraph, MessageCollector)]] = {
    _subGraphs.find(_._1 eq functionNode).map(_._2)
  }

  def addFunctionSubGraph(functionNode: FunctionNode, parameterTypes: Seq[WeaveType], graph: TypeGraph, messageCollector: MessageCollector): Unit = {
    getFunctionSubGraphs(functionNode) match {
      case Some(x) => {
        x.+=((parameterTypes, graph, messageCollector))
      }
      case None => {
        val tuple = (functionNode, ArrayBuffer((parameterTypes, graph, messageCollector)))
        _subGraphs.+=(tuple)
      }
    }
  }

  private def matches(expected: Seq[WeaveType], actual: Seq[WeaveType]): Boolean = {
    if (expected.size == actual.size) {
      expected.zip(actual).forall((t) => equalsWith(t._1, t._2))
    } else {
      false
    }
  }

  def findLocalNode(astNode: AstNode): Option[TypeNode] = {
    val maybeNode = nodesIndex
      .get(astNode)
      .orElse({
        parentGraph
          .flatMap((typeGraph) => {
            typeGraph.findLocalNode(astNode)
          })
      })

    maybeNode
  }

  def findNode(astNode: AstNode): Option[TypeNode] = {

    findLocalNode(astNode) match {
      case None => {
        var ret: Option[TypeNode] = None
        var i = 0
        val sg = subGraphs()
        val subGraphsCount = sg.length
        while (i < subGraphsCount && ret.isEmpty) {
          val foundNode = sg(i).findNode(astNode)
          if (foundNode.isDefined) {
            ret = foundNode
          }
          i += 1
        }
        ret
      }
      case found => found
    }

  }

  def dispose(): Unit = {
    nodes.foreach(_.dispose())
    _subGraphs.foreach(_._2.foreach(_._2.dispose()))
  }
}

object TypeGraph {
  def apply(parsingContext: ParsingContext, scopesNavigator: ScopesNavigator, documentNode: AstNode, inputs: Map[String, Option[WeaveType]] = Map(), expectedOutput: Option[WeaveType] = None, parentGraph: Option[TypeGraph] = None): TypeGraph = {
    new TypeGraphBuilder(parsingContext, scopesNavigator, parentGraph, inputs, expectedOutput).build(documentNode)
  }

  def apply(parsingContext: ParsingContext, parentGraph: TypeGraph, scopesNavigator: ScopesNavigator, functionNode: FunctionNode, invocationParameters: Seq[WeaveType], returnType: Option[WeaveType], resolver: ReferenceResolver): TypeGraph = {
    new TypeGraphBuilder(parsingContext, scopesNavigator, Some(parentGraph), expectedOutput = returnType).build(functionNode, invocationParameters, resolver.cleanResolver())
  }
}
