package org.mule.weave.v2.utils

import org.mule.weave.v2.parser.exception.InvalidEscapeException
import org.mule.weave.v2.parser.location.Location
import org.mule.weave.v2.parser.location.UnknownLocation
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.sdk.NameIdentifierHelper

import scala.annotation.switch

object StringEscapeHelper {

  /**
    *  Returns the unescaped representations of the specified string
    *
    * @param str
    * @param quotedBy
    * @return
    */
  def unescapeString(str: String, quotedBy: Char = '"', location: Location = UnknownLocation): String = {
    val charIterator: Iterator[Char] = str.iterator
    val result = new StringBuilder
    while (charIterator.hasNext) {
      val next: Char = charIterator.next()
      if (next == '\\' && charIterator.hasNext) {
        val controlChar: Char = charIterator.next()
        if (controlChar == quotedBy) {
          result.append(controlChar)
        } else {
          (controlChar: @switch) match {
            case '/' | '\\' => result.append(controlChar)
            case 'n'        => result.append('\n')
            case 'b'        => result.append('\b')
            case 'f'        => result.append('\f')
            case 'r'        => result.append('\r')
            case 't'        => result.append('\t')
            case '$'        => result.append('$')
            case 'u' => {
              val sb = new StringBuilder()
              var i = 0
              while (i < 4) {
                if (charIterator.hasNext) {
                  sb.append(charIterator.next())
                }
                i += 1
              }
              try {
                val codePoint = Integer.parseInt(sb.mkString, 16) // escaped unicode chars are hex and code points are decimal
                val str1 = Character.toString(codePoint.asInstanceOf[Char])
                result.append(str1)
              } catch {
                case _: NumberFormatException => {
                  throw new InvalidEscapeException("Unicode", "\\" + controlChar + sb.toString(), str, location)
                }
              }
            }
            case _ => result.append(next).append(controlChar)
          }
        }
      } else {
        result.append(next)
      }
    }
    result.toString()
  }

  /**
    * Escapes a string value to make it a valid weave string
    *
    * @param strValue The string to escape
    * @return
    */
  def escapeString(strValue: String, quotes: Char = '"', insertQuotes: Boolean = true): String = {
    val result = new StringBuilder
    var i = 0
    val len = strValue.length
    if (insertQuotes) {
      result.append(quotes)
    }
    while (i < len) {
      (strValue.charAt(i): @switch) match {
        case c if c == quotes => {
          result.append('\\').append(quotes)
        }
        case '\\' =>
          if (len >= i + 6 && strValue.substring(i + 1, i + 6).matches("u[0-9a-fA-F]{4}")) {
            result.append("\\")
          } else {
            result.append("\\\\")
          }
        case '$'                             => result.append("\\$")
        case '\b'                            => result.append("\\b")
        case '\f'                            => result.append("\\f")
        case '\n'                            => result.append("\\n")
        case '\r'                            => result.append("\\r")
        case '\t'                            => result.append("\\t")
        case c if isNonPrintableCharacter(c) => result.append("\\u%04x".format(c.toInt))
        case c                               => result.append(c)
      }
      i += 1
    }
    if (insertQuotes) {
      result.append(quotes)
    }
    result.mkString
  }

  private def isNonPrintableCharacter(c: Char): Boolean =
    (c == '\u00ad'
      || c >= '\u0000' && c <= '\u001f'
      || c >= '\u007f' && c <= '\u009f'
      || c >= '\u0600' && c <= '\u0604'
      || c == '\u070f'
      || c == '\u17b4'
      || c == '\u17b5'
      || c >= '\u200c' && c <= '\u200f'
      || c >= '\u2060' && c <= '\u206f'
      || c == '\ufeff'
      || c >= '\ufff0' && c <= '\uffff')

  /**
    * Returns true if a key needs to be quoted or not
    *
    * @param key
    * @return
    */
  def keyRequiresQuotes(key: String): Boolean = {
    !NameIdentifierHelper.isValidIdentifier(key)
  }
}
