package org.mule.weave.v2.ts

import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.ast.functions.FunctionNode
import org.mule.weave.v2.parser.ast.types.DynamicReturnTypeNode
import org.mule.weave.v2.scope.ScopesNavigator

object FunctionTypeHelper {

  val ANONYMOUS_FUNCTION_NAME = "AnonymousFunction"

  def createDynamicParameter(i: Int): WeaveType = {
    TypeParameter(s"${WeaveTypeResolver.UNSPECIFIED_PARAMETER_TYPE_PREFIX}${i}")
  }

  def isDynamicFunction(ft: FunctionType): Boolean = {
    ft.params.exists((param) => {
      val wtype = param.wtype
      isDynamicTypeParameter(wtype)
    })
  }

  def hasDynamicParameter(dynamicReturnType: DynamicReturnType): Boolean = {
    dynamicReturnType.typeParameters.map(_.wtype).exists(FunctionTypeHelper.isDynamicTypeParameter)
  }

  def collectDynamicFunctions(ft: FunctionType): Seq[FunctionType] = WeaveTypeTraverse.shallowCollectAll(ft, {
    case t: FunctionType if FunctionTypeHelper.isDynamicFunction(t) => Seq(t)
    case _ => Seq()
  })

  def isDynamicTypeParameter(wtype: WeaveType): Boolean = {
    wtype match {
      case tp: TypeParameter => tp.name.startsWith(WeaveTypeResolver.UNSPECIFIED_PARAMETER_TYPE_PREFIX)
      case _                 => false
    }
  }

  def isDynamicReturnType(weaveType: WeaveType): Boolean = {
    weaveType.isInstanceOf[DynamicReturnType]
  }

  def hasDynamicReturn(weaveType: WeaveType): Boolean = {
    WeaveTypeTraverse.exists(weaveType, {
      case _: DynamicReturnType => true
      case _                    => false
    })
  }

  def hasNothingType(weaveType: WeaveType): Boolean = {
    WeaveTypeTraverse.exists(weaveType, {
      case _: NothingType => true
      case _              => false
    })
  }

  def getFunctionSubGraphFor(drt: DynamicReturnType, ctx: WeaveTypeResolutionContext): Option[TypeNode] = {
    val DynamicReturnType(args, functionNode, _, _, _, _, _) = drt
    ctx.getFunctionSubGraph(functionNode, args.map(_.wtype)).flatMap(s => s._1.findLocalNode(functionNode))
  }

  def resolveReturnType(argTypes: Seq[WeaveType], returnType: Option[WeaveType], ctx: WeaveTypeResolutionContext, dynamicReturnType: DynamicReturnType, strict: Boolean, collector: MessageCollector): Option[WeaveType] = {
    val functionNode: FunctionNode = dynamicReturnType.node
    val scopesNavigator: ScopesNavigator = dynamicReturnType.scope
    val resolver: ReferenceResolver = dynamicReturnType.resolver
    val typeGraph: TypeGraph = dynamicReturnType.typeGraph

    val maybeSubGraph: Option[(TypeGraph, MessageCollector)] = ctx.getFunctionSubGraph(functionNode, argTypes)
    if (maybeSubGraph.isDefined) {
      val subGraph = maybeSubGraph.get
      val maybeTypeNode = subGraph._1.findLocalNode(functionNode)
      collector.mergeWith(subGraph._2)
      maybeTypeNode.get.resultType()
    } else {
      val subGraphParsingContext =
        if (strict) {
          ctx.currentParsingContext
        } else {
          ctx.currentParsingContext.withMessageCollector(collector)
        }
      //If the expected output type was not specified select from the function node
      val declaredReturnType: Option[WeaveType] =
        returnType
          .orElse(functionNode.returnType.flatMap((rt) => {
            rt match {
              case _: DynamicReturnTypeNode => None
              case _                        => Some(WeaveType(rt, scopesNavigator.rootScope.referenceResolver()))
            }
          }))

      val dataGraph: TypeGraph = TypeGraph(subGraphParsingContext, typeGraph, scopesNavigator, functionNode, argTypes, declaredReturnType, resolver)
      val subGraphMessageCollector = MessageCollector()
      ctx.addFunctionSubGraph(functionNode, argTypes, dataGraph, subGraphMessageCollector)
      val weaveTypePropagator = ctx.newExecutorWithContext(scopesNavigator, dataGraph, subGraphParsingContext)
      val nodeBaseMessageCollector = ctx.currentNodeMessageCollector
      weaveTypePropagator.run()

      val value = dataGraph.findLocalNode(functionNode).get
      val maybeType = value.resultType()
      if (maybeType.isEmpty && !strict && nodeBaseMessageCollector.nodeErrors.nonEmpty) {
        ctx.removeFunctionSubGraph(functionNode, argTypes)
      } else {
        nodeBaseMessageCollector.nodeErrors.foreach(nodeErrorMessage => subGraphMessageCollector.error(nodeErrorMessage._2, nodeErrorMessage._1))
        nodeBaseMessageCollector.nodeWarnings.foreach(nodeWarningMessage => subGraphMessageCollector.warning(nodeWarningMessage._2, nodeWarningMessage._1))
      }
      maybeType
    }
  }

}
