package org.mule.weave.v2.parser.ast.annotation

import org.mule.weave.v2.api.tooling.ast.DWAstNodeKind
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.Child
import org.mule.weave.v2.parser.ast.Children
import org.mule.weave.v2.parser.ast.LiteralValueAstNode
import org.mule.weave.v2.parser.ast.MutableAstNode
import org.mule.weave.v2.parser.ast.structure.ArrayNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

case class AnnotationNode(name: NameIdentifier, var args: Option[AnnotationArgumentsNode]) extends AstNode {
  override def children(): Seq[AstNode] = Children(name).+=(args).result()

  override def cloneAst(): AnnotationNode = super.cloneAst().asInstanceOf[AnnotationNode]

  override protected def doClone(): AstNode = copy(name.cloneAst(), args.map(_.cloneAst().asInstanceOf[AnnotationArgumentsNode]))

  override def getKind(): String = DWAstNodeKind.ANNOTATION_NODE
}

object AnnotationNodeHelper {

  /**
    * Collects all simple arguments
    * @param annotation from where to get the arguments
    * @return A map with the arguments
    */
  def argsAsMap(annotation: AnnotationNode): Map[String, String] = {
    annotation.args.toSeq
      .flatMap(_.args)
      .map((arg) => {
        (arg.name.name, arg.value)
      })
      .collect({ case (name: String, lvn: LiteralValueAstNode) => (name, lvn.literalValue) })
      .toMap
  }

  /**
    * Returns the argument value with the specified name and converts it into a Seq[String]
    * @param argName the name of the argument
    * @param annotation The annotation where to search
    * @return The Seq[String] if present
    */
  def argStringSeq(argName: String, annotation: AnnotationNode): Option[Seq[String]] = {
    annotation.args.flatMap((args) => {
      val matchingArg = args.args.find(_.name.name == argName)
      val targetValue = matchingArg
        .map(_.value)
        .collect({
          case an: ArrayNode => an.elements.collect({ case lvn: LiteralValueAstNode => lvn.literalValue })
        })
      targetValue
    })
  }

  /**
    * Returns the argument value with the specified name and converts it into a Seq[String]
    * @param argName the name of the argument
    * @param annotation The annotation where to search
    * @return The Seq[String] if present
    */
  def argString(argName: String, annotation: AnnotationNode): Option[String] = {
    annotation.args.flatMap((args) => {
      val matchingArg = args.args.find(_.name.name == argName)
      val targetValue = matchingArg
        .map(_.value)
        .collect({
          case an: LiteralValueAstNode => an.literalValue
        })
      targetValue
    })
  }

  def arg(argName: String, annotation: AnnotationNode): Option[AstNode] = {
    annotation.args.flatMap((args) => {
      val matchingArg = args.args.find(_.name.name == argName)
      val targetValue = matchingArg
        .map(_.value)

      targetValue
    })
  }
}

case class AnnotationArgumentsNode(args: Seq[AnnotationArgumentNode]) extends AstNode {
  override def children(): Seq[AstNode] = args
  override protected def doClone(): AstNode = copy(args.map(_.cloneAst().asInstanceOf[AnnotationArgumentNode]))

  override def getKind(): String = DWAstNodeKind.ANNOTATION_ARGUMENTS_NODE
}

case class AnnotationArgumentNode(var name: NameIdentifier, var value: AstNode) extends AstNode with MutableAstNode {
  override def children(): Seq[AstNode] = Child(name, value)

  override def doClone(): AnnotationArgumentNode = copy(name.cloneAst(), value.cloneAst())

  override def getKind(): String = DWAstNodeKind.ANNOTATION_ARGUMENT_NODE

  override def update(toBeReplaced: AstNode, withNode: AstNode): Unit = {
    if (value eq toBeReplaced) value = withNode
  }
}
