package org.mule.weave.v2.sdk

import org.mule.weave.v2.sdk.selectors.AllAttributesSelectorCustomTypeResolver
import org.mule.weave.v2.sdk.selectors.AttributeSelectorCustomTypeResolver
import org.mule.weave.v2.sdk.selectors.DescendantsSelectorTypeResolver
import org.mule.weave.v2.sdk.selectors.MultiAttributeSelectorCustomTypeResolver
import org.mule.weave.v2.sdk.selectors.MultiValueSelectorCustomTypeResolver
import org.mule.weave.v2.sdk.selectors.ObjectIndexSelectorCustomTypeResolver
import org.mule.weave.v2.sdk.selectors.ObjectKeyValueSelectorTypeResolver
import org.mule.weave.v2.sdk.selectors.ValueSelectorCustomTypeResolver
import org.mule.weave.v2.sdk.selectors.ZonedInstantValueSelectorTypeResolver
import org.mule.weave.v2.ts.FunctionType
import org.mule.weave.v2.ts.TypeHelper._
import org.mule.weave.v2.ts._

import scala.annotation.tailrec
import scala.util.Try

object SystemFunctionDefinitions {
  //System operators
  val addition: FunctionType = additionFunctionDefinition()
  val multiply: FunctionType = multiplyFunctionDefinition()
  val divide: FunctionType = divideFunctionDefinition()
  val minusOps: FunctionType = minusFunctionDefinition()

  //>>
  val rightShift: FunctionType = {
    val T = TypeParameter("T")
    val E = TypeParameter("E")
    overloadedBinaryFunction(
      Seq(
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", DateTimeType()), FunctionTypeParameter("rhs", TimeZoneType())), DateTimeType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", E), FunctionTypeParameter("rhs", ArrayType(T))), ArrayType(UnionType(Seq(E, T))))),
      ">>")
  }

  //<<
  val leftShift: FunctionType = {
    val T = TypeParameter("T")
    val E = TypeParameter("E")
    overloadedBinaryFunction(Seq(FunctionType(Seq(T, E), Seq(FunctionTypeParameter("lhs", ArrayType(T)), FunctionTypeParameter("rhs", E)), ArrayType(UnionType(Seq(T, E))))), "<<")
  }

  //<= < > >=
  val comparator = FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", AnyType()), FunctionTypeParameter("rhs", AnyType())), BooleanType(), name = Some("Comparator >= <= > <"))

  //AND OR
  val booleanOps = FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", BooleanType()), FunctionTypeParameter("rhs", BooleanType())), BooleanType(), name = Some("Logical Operator AND/OR"))

  //Type OPS
  val asDefinition: FunctionType = asFunctionDefinition()

  val metadataInjector: FunctionType = {
    val T = TypeParameter("T")
    FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", T), FunctionTypeParameter("rhs", ObjectType())), T, name = Some("<~"))
  }

  val metadataAddition: FunctionType = {
    val T = TypeParameter("T")
    FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", T), FunctionTypeParameter("rhs", AnyType())), T, name = Some("<~"))
  }

  val schemaSelector = FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", AnyType()), FunctionTypeParameter("rhs", NameType())), AnyType(), name = Some("Schema Selector"))

  val valueSelectorOps: FunctionType = overloadedBinaryFunction(
    Seq(
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", ObjectType(Seq())), FunctionTypeParameter("rhs", NameType())),
        AnyType(),
        customReturnTypeResolver = Some(ValueSelectorCustomTypeResolver)),
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", ObjectType(Seq())), FunctionTypeParameter("rhs", StringType())),
        AnyType(),
        customReturnTypeResolver = Some(ValueSelectorCustomTypeResolver)),
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", ArrayType(AnyType())), FunctionTypeParameter("rhs", NameType())),
        AnyType(),
        customReturnTypeResolver = Some(ValueSelectorCustomTypeResolver)),
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", ArrayType(AnyType())), FunctionTypeParameter("rhs", StringType())),
        AnyType(),
        customReturnTypeResolver = Some(ValueSelectorCustomTypeResolver)),
      FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", LocalDateType()), FunctionTypeParameter("rhs", NameType())), NumberType()),
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", DateTimeType()), FunctionTypeParameter("rhs", NameType())),
        UnionType(Seq(NumberType(), TimeZoneType())),
        customReturnTypeResolver = Some(ZonedInstantValueSelectorTypeResolver)),
      FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", LocalDateTimeType()), FunctionTypeParameter("rhs", NameType())), NumberType()),
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", TimeType()), FunctionTypeParameter("rhs", NameType())),
        UnionType(Seq(NumberType(), TimeZoneType())),
        customReturnTypeResolver = Some(ZonedInstantValueSelectorTypeResolver)),
      FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", LocalTimeType()), FunctionTypeParameter("rhs", NameType())), NumberType()),
      FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", PeriodType()), FunctionTypeParameter("rhs", NameType())), NumberType()),
      FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", NullType()), FunctionTypeParameter("rhs", NameType())), NullType())),
    "Value Selector")

  val objectKeyValueSelectorOps: FunctionType = overloadedBinaryFunction(
    Seq(
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", ObjectType(Seq())), FunctionTypeParameter("rhs", NameType())),
        ObjectType(),
        customReturnTypeResolver = Some(ObjectKeyValueSelectorTypeResolver)),
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", ObjectType(Seq())), FunctionTypeParameter("rhs", StringType())),
        ObjectType(),
        customReturnTypeResolver = Some(ObjectKeyValueSelectorTypeResolver)),
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", ObjectType(Seq())), FunctionTypeParameter("rhs", NumberType())),
        ObjectType(),
        customReturnTypeResolver = Some(ObjectKeyValuePairIndexSelector)),
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", ArrayType(AnyType())), FunctionTypeParameter("rhs", NameType())),
        AnyType(),
        customReturnTypeResolver = Some(ObjectKeyValueSelectorTypeResolver)),
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", ArrayType(AnyType())), FunctionTypeParameter("rhs", StringType())),
        AnyType(),
        customReturnTypeResolver = Some(ObjectKeyValueSelectorTypeResolver))),
    "Value Selector")

  val descendantsSelectorOps: FunctionType = overloadedUnaryFunction(
    Seq(
      FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", ObjectType(Seq()))), ArrayType(AnyType()), customReturnTypeResolver = Some(DescendantsSelectorTypeResolver)),
      FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", ArrayType(AnyType()))), ArrayType(AnyType()), customReturnTypeResolver = Some(DescendantsSelectorTypeResolver)),
      FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", NullType()), FunctionTypeParameter("rhs", NameType())), NullType())),
    "Descendant Selector")

  val multiValueSelectorOps: FunctionType = overloadedBinaryFunction(
    Seq(
      //MultiValueSelectorOpTypeResolver
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", ObjectType(Seq())), FunctionTypeParameter("rhs", NameType())),
        AnyType(),
        customReturnTypeResolver = Some(MultiValueSelectorCustomTypeResolver)),
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", ObjectType(Seq())), FunctionTypeParameter("rhs", StringType())),
        AnyType(),
        customReturnTypeResolver = Some(MultiValueSelectorCustomTypeResolver)),
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", ArrayType(AnyType())), FunctionTypeParameter("rhs", NameType())),
        AnyType(),
        customReturnTypeResolver = Some(MultiValueSelectorCustomTypeResolver)),
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", ArrayType(AnyType())), FunctionTypeParameter("rhs", StringType())),
        AnyType(),
        customReturnTypeResolver = Some(MultiValueSelectorCustomTypeResolver)),
      FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", NullType()), FunctionTypeParameter("rhs", NameType())), NullType())),
    "Multi Value Selector")

  val attributeSelectorOps: FunctionType = overloadedBinaryFunction(
    Seq(
      //AttributeSelectorOpTypeResolver
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", ArrayType(AnyType())), FunctionTypeParameter("rhs", NameType())),
        AnyType(),
        customReturnTypeResolver = Some(AttributeSelectorCustomTypeResolver)),
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", ArrayType(AnyType())), FunctionTypeParameter("rhs", StringType())),
        AnyType(),
        customReturnTypeResolver = Some(AttributeSelectorCustomTypeResolver)),
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", AnyType()), FunctionTypeParameter("rhs", NameType())),
        AnyType(),
        customReturnTypeResolver = Some(AttributeSelectorCustomTypeResolver)),
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", AnyType()), FunctionTypeParameter("rhs", StringType())),
        AnyType(),
        customReturnTypeResolver = Some(AttributeSelectorCustomTypeResolver)),
      FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", NullType()), FunctionTypeParameter("rhs", NameType())), NullType())),
    "Attribute Selector")

  val multiAttributeSelectorOps: FunctionType = overloadedBinaryFunction(
    Seq(
      //MultiAttributeSelectorOpTypeResolver
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", ArrayType(AnyType())), FunctionTypeParameter("rhs", NameType())),
        AnyType(),
        customReturnTypeResolver = Some(MultiAttributeSelectorCustomTypeResolver)),
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", ArrayType(AnyType())), FunctionTypeParameter("rhs", StringType())),
        AnyType(),
        customReturnTypeResolver = Some(MultiAttributeSelectorCustomTypeResolver)),
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", AnyType()), FunctionTypeParameter("rhs", NameType())),
        AnyType(),
        customReturnTypeResolver = Some(MultiAttributeSelectorCustomTypeResolver)),
      FunctionType(
        Seq(),
        Seq(FunctionTypeParameter("lhs", AnyType()), FunctionTypeParameter("rhs", StringType())),
        AnyType(),
        customReturnTypeResolver = Some(MultiAttributeSelectorCustomTypeResolver)),
      FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", NullType()), FunctionTypeParameter("rhs", NameType())), NullType())),
    "Multi Attribute Selector")

  //Range selectors
  val rangeSelectorOps: FunctionType = rangeSelectorFunctionDefinition()

  val indexSelectorOps: FunctionType = indexSelectorFunctionDefinition()

  val filterSelector = filterSelectorFunctionDefinition()

  //-
  val minusUnary: FunctionType = FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", NumberType())), NumberType(), name = Some("-"))

  val allSchemaSelector: FunctionType = FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", AnyType())), ObjectType(), name = Some("All Schema Selector"))

  val allAttributes: FunctionType = FunctionType(
    Seq(),
    Seq(FunctionTypeParameter("lhs", AnyType())),
    ObjectType(),
    name = Some("All Attribute Selector"),
    customReturnTypeResolver = Some(AllAttributesSelectorCustomTypeResolver))

  val namespaceSelector: FunctionType = FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", AnyType())), NamespaceType(), name = Some("Namespace Selector"))
  //not
  val notUnary = FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", BooleanType())), BooleanType(), name = Some("not"), customReturnTypeResolver = Some(NotTypeResolver))

  val dynamicSelectorOps = overloadedBinaryFunction(
    valueSelectorOps.overloads ++
      rangeSelectorOps.overloads ++
      indexSelectorOps.overloads,
    "selector")

  private def asFunctionDefinition() = {
    val T = TypeParameter("T")
    FunctionType(Seq(T), Seq(FunctionTypeParameter("lhs", AnyType()), FunctionTypeParameter("rhs", TypeType(T))), T, name = Some("as"))
  }

  private def divideFunctionDefinition(): FunctionType = {
    FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", NumberType()), FunctionTypeParameter("rhs", NumberType())), NumberType(), name = Some("/"))
  }

  private def minusFunctionDefinition() = {
    val T = TypeParameter("T")
    overloadedBinaryFunction(
      Seq(
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", NumberType()), FunctionTypeParameter("rhs", NumberType())), NumberType()),
        FunctionType(
          Seq(),
          Seq(FunctionTypeParameter("lhs", ObjectType()), FunctionTypeParameter("keyToRemove", KeyType(AnyType(), Seq()))),
          ObjectType(),
          customReturnTypeResolver = Some(ObjectSubtractionTypeResolver)),
        FunctionType(
          Seq(),
          Seq(FunctionTypeParameter("lhs", ObjectType()), FunctionTypeParameter("rhs", StringType())),
          ObjectType(),
          customReturnTypeResolver = Some(ObjectSubtractionTypeResolver)),
        FunctionType(Seq(T), Seq(FunctionTypeParameter("lhs", ArrayType(T)), FunctionTypeParameter("rhs", AnyType())), ArrayType(T)),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", LocalDateType()), FunctionTypeParameter("rhs", LocalDateType())), PeriodType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", LocalDateType()), FunctionTypeParameter("rhs", PeriodType())), LocalDateType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", PeriodType()), FunctionTypeParameter("rhs", LocalDateType())), LocalDateType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", LocalDateTimeType()), FunctionTypeParameter("rhs", PeriodType())), LocalDateTimeType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", PeriodType()), FunctionTypeParameter("rhs", LocalDateTimeType())), LocalDateTimeType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", DateTimeType()), FunctionTypeParameter("rhs", PeriodType())), DateTimeType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", PeriodType()), FunctionTypeParameter("rhs", DateTimeType())), DateTimeType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", TimeType()), FunctionTypeParameter("rhs", PeriodType())), TimeType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", PeriodType()), FunctionTypeParameter("rhs", TimeType())), TimeType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", LocalTimeType()), FunctionTypeParameter("rhs", PeriodType())), LocalTimeType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", PeriodType()), FunctionTypeParameter("rhs", LocalTimeType())), TimeType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", LocalDateTimeType()), FunctionTypeParameter("rhs", LocalDateTimeType())), PeriodType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", DateTimeType()), FunctionTypeParameter("rhs", DateTimeType())), PeriodType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", TimeType()), FunctionTypeParameter("rhs", TimeType())), PeriodType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", LocalTimeType()), FunctionTypeParameter("rhs", LocalTimeType())), PeriodType())),
      "-")
  }

  private def multiplyFunctionDefinition(): FunctionType = {
    FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", NumberType()), FunctionTypeParameter("rhs", NumberType())), NumberType(), name = Some("*"))
  }

  private def rangeSelectorFunctionDefinition(): FunctionType = {
    val T = TypeParameter("T")
    overloadedBinaryFunction(
      Seq(
        FunctionType(Seq(T), Seq(FunctionTypeParameter("lhs", ArrayType(T)), FunctionTypeParameter("rhs", RangeType())), ArrayType(T)),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", NullType()), FunctionTypeParameter("rhs", RangeType())), NullType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", StringType()), FunctionTypeParameter("rhs", RangeType())), StringType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", BinaryType()), FunctionTypeParameter("rhs", RangeType())), BinaryType())),
      "Range Selector")
  }

  private def indexSelectorFunctionDefinition(): FunctionType = {
    val T = TypeParameter("T")
    overloadedBinaryFunction(
      Seq(
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", StringType()), FunctionTypeParameter("rhs", NumberType())), StringType()),
        FunctionType(
          Seq(),
          Seq(FunctionTypeParameter("lhs", ObjectType()), FunctionTypeParameter("rhs", NumberType())),
          AnyType(),
          customReturnTypeResolver = Some(ObjectIndexSelectorCustomTypeResolver)),
        FunctionType(Seq(T), Seq(FunctionTypeParameter("lhs", ArrayType(T)), FunctionTypeParameter("rhs", NumberType())), T),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", RangeType()), FunctionTypeParameter("rhs", NumberType())), NumberType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", BinaryType()), FunctionTypeParameter("rhs", NumberType())), BinaryType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", NullType()), FunctionTypeParameter("rhs", NumberType())), NullType())),
      "Index Selector")
  }

  private def filterSelectorFunctionDefinition() = {
    val T = TypeParameter("T")
    val arrayFilter = FunctionType(
      Seq(T),
      Seq(
        FunctionTypeParameter("lhs", ArrayType(T)),
        FunctionTypeParameter("rhs", FunctionType(Seq(T), Seq(FunctionTypeParameter("lhs", T), FunctionTypeParameter("rhs", NumberType())), UnionType(Seq(BooleanType(), StringType()))))),
      ArrayType(T),
      name = Some("Array Filter Selector"))

    val K = TypeParameter("K")
    val V = TypeParameter("V")
    val filterObject = FunctionType(
      Seq(K, V),
      Seq(
        FunctionTypeParameter("lhs", ObjectType(Seq(KeyValuePairType(K, V, optional = true)))),
        FunctionTypeParameter("rhs", FunctionType(Seq(K, V), Seq(FunctionTypeParameter("key", V), FunctionTypeParameter("value", K)), BooleanType()))),
      ObjectType(),
      name = Some("Object Filter Selector"))

    overloadedBinaryFunction(Seq(arrayFilter, filterObject), "Filter Selector")
  }

  private def additionFunctionDefinition() = {
    val T = TypeParameter("T")
    val E = TypeParameter("E")
    overloadedBinaryFunction(
      Seq(
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", NumberType()), FunctionTypeParameter("rhs", NumberType())), NumberType(), name = Some("+")),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", LocalDateType()), FunctionTypeParameter("rhs", PeriodType())), LocalDateType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", PeriodType()), FunctionTypeParameter("rhs", LocalDateType())), LocalDateType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", LocalDateTimeType()), FunctionTypeParameter("rhs", PeriodType())), LocalDateTimeType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", PeriodType()), FunctionTypeParameter("rhs", LocalDateTimeType())), LocalDateTimeType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", DateTimeType()), FunctionTypeParameter("rhs", PeriodType())), DateTimeType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", PeriodType()), FunctionTypeParameter("rhs", DateTimeType())), DateTimeType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", TimeType()), FunctionTypeParameter("rhs", PeriodType())), TimeType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", PeriodType()), FunctionTypeParameter("rhs", TimeType())), TimeType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", LocalTimeType()), FunctionTypeParameter("rhs", PeriodType())), LocalTimeType()),
        FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", PeriodType()), FunctionTypeParameter("rhs", LocalTimeType())), TimeType()),
        FunctionType(Seq(T, E), Seq(FunctionTypeParameter("lhs", ArrayType(T)), FunctionTypeParameter("rhs", E)), ArrayType(UnionType(Seq(T, E))))),
      "+")
  }

  def overloadedBinaryFunction(overloads: Seq[FunctionType], name: String): FunctionType = {
    FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", AnyType()), FunctionTypeParameter("rhs", AnyType())), AnyType(), overloads, name = Some(name))
  }

  def overloadedUnaryFunction(overloads: Seq[FunctionType], name: String): FunctionType = {
    FunctionType(Seq(), Seq(FunctionTypeParameter("lhs", AnyType())), AnyType(), overloads, name = Some(name))
  }
}

object ObjectSubtractionTypeResolver extends CustomTypeResolver {

  override def resolve(invocationTypes: Seq[WeaveType], ctx: WeaveTypeResolutionContext, node: TypeNode, resolvedReturnType: WeaveType): Option[WeaveType] = {
    val keyToRemove = invocationTypes.last
    val objectType = invocationTypes.head
    objectType match {
      case IntersectionType(of) => resolve(TypeHelper.resolveIntersection(of), keyToRemove, ctx)
      case _                    => resolve(objectType, keyToRemove, ctx)
    }
  }

  def resolve(objectType: WeaveType, keyToRemove: WeaveType, ctx: WeaveTypeResolutionContext): Option[WeaveType] = {
    val maybeType = TypeCoercer.coerce(NameType(), keyToRemove, ctx)
    maybeType match {
      case Some(coercedType) => {
        coercedType match {
          case ut: UnionType => {
            Some(ut.of.foldLeft(objectType)((nameType, objectToRemove) => {
              val option = resolve(nameType, objectToRemove, ctx)
              option match {
                case Some(value) => value
                case None        => objectToRemove
              }
            }))
          }
          case nameType: NameType => {
            doResolve(objectType, nameType, ctx)
          }
          case _ => None
        }
      }
      case None => None
    }
  }

  @tailrec
  private def doResolve(weaveType: WeaveType, nameType: NameType, ctx: WeaveTypeResolutionContext): Option[WeaveType] = {
    weaveType match {
      case TypeParameter(_, topType, bottomType, _, _) => {
        doResolve(topType.orElse(bottomType).getOrElse(AnyType()), nameType, ctx)
      }
      case IntersectionType(of) => {
        TypeHelper.resolveIntersection(of) match {
          case _: IntersectionType => Some(ObjectType())
          case otherType           => resolve(otherType, nameType, ctx)
        }
      }
      case rt: ReferenceType => doResolve(rt.resolveType(), nameType, ctx)
      case ObjectType(properties, closed, _) => {
        val resultProps: Seq[KeyValuePairType] = properties.filter((property) => {
          property.key match {
            case KeyType(name, _) => {
              name match {
                case NameType(value) =>
                  value match {
                    case Some(x) =>
                      if (nameType.value.isDefined) {
                        !x.selectedBy(nameType.value.get)
                      } else {
                        true
                      }
                    case None => true
                  }
                case _ => true
              }
            }
            case _ => true
          }
        })
        Some(ObjectType(resultProps, closed))
      }
      case _ => Some(ObjectType())
    }
  }
}

object ObjectKeyValuePairIndexSelector extends CustomTypeResolver {

  override def resolve(invocationTypes: Seq[WeaveType], ctx: WeaveTypeResolutionContext, node: TypeNode, resolvedReturnType: WeaveType): Option[WeaveType] = {
    val objectType = invocationTypes.head
    val index = invocationTypes(1)
    val mayBeIndexNumber = index match {
      case NumberType(Some(number)) => Try(number.toInt).toOption
      case _                        => None
    }
    if (mayBeIndexNumber.isDefined) {
      resolveSelection(objectType, mayBeIndexNumber.get)
    } else {
      resolveOnePropertySelection(objectType)
    }
  }

  private def resolveOnePropertySelection(objectType: WeaveType): Option[WeaveType] = {
    objectType match {
      case ot: ObjectType if ot.properties.size == 1 => {
        Some(ObjectType(Seq(ot.properties.head.copy(repeated = false)), close = true))
      }
      case rt: ReferenceType => {
        resolveOnePropertySelection(rt.resolveType())
      }
      case it: IntersectionType => {
        val resolvedIntersection = TypeHelper.resolveIntersection(it.of)
        resolvedIntersection match {
          case IntersectionType(_) => Some(ObjectType())
          case otherType           => resolveOnePropertySelection(otherType)
        }
      }
      case ut: UnionType => {
        val types = ut.of.flatMap((wtype) => {
          resolveOnePropertySelection(wtype)
        })
        if (types.isEmpty) {
          Some(ObjectType())
        } else {
          Some(TypeHelper.unify(types))
        }
      }
      case _ => Some(ObjectType())
    }
  }

  private def resolveSelection(objectType: WeaveType, index: Int): Option[WeaveType] = {
    objectType match {
      case it: IntersectionType => {
        val resolvedInter = TypeHelper.resolveIntersection(it.of)
        resolvedInter match {
          case IntersectionType(_) => Some(ObjectType())
          case otherType =>
            resolveSelection(otherType, index)
        }
      }
      case ut: UnionType => {
        val types = ut.of.flatMap((wtype) => {
          resolveSelection(wtype, index)
        })
        if (types.isEmpty) {
          Some(ObjectType())
        } else {
          Some(TypeHelper.unify(types))
        }
      }
      case rt: ReferenceType => resolveSelection(rt.resolveType(), index)
      case ot: ObjectType => {
        val indexToSelect = if (index < 0) ot.properties.size + index else index
        if (ot.properties.size > indexToSelect && indexToSelect >= 0) {
          Some(ObjectType(Seq(ot.properties(indexToSelect).copy(repeated = false)), close = true))
        } else {
          Some(ObjectType())
        }
      }
      case _ => {
        Some(ObjectType())
      }
    }
  }
}

object NotTypeResolver extends CustomTypeResolver {
  override def resolve(invocationTypes: Seq[WeaveType], ctx: WeaveTypeResolutionContext, node: TypeNode, resolvedReturnType: WeaveType): Option[WeaveType] = {
    invocationTypes match {
      case Seq(BooleanType(maybeValue, constraints)) => Some(BooleanType(maybeValue.map(!_), constraints.negate()))
      case _                                         => None
    }
  }
}
