package org.mule.weave.v2.sdk

import org.mule.weave.v2.annotations.WeaveApi
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

/**
  * Resolves a Module from its NameIdentifier
  */
@WeaveApi(Seq("data-weave-agent"))
trait WeaveResourceResolver {

  /**
    * Returns the resource if it could be resolved
    *
    * @param name The name of the module to be resolved
    * @return The Module if resolved
    */
  def resolve(name: NameIdentifier): Option[WeaveResource]

  /**
    * Returns the binary resource if it could be resolved
    *
    * @param name The name of the module to be resolved
    * @return The Module if resolved
    */
  def resolveBinary(name: NameIdentifier): Option[BinaryWeaveResource] =
    resolveBinaryPath(NameIdentifierHelper.toWeaveBinaryFilePath(name))

  /**
    * Resolves the file path and returns the resource
    *
    * @param path The path to be resolved
    * @return
    */
  def resolvePath(path: String): Option[WeaveResource] = {
    resolve(NameIdentifierHelper.fromWeaveFilePath(path))
  }

  /**
    * Resolves the file path and returns the binary resource
    *
    * @param path The path to be resolved
    * @return
    */
  def resolveBinaryPath(path: String): Option[BinaryWeaveResource] = None

  /**
    * Returns all the resources
    *
    * @param name The name of the module to be resolved
    * @return The Module if resolved
    */
  def resolveAll(name: NameIdentifier): Seq[WeaveResource] = resolve(name).toSeq

  /**
    * Returns if the resource can be resolved by this resolver
    *
    * @param name The name of the resource to be resolved
    * @return if it can be resolved.
    */
  def canResolveResource(name: NameIdentifier): Boolean = resolve(name).isDefined

}

object EmptyWeaveResourceResolver extends WeaveResourceResolver {
  override def resolve(name: NameIdentifier): Option[WeaveResource] = None

  override def resolveAll(name: NameIdentifier): Seq[WeaveResource] = Seq()
}

class WLangOnlyWeaveResourceResolver(resourceResolver: WeaveResourceResolver) extends WeaveResourceResolver {

  override def resolveAll(name: NameIdentifier): Seq[WeaveResource] = {
    if (isWlangModule(name)) {
      resourceResolver.resolveAll(name);
    } else {
      Seq.empty
    }
  }

  private def isWlangModule(name: NameIdentifier) = {
    name.nameElements().length > 1 && name.nameElements().head.equals("dw")
  }

  override def resolve(name: NameIdentifier): Option[WeaveResource] = {
    if (isWlangModule(name)) {
      resourceResolver.resolve(name);
    } else {
      None
    }
  }

  override def resolveBinary(name: NameIdentifier): Option[BinaryWeaveResource] = {
    if (isWlangModule(name)) {
      resourceResolver.resolveBinary(name);
    } else {
      None
    }
  }

  override def canResolveResource(name: NameIdentifier): Boolean = {
    isWlangModule(name) && resourceResolver.canResolveResource(name)
  }

}

class TwoLevelWeaveResourceResolver(parent: WeaveResourceResolver, child: () => WeaveResourceResolver) extends WeaveResourceResolver {

  override def resolve(name: NameIdentifier): Option[WeaveResource] = {
    parent.resolve(name).orElse(child().resolve(name))
  }

  override def resolveBinary(name: NameIdentifier): Option[BinaryWeaveResource] = {
    parent.resolveBinary(name).orElse(child().resolveBinary(name))
  }

  override def resolveAll(name: NameIdentifier): Seq[WeaveResource] = {
    parent.resolveAll(name) ++ child().resolveAll(name)
  }

  override def canResolveResource(name: NameIdentifier): Boolean = parent.canResolveResource(name) || child().canResolveResource(name)
}
