package org.mule.weave.v2.parser.phase

import org.mule.weave.v2.annotations.WeaveApi

import java.io.PrintWriter
import java.io.StringWriter
import org.mule.weave.v2.parser.Message
import org.mule.weave.v2.parser.MessageKind
import org.mule.weave.v2.parser.TypePhaseCategory
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.exception.{ LocatableException, ParseTimeOutException }
import org.mule.weave.v2.parser.location.UnknownLocation
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.scope.ScopesNavigator
import org.mule.weave.v2.ts.TypeGraph
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.ts.WeaveTypeResolutionContext
import org.mule.weave.v2.utils.DataGraphDotEmitter

class TypeCheckingPhase[T <: AstNode]() extends CompilationPhase[ScopeGraphResult[T], TypeCheckingResult[T]] {

  override def doCall(input: ScopeGraphResult[T], ctx: ParsingContext): PhaseResult[TypeCheckingResult[T]] = {
    try {
      val implicitInputMap: Map[String, Option[WeaveType]] = ctx.implicitInput.map((ii) => ii.name -> ii.wtype).toMap
      val dataGraph: TypeGraph = TypeGraph(ctx, input.scope, input.astNode, implicitInputMap, ctx.expectedOutputType)
      val context: WeaveTypeResolutionContext = new WeaveTypeResolutionContext(dataGraph)
      if (System.getProperty("type_check_debug") != null) {
        println(DataGraphDotEmitter.print(dataGraph, name = "TypeCheckPre"))
      }
      context.newExecutorWithContext(input.scope, dataGraph, ctx).run()
      if (System.getProperty("type_check_debug") != null) {
        println(DataGraphDotEmitter.print(dataGraph, name = "TypeCheckPost"))
      }
      SuccessResult(new TypeCheckingResult(input.input, input.astNode, input.scope, dataGraph), ctx)
    } catch {
      case ptoe: ParseTimeOutException =>
        ctx.messageCollector.error(Message(MessageKind.UNEXPECTED_EXCEPTION_MESSAGE_KIND, s"Type checking took: ${ptoe.timeTaken}ms, which exceeds the max time: ${ptoe.maxTime}ms", TypePhaseCategory), UnknownLocation)
        FailureResult(ctx)
      case lm: LocatableException => {
        ctx.messageCollector.error(Message(MessageKind.UNEXPECTED_EXCEPTION_MESSAGE_KIND, lm.getMessage, TypePhaseCategory), lm.location.asInstanceOf[WeaveLocation])
        FailureResult(ctx)
      }
      case lm: Throwable =>
        lm.printStackTrace()
        val stackTrace = new StringWriter()
        lm.printStackTrace(new PrintWriter(stackTrace))
        ctx.messageCollector.error(Message(MessageKind.UNEXPECTED_EXCEPTION_MESSAGE_KIND, "Exception while running type check " + lm.getClass.getName + " :\n" + stackTrace.toString, TypePhaseCategory), UnknownLocation)
        FailureResult(ctx)
    }
  }

}

trait DataGraphResultAware {
  def typeGraph: TypeGraph
}

@WeaveApi(Seq("data-weave-agent"))
class TypeCheckingResult[T <: AstNode](_input: ParsingContentInput, _astNode: T, _scope: ScopesNavigator, val typeGraph: TypeGraph) extends ScopeGraphResult[T](_input, _astNode, _scope) with DataGraphResultAware