package org.mule.weave.v2.ts

import org.mule.weave.v2.parser.ast.annotation.{ AnnotationCapableNode, AnnotationNode }
import org.mule.weave.v2.parser.ast.conditional.{ IfNode, UnlessNode }
import org.mule.weave.v2.parser.ast.functions._
import org.mule.weave.v2.parser.ast.header.directives._
import org.mule.weave.v2.parser.ast.logical.{ AndNode, OrNode }
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.operators.OpNode
import org.mule.weave.v2.parser.ast.patterns._
import org.mule.weave.v2.parser.ast.structure._
import org.mule.weave.v2.parser.ast.types.{ DynamicReturnTypeNode, TypeReferenceNode, WeaveTypeNode }
import org.mule.weave.v2.parser.ast.updates.{ UpdateExpressionNode, UpdateNode, UpdateSelectorNode }
import org.mule.weave.v2.parser.ast.variables.{ NameIdentifier, VariableReferenceNode }
import org.mule.weave.v2.parser.ast.{ AstNode, AstNodeHelper, CommentNode }
import org.mule.weave.v2.parser.exception.ParseException
import org.mule.weave.v2.parser.location.UnknownLocation
import org.mule.weave.v2.parser.phase.listener.ParsingNotificationManager
import org.mule.weave.v2.parser.phase.{ ParsingContext, PhaseResult, TypeCheckingResult }
import org.mule.weave.v2.scope.{ Reference, ScopesNavigator }
import org.mule.weave.v2.ts.WeaveTypeCloneHelper.copyLocation
import org.mule.weave.v2.ts.resolvers.{ LiteralTypeResolver, PassThroughTypeResolver, PassThroughWithDocs, UnknownTypeResolver }

import scala.collection.{ Map, mutable }

/**
  * Builds the type graph
  */
class TypeGraphBuilder(val parsingContext: ParsingContext, val scope: ScopesNavigator, parentGraph: Option[TypeGraph] = None, implicitInputs: Map[String, Option[WeaveType]] = Map(), expectedOutput: Option[WeaveType] = None) {

  private val _nodes = mutable.ArrayBuffer[TypeNode]()
  private val _referenceEdges = mutable.ArrayBuffer[ReferenceEdge]()
  private val typeReferenceResolver: ScopeGraphTypeReferenceResolver = scope.rootScope.referenceResolver()
  private val notificationManager: ParsingNotificationManager = parsingContext.notificationManager
  private val syntaxVersion: VersionDirective = scope.astNavigator()
    .versionDirective()
    .getOrElse(new VersionDirective())

  def build(node: AstNode): TypeGraph = {
    node match {
      case mn: ModuleNode => {
        processDirectives(mn.elements, DefaultResolver)
      }
      case _ => traverse(node, DefaultResolver)
    }
    createTypeGraph()
  }

  def build(functionNode: FunctionNode, arguments: Seq[WeaveType], resolver: ReferenceResolver): TypeGraph = {
    val paramList: Seq[FunctionParameter] = functionNode.params.paramList
    if (arguments.size != paramList.length) {
      if (paramList.head.defaultValue.isDefined) {
        val defaultValued: Int = paramList.length - arguments.size
        //Map first with default values
        paramList
          .slice(0, defaultValued)
          .foreach((param) => {
            if (param.defaultValue.isDefined) {
              val paramNode: TypeNode = traverse(param.variable, resolver)
              Edge(traverse(param.defaultValue.get, resolver), paramNode)
            } else {
              createNode(param.variable, resolver)
            }
          })
        paramList
          .slice(defaultValued, paramList.length)
          .zipWithIndex
          .foreach((param) => {
            createNode(param._1.variable, LiteralTypeResolver(arguments(param._2)))
          })
      } else {
        paramList.zipWithIndex.foreach((param) => {
          if (arguments.size <= param._2) {
            if (param._1.defaultValue.isDefined) {
              val paramNode: TypeNode = traverse(param._1.variable, resolver)
              Edge(traverse(param._1.defaultValue.get, resolver), paramNode)
            } else {
              createNode(param._1.variable, resolver)
            }
          } else {
            createNode(param._1.variable, LiteralTypeResolver(arguments(param._2)))
          }
        })
      }
    } else {
      paramList.zipWithIndex.foreach((param) => {
        createNode(param._1.variable, LiteralTypeResolver(arguments(param._2)))
      })
    }

    val bodyNode: TypeNode = traverse(functionNode.body, resolver)
    val result: TypeNode = createNode(functionNode, PassThroughTypeResolver).internalNode()
    Edge(bodyNode, result, expected = expectedOutput, supportsCoercion = false)
    createTypeGraph()
  }

  private def createTypeGraph(): TypeGraph = {
    _referenceEdges.foreach((referencePair) => {
      val reference: Reference = referencePair.source
      val referenceNode: TypeNode = referencePair.target
      val mayBeLabel = referencePair.label
      val mayBeReferencedType: Option[TypeNode] = resolveReferenceNode(reference)
      mayBeReferencedType match {
        case Some(referencedNode) if reference.isCrossModule => {
          val typeResolver = referencedNode.resultType().map(LiteralTypeResolver).getOrElse(UnknownTypeResolver)
          //We create a new type node as we can not link type graph it generate leaks when type graphs are cached
          val newSourceType = TypeNode(referencedNode.astNode, typeResolver)
          Edge(newSourceType, referenceNode, mayBeLabel)
        }
        case Some(referencedNode) if reference.isLocalReference => {
          Edge(referencedNode, referenceNode, mayBeLabel)
        }
        case None => {
          if (parsingContext.strictMode) {
            //This should never happen or it means that the scope phase didn't work correctly. That is why we should fail
            val msg = "Unable to resolve reference to " + reference.referencedNode.name + " from " + reference.moduleSource.getOrElse("local module")
            throw new ParseException(msg, reference.referencedNode.location())
          }
        }
      }
    })
    new TypeGraph(_nodes, parentGraph, syntaxVersion)
  }

  def resolveReferenceNode(referenceNode: Reference): Option[TypeNode] = {
    notificationManager.progress()
    //If no parent is a local reference else is a reference to another module.
    if (referenceNode.moduleSource.isEmpty) {
      findNode(referenceNode.referencedNode)
    } else {
      val parent: Option[NameIdentifier] = referenceNode.moduleSource
      val module: PhaseResult[TypeCheckingResult[ModuleNode]] = parsingContext.getTypeCheckingForModule(parent.get)
      module.getResult().typeGraph.findLocalNode(referenceNode.referencedNode)
    }
  }

  private def traverse(node: AstNode, resolver: ReferenceResolver): TypeNode = {
    traverseNode(node, createNode(node, resolver), resolver)
  }

  private def processAnnotationsNodes(annotationNodes: Seq[AnnotationNode], resolver: ReferenceResolver): Unit = {
    annotationNodes.foreach((an) => {
      an.args.foreach((args) => {
        args.args.foreach((arg) => {
          traverse(arg.value, resolver)
        })
      })
    })
  }

  // TODO is this method used?
  def transformSelector(origin: TypeNode, selector: UpdateSelectorNode): TypeNode = {
    val selectorExpr = traverse(selector.selector, DefaultResolver)
    val selectorNode: TypeNode = createNode(selector, DefaultResolver)
    Edge(selectorExpr, selectorNode, Some(EdgeLabels.PROPERTY_SELECTOR))
    Edge(origin, selectorNode, Some(EdgeLabels.VALUE))
    selector.child match {
      case Some(usn: UpdateSelectorNode) => transformSelector(selectorNode, usn)
      case _                             => selectorNode
    }
  }

  private def traverseNode(node: AstNode, target: TypeNode, resolver: ReferenceResolver): TypeNode = {
    notificationManager.progress()
    node match {
      case documentNode: DocumentNode => {
        val directives: Seq[DirectiveNode] = documentNode.header.directives

        val directivesResult = processDirectives(directives, resolver)
        val outputType = expectedOutput.orElse(directivesResult.output)

        val rootNode: TypeNode = traverse(documentNode.root, resolver)
        Edge(rootNode, target, Some(EdgeLabels.OUTPUT), outputType)
      }
      case doBlock: DoBlockNode => {
        val directives: Seq[DirectiveNode] = doBlock.header.directives
        processDirectives(directives, resolver)
        val rootNode: TypeNode = traverse(doBlock.body, resolver)
        Edge(rootNode, target)
      }
      case UsingNode(vars, expr, _) => {
        vars.assignmentSeq.foreach((param) => {
          val name = traverse(param.name, resolver)
          val value = traverse(param.value, resolver)
          Edge(value, name)
        })
        val source = traverse(expr, resolver)
        Edge(source, target)
      }
      case FunctionNode(args, _, returnType, _) => {
        args.paramList.foreach((arg) => {
          val defaultValue: Option[AstNode] = arg.defaultValue
          if (arg.wtype.isDefined) {
            val argType: WeaveType = WeaveType(arg.wtype.get, typeReferenceResolver)
            val argumentNode: TypeNode = createNode(arg.variable, LiteralTypeResolver(argType))
            Edge(source = argumentNode, target = target, label = EdgeLabels.PARAM_TYPE(arg.variable.name))
            if (defaultValue.isDefined) {
              val defaultValueNode = traverse(defaultValue.get, resolver)
              Edge(source = defaultValueNode, target = argumentNode, expected = Some(argType), supportsCoercion = false)
              Edge(source = defaultValueNode, target = target, label = EdgeLabels.DEFAULT_VALUE(arg.variable.name))
            }
          } else if (defaultValue.isDefined) {
            val defaultValueNode = traverse(defaultValue.get, resolver)
            val argumentTypeNode: TypeNode = createNode(arg.variable, resolver)
            Edge(source = defaultValueNode, target = argumentTypeNode, supportsCoercion = false)
            Edge(source = defaultValueNode, target = target, label = EdgeLabels.DEFAULT_VALUE(arg.variable.name))
          }
        })

        returnType.foreach({
          case _: DynamicReturnTypeNode => //Ignore it
          case rt => {
            val returnType = WeaveType(rt, typeReferenceResolver)
            val returnTypeNode: TypeNode = createNode(rt, LiteralTypeResolver(returnType))
            Edge(source = returnTypeNode, target = target, label = EdgeLabels.RETURN_TYPE)
          }
        })
      }
      case ofn: OverloadedFunctionNode => {
        for (function <- ofn.functionDirectives) {
          val variable = traverse(function.variable, resolver)
          Edge(traverse(function.literal, resolver), variable)
          Edge(variable, target)
        }
      }
      case vr: VariableReferenceNode => {
        resolver.resolveVariable(this, vr.variable, target)
      }
      case nn: NamespaceNode => {
        resolver.resolveVariable(this, nn.prefix, target)
      }
      case StringInterpolationNode(_) => {
        node
          .children()
          .foreach((child) => {
            Edge(traverse(child, resolver), target, expected = Some(StringType()))
          })
      }
      case KeyValuePairNode(key, value, cond) => {
        Edge(traverse(key, resolver), target, EdgeLabels.NAME)
        Edge(traverse(value, resolver), target, Some(EdgeLabels.VALUE))
        if (cond.isDefined) {
          Edge(traverse(cond.get, resolver), target, expected = Some(BooleanType()))
        }
      }
      case NameValuePairNode(key, value, cond) => {
        Edge(traverse(key, resolver), target, EdgeLabels.NAME)
        Edge(traverse(value, resolver), target, EdgeLabels.VALUE)
        if (cond.isDefined) {
          Edge(traverse(cond.get, resolver), target, expected = Some(BooleanType()))
        }
      }
      case KeyNode(keyName, ns, atr, _) => {
        Edge(traverse(keyName, resolver), target, EdgeLabels.LOCAL_NAME)
        if (ns.isDefined) {
          Edge(traverse(ns.get, resolver), target, EdgeLabels.NAMESPACE)
        }
        if (atr.isDefined) {
          val children: Seq[AstNode] = atr.get.children()
          children.foreach((child) => {
            Edge(traverse(child, resolver), target, EdgeLabels.ATTRIBUTE)
          })
        }
      }
      case DynamicKeyNode(keyName, atr) => {
        Edge(traverse(keyName, resolver), target, EdgeLabels.LOCAL_NAME)
        if (atr.isDefined) {
          val children: Seq[AstNode] = atr.get.children()
          children.foreach((child) => {
            Edge(traverse(child, resolver), target, EdgeLabels.ATTRIBUTE)
          })
        }
      }
      case NameNode(keyName, ns, _) => {
        Edge(traverse(keyName, resolver), target, EdgeLabels.LOCAL_NAME)
        if (ns.isDefined) {
          Edge(traverse(ns.get, resolver), target, EdgeLabels.NAMESPACE)
        }
      }
      case DynamicNameNode(keyName) => {
        Edge(traverse(keyName, resolver), target, EdgeLabels.LOCAL_NAME)
      }
      case UpdateNode(expression, matchers, _) => {
        val updateToExpr = traverse(expression, resolver)
        matchers.expressions.zipWithIndex.map(pair => {
          val updateNode: UpdateExpressionNode = pair._1
          val index: Int = pair._2

          val valueMatchVariableName = updateNode.name
          val indexMatchVariableName = updateNode.indexId

          if (updateNode.condition.isDefined) {
            val conditionalNode = updateNode.condition.get
            val conditionalExpression = FunctionNode(FunctionParameters(Seq(FunctionParameter(valueMatchVariableName), FunctionParameter(indexMatchVariableName))), conditionalNode)
            Edge(traverse(conditionalExpression, resolver), target, EdgeLabels.CONDITION(index.toString))
          }

          val classType: Class[NamespaceNode] = NamespaceNode.clazz
          val namespaceNodes = AstNodeHelper.collectChildrenWith(updateNode.selector, classType)
          namespaceNodes.foreach((nn) => {
            val namespaceLabel = EdgeLabels.NAMESPACE_PREFIX(nn.prefix.name)
            if (!target.containsEdge(namespaceLabel)) {
              resolver.resolveVariable(this, nn.prefix, target, Some(namespaceLabel))
            }
          })

          //The update expression
          val updateExpression = FunctionNode(FunctionParameters(Seq(FunctionParameter(valueMatchVariableName), FunctionParameter(indexMatchVariableName))), updateNode.updateExpression)
          Edge(traverse(updateExpression, resolver), target, EdgeLabels.FUNCTION(index.toString))
        })
        Edge(updateToExpr, target, EdgeLabels.UPDATE_TO)
      }
      case PatternMatcherNode(lhs, patterns, _) => {
        val matchNode: TypeNode = traverse(lhs, resolver)
        patterns.patterns
          .map({
            case patternExpressionNode @ RegexPatternNode(caseExpressionNode, name, matchExpressionNode) => {
              val patternExpression = createNode(patternExpressionNode, WeaveTypeResolver(patternExpressionNode, typeReferenceResolver, scope, resolver))
              Edge(matchNode, patternExpression, EdgeLabels.PATTERN_EXPRESSION, StringType())
              Edge(traverse(FunctionNode(FunctionParameters(Seq(FunctionParameter(name))), matchExpressionNode), resolver), patternExpression, EdgeLabels.MATCH_EXPRESSION)
              Edge(traverse(FunctionNode(FunctionParameters(Seq(FunctionParameter(name))), caseExpressionNode), resolver), patternExpression, EdgeLabels.CASE_EXPRESSION)
              patternExpression
            }
            case patternExpressionNode @ TypePatternNode(caseExpressionNode, name, matchExpression) => {
              buildPattern(patternExpressionNode, matchNode, caseExpressionNode, FunctionNode(FunctionParameters(Seq(FunctionParameter(name))), matchExpression), resolver = resolver)
            }
            case patternExpressionNode @ LiteralPatternNode(caseExpressionNode, name, matchExpression) => {
              buildPattern(
                patternExpressionNode,
                matchNode,
                caseExpressionNode,
                FunctionNode(FunctionParameters(Seq(FunctionParameter(name))), matchExpression),
                requiredPattern = false,
                resolver)
            }
            case ex @ ExpressionPatternNode(caseExpression, name, matchExpression) => {
              val expressionTypeNode = TypeNode(ex, WeaveTypeResolver(ex, typeReferenceResolver, scope, resolver))
              _nodes.+=(expressionTypeNode)

              //Condition
              val conditionFunction: FunctionNode = FunctionNode(FunctionParameters(Seq(FunctionParameter(name))), caseExpression)
              val conditionFunctionTypeNode: TypeNode = traverse(conditionFunction, resolver)
              //On Match
              val onMatch: FunctionNode = FunctionNode(FunctionParameters(Seq(FunctionParameter(name))), matchExpression)
              val onMatchFunctionTypeNode: TypeNode = traverse(onMatch, resolver)

              Edge(matchNode, expressionTypeNode, EdgeLabels.PATTERN_EXPRESSION)
              Edge(onMatchFunctionTypeNode, expressionTypeNode, EdgeLabels.MATCH_EXPRESSION)
              Edge(conditionFunctionTypeNode, expressionTypeNode, EdgeLabels.CASE_EXPRESSION)
              expressionTypeNode

            }
            case patternExpressionNode @ DefaultPatternNode(value, name) => {
              val patternNode = createNode(patternExpressionNode, WeaveTypeResolver(patternExpressionNode, typeReferenceResolver, scope, resolver))
              Edge(matchNode, patternNode, EdgeLabels.PATTERN_EXPRESSION)
              Edge(traverse(FunctionNode(FunctionParameters(Seq(FunctionParameter(name))), value), resolver), patternNode, EdgeLabels.MATCH_EXPRESSION)
              patternNode
            }
            case patternExpressionNode @ DeconstructArrayPatternNode(head, tail, matchExpression) => {
              val patternNode = createNode(patternExpressionNode, WeaveTypeResolver(patternExpressionNode, typeReferenceResolver, scope, resolver))
              Edge(matchNode, patternNode, EdgeLabels.PATTERN_EXPRESSION)
              val fnParameters = FunctionParameters(Seq(FunctionParameter(head), FunctionParameter(tail)))
              val functionNode = FunctionNode(fnParameters, matchExpression)
              Edge(traverse(functionNode, resolver), patternNode, EdgeLabels.MATCH_EXPRESSION)
              patternNode
            }

            case patternExpressionNode @ EmptyArrayPatternNode(matchExpression) => {
              val patternNode = createNode(patternExpressionNode, WeaveTypeResolver(patternExpressionNode, typeReferenceResolver, scope, resolver))
              Edge(matchNode, patternNode, EdgeLabels.PATTERN_EXPRESSION)
              val functionNode = FunctionNode(FunctionParameters(Seq()), matchExpression)
              Edge(traverse(functionNode, resolver), patternNode, EdgeLabels.MATCH_EXPRESSION)
              patternNode
            }

            case patternExpressionNode @ EmptyObjectPatternNode(matchExpression) => {
              val patternNode = createNode(patternExpressionNode, WeaveTypeResolver(patternExpressionNode, typeReferenceResolver, scope, resolver))
              Edge(matchNode, patternNode, EdgeLabels.PATTERN_EXPRESSION)
              val functionNode = FunctionNode(FunctionParameters(Seq()), matchExpression)
              Edge(traverse(functionNode, resolver), patternNode, EdgeLabels.MATCH_EXPRESSION)
              patternNode
            }

            case patternExpressionNode @ DeconstructObjectPatternNode(headKey, headValue, tail, matchExpression) => {
              val patternNode = createNode(patternExpressionNode, WeaveTypeResolver(patternExpressionNode, typeReferenceResolver, scope, resolver))
              Edge(matchNode, patternNode, EdgeLabels.PATTERN_EXPRESSION)
              val fnParameters = FunctionParameters(Seq(FunctionParameter(headKey), FunctionParameter(headValue), FunctionParameter(tail)))
              val functionNode = FunctionNode(fnParameters, matchExpression)
              Edge(traverse(functionNode, resolver), patternNode, EdgeLabels.MATCH_EXPRESSION)
              patternNode
            }

          })
          .foreach(Edge(_, target))
      }
      case AndNode(lhs, rhs, _) => {
        val lhsBoolean = traverse(lhs, resolver)
        val rhsBoolean = traverse(rhs, resolver)
        Edge(lhsBoolean, target, None, Some(BooleanType()))
        Edge(rhsBoolean, target, None, Some(BooleanType()))
      }
      case OrNode(lhs, rhs, _) => {
        val lhsBoolean = traverse(lhs, resolver)
        val rhsBoolean = traverse(rhs, resolver)
        Edge(lhsBoolean, target, None, Some(BooleanType()))
        Edge(rhsBoolean, target, None, Some(BooleanType()))
      }
      case IfNode(ifExpr, condition, elseExpr, _) => {
        val conditionNode = traverse(condition, resolver)
        Edge(traverse(ifExpr, new WrappedResolver(resolver, conditionNode, true)), target, EdgeLabels.IF_LABEL)
        Edge(conditionNode, target, EdgeLabels.CONDITION, BooleanType())
        Edge(traverse(elseExpr, new WrappedResolver(resolver, conditionNode, false)), target, EdgeLabels.ELSE_LABEL)
      }
      case UnlessNode(ifExpr, condition, elseExpr, _) => {
        val conditionNode = traverse(condition, resolver)
        Edge(traverse(ifExpr, new WrappedResolver(resolver, conditionNode, true)), target, EdgeLabels.IF_LABEL)
        Edge(conditionNode, target, EdgeLabels.CONDITION, BooleanType())
        Edge(traverse(elseExpr, new WrappedResolver(resolver, conditionNode, false)), target, EdgeLabels.ELSE_LABEL)
      }
      case FunctionCallNode(function, args, typeParameters, _) => {
        if (typeParameters.isDefined) {
          typeParameters.get
            .children()
            .foreach(tp => {
              Edge(traverse(tp, resolver), target, EdgeLabels.TYPE_PARAMETER)
            })
        }
        Edge(traverse(function, resolver), target, EdgeLabels.FUNCTION)
        val children: Seq[AstNode] = args.children()
        children.foreach((child) => {
          Edge(traverse(child, resolver), target, EdgeLabels.ARGUMENT)
        })
      }
      case _: ObjectNode => {
        val children: Seq[AstNode] = node.children()
        children.foreach {
          case kvn: KeyValuePairNode          => Edge(traverse(kvn, resolver), target)
          case annotationNode: AnnotationNode =>
          case child => {
            val arrayType = ArrayType(ObjectType())
            copyLocation(arrayType, child.location())
            val objectType = ObjectType()
            copyLocation(objectType, child.location())
            val unionType = UnionType(Seq(arrayType, objectType, NullType()))
            copyLocation(unionType, child.location())
            Edge(traverse(child, resolver), target, expected = Some(unionType))
          }
        }
      }

      case AttributesNode(_) => {
        val children: Seq[AstNode] = node.children()
        children.foreach {
          case kvn: NameValuePairNode => Edge(traverse(kvn, resolver), target)
          case child                  => Edge(traverse(child, resolver), target, expected = Some(ObjectType()))
        }
      }
      case TypeReferenceNode(_, _, _, _, _) =>
      case HeadTailArrayNode(head, tail, _) =>
        Edge(traverse(head, resolver), target, EdgeLabels.HEAD)
        Edge(traverse(tail, resolver), target, EdgeLabels.TAIL)

      case HeadTailObjectNode(headKey, headValue, tail, _) =>
        Edge(traverse(headKey, resolver), target, EdgeLabels.HEAD_KEY)
        Edge(traverse(headValue, resolver), target, EdgeLabels.HEAD_VALUE)
        Edge(traverse(tail, resolver), target, EdgeLabels.TAIL)
      case _: OpNode => {
        traverseChildren(node, target, resolver, Some(EdgeLabels.ARGUMENT))
      }
      case typeNode: WeaveTypeNode =>
        Edge(createNode(typeNode, resolver), target)
        processChildAnnotationNodes(typeNode, resolver)
      case ConditionalNode(value, condition) =>
        Edge(traverse(value, resolver), target, EdgeLabels.VALUE)
        Edge(traverse(condition, resolver), target, EdgeLabels.CONDITION, BooleanType())
      case _ => traverseChildren(node, target, resolver)
    }
    node match {
      case annotationCapableNode: AnnotationCapableNode => processAnnotationsNodes(annotationCapableNode.codeAnnotations, resolver)
      case _ =>
    }
    target
  }

  private case class DirectivesResult(output: Option[WeaveType]) {
    def mergeWith(dr: DirectivesResult): DirectivesResult = {
      DirectivesResult(output = output.orElse(dr.output))
    }
  }

  private def processDirectives(nodes: Seq[DirectiveNode], resolver: ReferenceResolver): DirectivesResult = {
    nodes
      .map(processDirective(_, resolver))
      .reduceOption((dr1, dr2) => {
        dr1.mergeWith(dr2)
      })
      .getOrElse(DirectivesResult(None))
  }

  private def processDirective(node: DirectiveNode, resolver: ReferenceResolver): DirectivesResult = {
    var outputType: Option[WeaveType] = None
    node match {

      case FunctionDirectiveNode(variable, literal, annotationNodes) => {
        processAnnotationsNodes(annotationNodes, resolver)
        processFunctionDirective(variable, literal, resolver)
      }
      case vd @ VarDirective(variable, literal, wtype, annotationNodes) => {
        processAnnotationsNodes(annotationNodes, resolver)
        processVarDirective(variable, literal, wtype, vd.weaveDoc, resolver)
      }
      case td @ TypeDirective(variable, _, literal, annotationNodes) => {
        processAnnotationsNodes(annotationNodes, resolver)
        processTypeNodeDirective(variable, literal, td.weaveDoc, resolver)
      }
      case NamespaceDirective(prefix, uri, annotationNodes) => {
        processAnnotationsNodes(annotationNodes, resolver)
        processNamespaceDirective(prefix, uri, resolver)
      }
      case InputDirective(name, _, _, _, wtype, annotationNodes) => {
        processAnnotationsNodes(annotationNodes, resolver)
        if (wtype.isDefined) {
          val inputType = WeaveType(wtype.get, typeReferenceResolver)
          //We don't validate against the implicit input since implicit inputs are inferred and they may not be very accurate.
          //For example literal types are not inferred so if the type in the input directive has a literal type it fails.
          //For now we just avoid the check in the future we may put some logic to see if we can detect errors
          createNode(name, LiteralTypeResolver(inputType))
        } else {
          val maybeImplicitInput = implicitInputs.get(name.name)
          maybeImplicitInput match {
            case Some(Some(implicitInput)) => {
              createNode(name, LiteralTypeResolver(implicitInput))
            }
            case _ => {
              createNode(name, UnknownTypeResolver)
            }
          }
        }
      }
      case OutputDirective(_, _, _, wtype, annotationNodes) => {
        processAnnotationsNodes(annotationNodes, resolver)
        outputType = if (outputType.isEmpty) wtype.map(WeaveType(_, typeReferenceResolver)) else outputType
      }
      case AnnotationDirectiveNode(nameIdentifier, params, codeAnnotations) => {
        processAnnotationsNodes(codeAnnotations, resolver)
      }
      case ImportDirective(importedModule, subElements, codeAnnotations) => {
        processAnnotationsNodes(codeAnnotations, resolver)
      }
      case VersionDirective(major, minor, codeAnnotations) => {
        processAnnotationsNodes(codeAnnotations, resolver)
      }
      case err: ErrorDirectiveNode =>
        processAnnotationsNodes(err.codeAnnotations, resolver)
    }

    DirectivesResult(output = outputType)
  }

  def processNamespaceDirective(prefix: NameIdentifier, uri: UriNode, resolver: ReferenceResolver): Edge = {
    val namespace: TypeNode = createNode(prefix, LiteralTypeResolver(NamespaceType(Some(prefix.name), Some(UriType(Some(uri.literalValue))))))
    val uriType = UriType()
    copyLocation(uriType, uri.location())
    Edge(traverse(uri, resolver), namespace, expected = Some(uriType))
    Edge(namespace, traverse(prefix, resolver))
  }

  def processFunctionDirective(variable: NameIdentifier, literal: AstNode, resolver: ReferenceResolver): Edge = {
    Edge(traverse(literal, resolver), traverse(variable, resolver))
  }

  def processVarDirective(variable: NameIdentifier, literal: AstNode, wtype: Option[WeaveTypeNode], wdoc: Option[CommentNode], resolver: ReferenceResolver): Edge = {
    val variableType: Option[WeaveType] = wtype.map((typeNode) => {
      val weaveType = WeaveType(typeNode, typeReferenceResolver)
      weaveType.withDocumentation(wdoc.map(_.literalValue), wdoc.map(_.location()).getOrElse(UnknownLocation))
    })
    val variableNode: TypeNode =
      if (variableType.isDefined) {
        createNode(variable, LiteralTypeResolver(variableType.get))
      } else {
        createNode(variable, new PassThroughWithDocs(wdoc.map(_.literalValue)))
      }
    Edge(traverse(literal, resolver), variableNode, expected = variableType, supportsCoercion = false)
  }

  def processTypeNodeDirective(variable: NameIdentifier, literal: WeaveTypeNode, wdoc: Option[CommentNode], resolver: ReferenceResolver): Edge = {
    processChildAnnotationNodes(literal, resolver)
    Edge(createNode(literal, resolver), createNode(variable, new PassThroughWithDocs(wdoc.map(_.literalValue))))
  }

  def processChildAnnotationNodes(literal: WeaveTypeNode, resolver: ReferenceResolver): Unit = {
    AstNodeHelper.traverse(literal, {
      case astNode: AnnotationCapableNode => {
        processAnnotationsNodes(astNode.codeAnnotations, resolver)
        true
      }
      case _ => true
    })
  }

  def createNode(node: AstNode, referenceResolver: ReferenceResolver): TypeNode = {
    val resolver: WeaveTypeResolver = WeaveTypeResolver(node, typeReferenceResolver, scope, referenceResolver)
    val result: TypeNode = createNode(node, resolver)
    result
  }

  def createNode(node: AstNode, resolver: WeaveTypeResolver): TypeNode = {
    val result = TypeNode(node, resolver)
    _nodes.+=(result)
    result
  }

  def addReferenceEdge(referenceNode: Reference, result: TypeNode, label: Option[EdgeLabel]): Unit = {
    _referenceEdges.+=(ReferenceEdge(referenceNode, result, label))
  }

  def buildPattern(patternExpressionNode: PatternExpressionNode, matchNode: TypeNode, caseNode: AstNode, onMatch: FunctionNode, requiredPattern: Boolean = true, resolver: ReferenceResolver): TypeNode = {
    val patternExpression = TypeNode(patternExpressionNode, WeaveTypeResolver(patternExpressionNode, typeReferenceResolver, scope, resolver))
    _nodes.+=(patternExpression)
    val matchExpression = traverse(onMatch, resolver)
    val caseExpression = traverse(caseNode, resolver)
    if (requiredPattern) {
      Edge(matchNode, patternExpression, EdgeLabels.PATTERN_EXPRESSION)
    }
    Edge(matchExpression, patternExpression, EdgeLabels.MATCH_EXPRESSION)
    Edge(caseExpression, patternExpression, EdgeLabels.CASE_EXPRESSION)
    patternExpression
  }

  def findNode(referenceNode: AstNode): Option[TypeNode] = {
    val localNode: Option[TypeNode] = findLocalNode(referenceNode)
    localNode
      .orElse({
        parentGraph.flatMap((dataGraph) => dataGraph.findLocalNode(referenceNode))
      })
  }

  private def findLocalNode(referenceNode: AstNode): Option[TypeNode] = {
    _nodes.find((node) => {
      val result: Boolean = node.astNode eq referenceNode
      result
    })
  }

  def traverseChildren(node: AstNode, result: TypeNode, resolver: ReferenceResolver, label: Option[EdgeLabel] = None): Unit = {
    val children: Seq[AstNode] = node.children()
    children
      .filterNot(astNode => astNode.isInstanceOf[AnnotationNode])
      .foreach((child) => {
        Edge(traverse(child, resolver), result, label)
      })
  }

}
