package org.mule.weave.v2.api.tooling.impl.ts.catalog

import org.mule.weave.v2.api.tooling.ast.DWAstNode
import org.mule.weave.v2.api.tooling.ast.DWAstNodeFactory
import org.mule.weave.v2.api.tooling.ast.DWAstNodeFactoryResult
import org.mule.weave.v2.api.tooling.internal.DefaultDWAstNode
import org.mule.weave.v2.api.tooling.location.ResourceIdentifier
import org.mule.weave.v2.api.tooling.message.Message
import org.mule.weave.v2.parser.DefaultMessage
import org.mule.weave.v2.parser.ModuleParser
import org.mule.weave.v2.parser.ParsingPhaseCategory
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.phase.ModuleParsingPhasesManager
import org.mule.weave.v2.scope.AstNavigator
import org.mule.weave.v2.sdk.ParsingContextFactory
import org.mule.weave.v2.sdk.WeaveResourceFactory

import java.util.Optional

object DWAstNodeFactoryUnknownErrorMessage extends DefaultMessage(
  "Unknown Error",
  "TypeChecking failed to return result. A possible reason is that the identifier cannot be resolved.",
  ParsingPhaseCategory)

class ResourceDWAstNodeFactory(phasesManager: ModuleParsingPhasesManager) extends DWAstNodeFactory {

  @Override
  def parse(content: String): Optional[DWAstNode] = {
    val weaveResource = WeaveResourceFactory.fromContent(content)

    val parsingContext = ParsingContextFactory.createParsingContext(false)

    val parseResult = ModuleParser.parse(ModuleParser.parsingPhase(), weaveResource, parsingContext)
    if (parseResult.hasErrors || !parseResult.hasResult) {
      return Optional.empty.asInstanceOf[Optional[DWAstNode]]
    }

    val result = parseResult.getResult()
    val node = result.astNode
    Optional.of(DefaultDWAstNode(node, new AstNavigator(node), Option.empty, Option.empty))
  }

  @Override
  def parse(identifier: ResourceIdentifier): DWAstNodeFactoryResult = {
    val moduleContext = ParsingContextFactory.createParsingContext(phasesManager, strictMode = true)
    val typeCheckingResult = phasesManager.typeCheckModule(NameIdentifier.fromFQN(identifier.getFQNIdentifier), moduleContext)

    typeCheckingResult.map(phaseResult => {
      val errorMessage: Array[Message] = if (phaseResult.hasErrors()) {
        phaseResult.messages().errorMessages.map(_._2).toArray
      } else Array.empty

      val node: Optional[DWAstNode] = if (phaseResult.hasResult()) {
        val result = phaseResult.getResult()
        Optional.of(DefaultDWAstNode(result.astNode, result.astNavigator, Some(result.typeGraph), Some(result.scope)))
      } else Optional.empty()

      new DWAstNodeFactoryResult(node, errorMessage)
    }).getOrElse({
      new DWAstNodeFactoryResult(Optional.empty(), Array(DWAstNodeFactoryUnknownErrorMessage))
    })
  }
}
