package org.mule.weave.v2.api.tooling.internal

import org.mule.weave.v2.api.tooling.ast.DWAstNode
import org.mule.weave.v2.api.tooling.location.Location
import org.mule.weave.v2.api.tooling.ts.DWType
import org.mule.weave.v2.codegen.CodeGenerator
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.LiteralValueAstNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.scope.AstNavigator
import org.mule.weave.v2.scope.ScopesNavigator
import org.mule.weave.v2.ts.TypeGraph
import org.mule.weave.v2.utils.Optionals.toJavaOptional

import java.util.Optional
import scala.language.postfixOps

case class DefaultDWAstNode(
  node: AstNode,
  astNavigator: AstNavigator,
  typeGraph: Option[TypeGraph],
  scopeGraph: Option[ScopesNavigator])
    extends DWAstNode {

  override def getReference: Optional[DWAstNode] = {
    val maybeWrapper: Option[DWAstNode] = scopeGraph
      .flatMap(_.resolveReference(node))
      .map(astNode => DefaultDWAstNode(astNode, astNavigator, typeGraph, scopeGraph))
    maybeWrapper asJava
  }

  override def getTypeOf: Optional[DWType] = {
    val maybeWrapper: Option[DWType] = typeGraph
      .flatMap(tgraph => tgraph.findNode(node))
      .flatMap(_.resultType())
    maybeWrapper asJava
  }

  override def getChildren: Array[DWAstNode] = {
    node.children().map(astNode => DefaultDWAstNode(astNode, astNavigator, typeGraph, scopeGraph)).toArray
  }

  override def getKind: String = {
    node.getKind()
  }

  override def getValue: Optional[String] = {
    node match {
      case ni: NameIdentifier      => Optional.of(ni.fullQualifiedName())
      case lv: LiteralValueAstNode => Optional.of(lv.literalValue)
      case _                       => Optional.empty()
    }
  }

  override def emitCode(): String = {
    CodeGenerator.generate(node)
  }

  override def getLocation: Location = {
    node.location()
  }

  override def getDocumentation: Optional[DWAstNode] = {
    Optional.ofNullable(node.weaveDoc.map(weaveDocNode => DefaultDWAstNode(weaveDocNode, astNavigator, typeGraph, scopeGraph)).orNull)
  }

  override def getParent: Optional[DWAstNode] = {
    val maybeWrapper: Option[DWAstNode] = astNavigator
      .parentOf(node)
      .map(astNode => DefaultDWAstNode(astNode, astNavigator, typeGraph, scopeGraph))
    maybeWrapper asJava
  }
}
