package org.mule.weave.v2.utils

import org.mule.weave.v2.codegen.StringCodeWriter
import org.mule.weave.v2.ts.ArrayMetadataValue
import org.mule.weave.v2.ts.BooleanType
import org.mule.weave.v2.ts.DateTimeType
import org.mule.weave.v2.ts.KeyValuePairMetadataValue
import org.mule.weave.v2.ts.LiteralMetadataValue
import org.mule.weave.v2.ts.LocalDateTimeType
import org.mule.weave.v2.ts.LocalDateType
import org.mule.weave.v2.ts.LocalTimeType
import org.mule.weave.v2.ts.Metadata
import org.mule.weave.v2.ts.MetadataValue
import org.mule.weave.v2.ts.NullType
import org.mule.weave.v2.ts.NumberType
import org.mule.weave.v2.ts.ObjectMetadataValue
import org.mule.weave.v2.ts.PeriodType
import org.mule.weave.v2.ts.RegexType
import org.mule.weave.v2.ts.StringType
import org.mule.weave.v2.ts.TimeType
import org.mule.weave.v2.ts.TimeZoneType
import org.mule.weave.v2.ts.UnknownMetadataValue
import org.mule.weave.v2.ts.UriType
import org.mule.weave.v2.ts.VariableReferenceMetadataValue

class WeaveTypeMetadataEmitter(config: WeaveTypeMetadataEmitterConfig = WeaveTypeMetadataEmitterConfig()) {

  def toString(metadata: Metadata): String = {
    val builder = new StringCodeWriter()
    builder.printQuoted(metadata.name).printSpace(":")
    printMetadataValue(builder, metadata.value)
    builder.toString
  }

  def printMetadataValue(value: MetadataValue): String = {
    val builder = new StringCodeWriter()
    printMetadataValue(builder, value)
    builder.toString
  }

  private def printMetadataValue(builder: StringCodeWriter, value: MetadataValue): Unit = {
    value match {
      case LiteralMetadataValue(value, weaveType, _) =>
        weaveType match {
          case _: BooleanType =>
            builder.print(value)
          case _: DateTimeType =>
            builder.print(s"|$value|")
          case _: LocalDateType =>
            builder.print(s"|$value|")
          case _: LocalDateTimeType =>
            builder.print(s"|$value|")
          case _: LocalTimeType =>
            builder.print(s"|$value|")
          case _: PeriodType =>
            builder.print(s"|$value|")
          case _: TimeType =>
            builder.print(s"|$value|")
          case _: TimeZoneType =>
            builder.print(s"|$value|")
          case _: NullType =>
            builder.print(value)
          case _: NumberType =>
            builder.print(value)
          case _: RegexType =>
            builder.print(s"/$value/")
          case _: StringType =>
            builder.print(StringEscapeHelper.escapeString(value))
          case _: UriType =>
            builder.print(s"$value")
          case _ =>
            builder.print(StringEscapeHelper.escapeString(value))
        }

      case ArrayMetadataValue(elements, _) =>
        builder.print("[")
        elements.zipWithIndex.foreach(element => {
          if (element._2 > 0) {
            builder.printSpace(",")
          }
          printMetadataValue(builder, element._1)
        })
        builder.print("]")

      case KeyValuePairMetadataValue(key, value, _) =>
        builder.printQuoted(key).printSpace(":")
        printMetadataValue(builder, value)

      case VariableReferenceMetadataValue(fullQualifiedName, _) =>
        builder.print(StringEscapeHelper.escapeString(fullQualifiedName))

      case ObjectMetadataValue(properties, _) =>
        builder.print("{")
        builder.indent()
        // In case is one property key value pair we don't print the new lines
        val inlineObject = properties.isEmpty || properties.size == 1
        if (inlineObject) {
          builder.printSpace()
          properties.foreach(property => {
            printMetadataValue(builder, property)
          })
          builder.printSpace()
        } else {
          properties.zipWithIndex.foreach(property => {
            if (property._2 > 0) {
              builder.printSpace(",")
            }
            if (config.prettyPrint) {
              builder.println()
              builder.printIndent()
            }
            printMetadataValue(builder, property._1)
          })
        }
        builder.dedent()
        if (config.prettyPrint && !inlineObject) {
          builder.println()
          builder.printIndent()
        }
        builder.print("}")

      case _: UnknownMetadataValue =>
        builder.printQuoted("unknown")
    }
  }
}

object WeaveTypeMetadataEmitter {
  def apply(): WeaveTypeMetadataEmitter = new WeaveTypeMetadataEmitter()
  def apply(config: WeaveTypeMetadataEmitterConfig): WeaveTypeMetadataEmitter = new WeaveTypeMetadataEmitter(config)
}

case class WeaveTypeMetadataEmitterConfig(prettyPrint: Boolean = true)
