package org.mule.weave.v2.grammar.structure

import org.mule.weave.v2.grammar.Grammar
import org.mule.weave.v2.grammar.Tokens
import org.mule.weave.v2.grammar.location.PositionTracking
import org.mule.weave.v2.parser.annotation.TrailingCommaAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.structure.ArrayNode
import org.mule.weave.v2.parser.ast.structure.ConditionalNode
import org.mule.weave.v2.parser.ast.structure.HeadTailArrayNode
import org.mule.weave.v2.parser.location.WeaveLocation
import org.parboiled2._

import scala.collection.immutable.Seq

trait Array extends PositionTracking with Tokens {
  this: Array with Grammar =>

  def expr: Rule1[AstNode]

  def array: Rule1[AstNode] = namedRule("Array") {
    pushPosition ~ (squareBracketOpen ~!~ (emptyArray | arrayExpression | missingToken("missing `]` for closing the array expression. i.e [1] or [1 ~ something()]", "]"))) ~ injectPosition
  }

  def emptyArray: Rule1[ArrayNode] = namedRule("Empty Array") {
    squareBracketEnd ~> Array.createEmptyArray
  }

  def arrayExpression = namedRule("Array") {
    (conditionalArrayElement | expr) ~!~ (tailArray | arrayElements | fail("missing Tail Expression i.e [1 ~ something()] or a `]` for an array expression i.e [1,2] "))
  }

  def arrayElements = namedRule("Array Elements") {
    optional(commaSep ~ oneOrMore(conditionalArrayElement | expr).separatedBy(commaSep)) ~!~ (commaSep ~ push(true) | push(false)) ~!~ (squareBracketEnd | fail("']' or ',' for the array expression.")) ~> Array.createArrayNode
  }

  def tailArray = namedRule("Head ~ Tail") {
    tilde ~!~ (expr | missingExpression("missing tail expression. i.e [1 ~ something()]")) ~!~ (squareBracketEnd | fail("missing `]` for the Head Tail array expression i.e [1 ~ something()].")) ~> Array.createHeadTailNode
  }

  private def conditionalArrayElement: Rule1[AstNode] = namedRule("Conditional Element<'('Expr')' 'if' '(' Expr ')'>") {
    pushPosition ~ (parenStart ~ expr ~ parenEnd ~ fws ~ ifKeyword ~!~ ((ws ~ ifCondition) | (fws ~ expr)) ~> Array.createConditionalArrayElement) ~ injectPosition
  }
}

object Array {
  val createHeadTailNode: (AstNode, AstNode) => HeadTailArrayNode = (head: AstNode, tail: AstNode) => {
    HeadTailArrayNode(head, tail)
  }

  val createEmptyArray: () => ArrayNode = () => {
    ArrayNode(Seq())
  }

  val createArrayNode: (AstNode, Option[Seq[AstNode]], Boolean) => ArrayNode = (head: AstNode, elements: Option[Seq[AstNode]], hasTrailingComma: Boolean) => {
    val node = ArrayNode(Seq(head).++(elements.getOrElse(Seq())))
    if (hasTrailingComma) {
      TrailingCommaAnnotation.annotate(node)
    }
    node
  }

  val createConditionalArrayElement: (AstNode, AstNode) => ConditionalNode = (v: AstNode, condition: AstNode) => {
    ConditionalNode(v, condition)
  }
}
