package org.mule.weave.v2.ts.resolvers

import org.mule.weave.v2.ts._

class EqTypeResolver(refLhs: Option[SourceReference], refRhs: Option[SourceReference]) extends WeaveTypeResolver {

  override def resolveReturnType(node: TypeNode, ctx: WeaveTypeResolutionContext): Option[WeaveType] = {
    val lhs = node.incomingEdges().head
    val rhs = node.incomingEdges().last

    val leftConstraints = getConstraints(refLhs, rhs.mayBeIncomingType(), node, ctx)
    val rightConstraints = getConstraints(refRhs, lhs.mayBeIncomingType(), node, ctx)

    val value = (rhs.mayBeIncomingType(), lhs.mayBeIncomingType()) match {
      case (Some(rType), Some(lType)) if TypeHelper.isSingletonType(rType) && TypeHelper.isSingletonType(lType) =>
        Some(TypeHelper.areEqualStructurally(rType, lType))
      case _ => None
    }
    Some(BooleanType(value, leftConstraints.combine(rightConstraints)))
  }

  override def resolveExpectedType(node: TypeNode, incomingExpectedType: Option[WeaveType], ctx: WeaveTypeResolutionContext): Seq[(Edge, WeaveType)] = {
    node.incomingEdges().map((edge) => (edge, BooleanType()))
  }

  protected def getConstraints(maybeSource: Option[SourceReference], maybeType: Option[WeaveType], node: TypeNode, ctx: WeaveTypeResolutionContext): VariableConstraints = {
    maybeSource match {
      case Some(source) =>
        maybeType match {
          case Some(typ) =>

            val reference = source.reference
            val posConstraint = source.createPositiveConstraint(typ, node, ctx)
            val negative: VariableConstraints.ConstraintsMap = {
              if (TypeHelper.isSingletonType(typ)) {
                Some(Map(reference -> Seq(source.createNegativeConstraint(typ, node, ctx))))
              } else {
                None
              }
            }
            VariableConstraints(Some(Map(reference -> Seq(posConstraint))), negative)
          case None => VariableConstraints.emptyConstraints()
        }
      case None => VariableConstraints.emptyConstraints()
    }
  }
}
