package org.mule.weave.v2.ts

import org.mule.weave.v2.parser.InvalidDynamicReturnMessage
import org.mule.weave.v2.parser.Message
import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.NestedTypeMessage
import org.mule.weave.v2.parser.NotEnoughArgumentMessage
import org.mule.weave.v2.parser.TooManyArgumentMessage
import org.mule.weave.v2.parser.TypeCoercedMessage
import org.mule.weave.v2.parser.TypeMessage
import org.mule.weave.v2.parser.TypeMismatch
import org.mule.weave.v2.parser.ast.functions.FunctionNode
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.ts.FunctionTypeHelper.isDynamicTypeParameter
import org.mule.weave.v2.ts.WeaveTypeCloneHelper.copyLocation
import org.mule.weave.v2.utils.IdentityHashMap
import org.mule.weave.v2.utils.SeqUtils

import scala.collection.mutable.ArrayBuffer

case class Constraint(expectedType: WeaveType, actualType: WeaveType, constraintType: Int) {

  override def toString(): String = {
    val assignment: String = constrainKind()
    expectedType.toString(prettyPrint = false, namesOnly = true) + assignment + actualType.toString(prettyPrint = false, namesOnly = true)
  }

  def constrainKind(): String = {
    val assignment: String = constraintType match {
      case Constraint.ASSIGNMENT => " = "
      case Constraint.TOP        => " < "
      case Constraint.BOTTOM     => " > "
    }
    assignment
  }
}

sealed trait ConstraintSet {
  /**
    * Merge two ConstraintSet it will compose all the problems
    *
    * @param constrainsResult The ConstraintSet to merge with
    * @return The new ConstraintSet with the combination of problems
    */
  def merge(constrainsResult: => ConstraintSet): ConstraintSet

  /**
    * Resolves the set of problems defined in the constraint set and return a Result.
    * The result can be a ErrorResult if there is no way to satisfy all the problems. Or a SolutionResult with the Substitution that satisfies all this problems
    *
    * @param ctx                        The context
    * @param coerce                     If we should take into account Coercion when resolving the problems
    * @param typeParametersToBeResolved The Type parameters that we are looking for a solution
    * @return The result
    */
  def resolve(ctx: WeaveTypeResolutionContext, coerce: Boolean, typeParametersToBeResolved: Seq[TypeParameter], theSubstitution: Substitution = emptySubstitution, warningMessages: Seq[Message] = Seq()): ConstraintResult

  def emptySubstitution: Substitution = {
    Substitution()
  }
}

case class NoSolutionSet(problems: Seq[(WeaveLocation, Message)]) extends ConstraintSet {
  override def merge(constrainsResult: => ConstraintSet): ConstraintSet = constrainsResult match {
    case NoSolutionSet(otherProblems) => NoSolutionSet(problems ++ otherProblems)
    case _                            => this
  }

  override def resolve(ctx: WeaveTypeResolutionContext, coerce: Boolean, typeParametersToBeResolved: Seq[TypeParameter], theSubstitution: Substitution = emptySubstitution, warningMessages: Seq[Message] = Seq()): ConstraintResult = {
    ErrorResult(problems)
  }

  override def toString: String = {
    s"NoSolutionSet - ${problems.headOption.map(_._2.message).getOrElse("")}"
  }
}

object EmptyConstrainProblem extends ConstrainProblem(Seq())

case class MultiOptionConstrainProblem(constraintsSets: Seq[ConstraintSet]) extends ConstraintSet {

  override def merge(constrainsResult: => ConstraintSet): ConstraintSet = {
    constrainsResult match {
      case ns: NoSolutionSet => ns
      case cp: ConstrainProblem => {
        MultiOptionConstrainProblem(constraintsSets.map(_.merge(cp)))
      }
      case mcp: MultiOptionConstrainProblem => {
        // Keep it lazy
        // Don't force resolution of constraintSets with ".toArray" here since during resolution only the first one is consumed.
        val constraintSets = SeqUtils.combine(Seq(constraintsSets, mcp.constraintsSets)).map((constraints) => constraints.reduce(_.merge(_)))
        MultiOptionConstrainProblem(constraintSets)
      }
    }
  }

  override def resolve(ctx: WeaveTypeResolutionContext, coerce: Boolean, typeParametersToBeResolved: Seq[TypeParameter], theSubstitution: Substitution, warningMessages: Seq[Message]): ConstraintResult = {
    val constraintResults: Seq[ConstraintResult] = constraintsSets.map((constraintSet) => {
      constraintSet.resolve(ctx, coerce, typeParametersToBeResolved, theSubstitution, warningMessages)
    })
    //We only want the first occurrence from left to right
    val results = constraintResults.collectFirst({ case sr: SolutionResult => sr })
    if (results.isEmpty) {
      ErrorResult(constraintResults.collect({ case em: ErrorResult => em.problems }).flatten.distinct)
    } else {
      results.get
    }
  }

  override def toString: String = {
    s"MultiOptionConstrainProblem(\n\t- ${constraintsSets.mkString("\n\t- ")}\n)"
  }
}

case class ConstrainProblem(constraints: Seq[Constraint]) extends ConstraintSet {
  override def merge(constrainsResult: => ConstraintSet): ConstraintSet = {
    constrainsResult match {
      case ns: NoSolutionSet => ns
      case ConstrainProblem(toMerge) => {
        ConstrainProblem(this.constraints ++ toMerge)
      }
      case mcp: MultiOptionConstrainProblem => {
        mcp.merge(this)
      }
    }
  }

  override def toString: String = {
    s"ConstrainProblem{\n\t${constraints.mkString(",\n\t")}\n})"
  }

  def substituteTypeParamWithDefaultValue(expectedType: WeaveType): WeaveType = {
    WeaveTypeTraverse.treeMap(
      expectedType, {
      case DynamicReturnType(arguments, functionNode, typeGraph, scopesNavigator, functionName, expectedReturnType, resolver) => {
        val substitutedParameters: Seq[FunctionTypeParameter] = arguments.map({
          case FunctionTypeParameter(name, wtype, optional, defaultValueType) => {
            val paramType = defaultValueType match {
              case Some(defaultType) if (TypeHelper.isJustTypeParameter(wtype)) => {
                TypeHelper.toTopType(defaultType)
              }
              case _ => wtype
            }
            FunctionTypeParameter(name, paramType, optional, defaultValueType)
          }
        })
        DynamicReturnType(substitutedParameters, functionNode, typeGraph, scopesNavigator, functionName, expectedReturnType, resolver)
      }
    })
  }

  def unifyForSameParameter(unifiedSubstitution: Substitution, substitutionC: Substitution, ctx: WeaveTypeResolutionContext): Substitution = {
    val typeParam: TypeParameter = unifiedSubstitution.solutions.head._1
    val firstWeaveType: WeaveType = unifiedSubstitution.solutions.head._2
    val secondWeaveType: WeaveType = substitutionC.solutions(typeParam)

    if (secondWeaveType.isInstanceOf[DynamicReturnType]) {
      Substitution(IdentityHashMap((typeParam, firstWeaveType.baseType())))
    } else {
      if (TypeHelper.canBeSubstituted(secondWeaveType, firstWeaveType, ctx)) {
        unifiedSubstitution
      } else {
        if (TypeHelper.canBeSubstituted(firstWeaveType, secondWeaveType, ctx)) {
          substitutionC
        } else {
          if (TypeHelper.canBeAssignedTo(secondWeaveType.baseType(), firstWeaveType.baseType(), ctx) || TypeHelper.canBeAssignedTo(firstWeaveType.baseType(), secondWeaveType.baseType(), ctx)) {
            Substitution(IdentityHashMap((typeParam, TypeHelper.unify(Seq(firstWeaveType, secondWeaveType)))))
          } else {
            //This is the case when the two substitutions can't be unify, so we return the first substitution, and late in the algorithm of resolving the constraints, it will fail (because it will update the typeParam with the first subsitution that don't match with the second one).
            unifiedSubstitution
          }
        }
      }
    }
  }

  override def resolve(ctx: WeaveTypeResolutionContext, coerce: Boolean, typeParametersToBeResolved: Seq[TypeParameter], theSubstitution: Substitution = emptySubstitution, warningMessages: Seq[Message] = Seq()): ConstraintResult = {
    if (constraints.isEmpty) {
      SolutionResult(theSubstitution, warningMessages)
    } else {
      val updatedConstraints: Seq[Constraint] = theSubstitution.update(ctx, constraints)
      val sortedConstraints: Seq[Constraint] = sortConstraints(updatedConstraints)
      var head: Constraint = sortedConstraints.head
      //If the constraint has dynamic returns we must force the resolution so that we can collect errors if necessary.
      if (hasDynamicReturn(head)) {
        val expectedCollector: MessageCollector = new MessageCollector()
        val actualCollector: MessageCollector = new MessageCollector()
        //We do this only if there is no substitution. This means that the default value is the only type constrain that is known
        val originalExpectedType = substituteTypeParamWithDefaultValue(head.expectedType)
        val originalActualType = substituteTypeParamWithDefaultValue(head.actualType)

        val expectedType = theSubstitution.apply(ctx, originalExpectedType, expectedCollector)
        val actualType = theSubstitution.apply(ctx, originalActualType, actualCollector)

        if (actualCollector.hasErrors() || expectedCollector.hasErrors()) {
          val messages = actualCollector.errorMessages.map((errorMessage) => {
            (errorMessage._1, InvalidDynamicReturnMessage(errorMessage._2, errorMessage._1))
          }) ++
            expectedCollector.errorMessages.map((errorMessage) => {
              (errorMessage._1, InvalidDynamicReturnMessage(errorMessage._2, errorMessage._1))
            })
          return ErrorResult(messages)
        } else {
          head = Constraint(expectedType, actualType, Constraint.ASSIGNMENT)
        }
      }
      val result: ConstraintResult = resolveConstraint(coerce, warningMessages, head, typeParametersToBeResolved, ctx)
      result match {
        case SolutionResult(substitution, _) =>
          ///foreach sortedConstraint.tail -> resolveConstraint if any has another solution for same type variable try to unify
          var unifiedSubstitution = substitution
          sortedConstraints.tail.foreach((constraint: Constraint) => {
            val resultC: ConstraintResult = resolveConstraint(coerce, warningMessages, constraint, typeParametersToBeResolved, ctx)
            resultC match {
              case SolutionResult(substitutionC, _) => {
                unifiedSubstitution =
                  if (substitution.solutions.nonEmpty && substitutionC.solutions.contains(substitution.solutions.head._1)) {
                    unifyForSameParameter(unifiedSubstitution, substitutionC, ctx)
                  } else {
                    unifiedSubstitution
                  }
              }
              case _ => unifiedSubstitution
            }
          })

          val mergedSubstitution = unifiedSubstitution.compose(ctx, theSubstitution)

          ConstrainProblem(sortedConstraints.tail).resolve(ctx, coerce, typeParametersToBeResolved, mergedSubstitution, warningMessages)
        case _ => result
      }
    }
  }

  private def sortConstraints(updatedConstraints: Seq[Constraint]) = {
    //We sort first type parameters. Resolving this first will enabled other unresolved types.
    //Second simple types
    //Third Elements with Dynamic Return
    //Forth dynamic return

    updatedConstraints.sortBy((constraint) => {
      val expectedType = constraint.expectedType
      val actualType = constraint.actualType
      var baseWeight =
        if (expectedType.isInstanceOf[TypeParameter] && !WeaveTypeTraverse.containsTypeParameter(actualType)) {
          1
        } else if (actualType.isInstanceOf[TypeParameter] && !WeaveTypeTraverse.containsTypeParameter(expectedType)) {
          2
        } else {
          10
        }
      if (constraint.actualType.isInstanceOf[TypeParameter]) {
        baseWeight = baseWeight + 2
      }
      if (constraint.actualType.isInstanceOf[DynamicReturnType]) {
        baseWeight = baseWeight + 100
      } else if (hasDynamicReturn(constraint)) {
        baseWeight = baseWeight + 50
      } else if (constraint.constraintType == Constraint.BOTTOM || constraint.constraintType == Constraint.TOP) {
        baseWeight = baseWeight + 40
      } else if (hasNothing(constraint)) {
        baseWeight = baseWeight + 25
      }
      baseWeight
    })
  }

  private def resolveConstraint(coerce: Boolean, warningMessages: Seq[Message], head: Constraint, typeParametersToBeResolved: Seq[TypeParameter], ctx: WeaveTypeResolutionContext): ConstraintResult = {
    val messageCollector = MessageCollector()
    head match {
      case Constraint(et: TypeParameter, actualType, Constraint.TOP) if et.isAbstract() =>
        buildSubstitution(et, actualType.baseType(), typeParametersToBeResolved, ctx)
      case Constraint(et: TypeParameter, actualType, _) if et.isAbstract() =>
        buildSubstitution(et, actualType, typeParametersToBeResolved, ctx)
      case Constraint(et, atp: TypeParameter, Constraint.ASSIGNMENT) if atp.isAbstract() =>
        buildSubstitution(atp, et, typeParametersToBeResolved, ctx)
      case _ => {
        if (!TypeHelper.canBeSubstituted(head.actualType, head.expectedType, ctx, messageCollector)) {
          if (!coerce || TypeCoercer.coerce(head.expectedType, head.actualType, ctx).isEmpty) {
            ErrorResult(messageCollector.errorMessages)
          } else {
            SolutionResult(emptySubstitution, warningMessages ++ Seq(TypeCoercedMessage(head.expectedType, head.actualType)))
          }
        } else {
          SolutionResult(emptySubstitution, warningMessages)
        }
      }
    }
  }

  def hasDynamicReturn(constraint: Constraint): Boolean = {
    FunctionTypeHelper.hasDynamicReturn(constraint.expectedType) || FunctionTypeHelper.hasDynamicReturn(constraint.actualType)
  }

  private def hasNothing(constraint: Constraint) = {
    FunctionTypeHelper.hasNothingType(constraint.expectedType) || FunctionTypeHelper.hasNothingType(constraint.actualType) || constraint.actualType.isInstanceOf[NothingType] || constraint.expectedType.isInstanceOf[NothingType]
  }

  def buildSubstitution(t: TypeParameter, wt: WeaveType, typeParametersToBeResolved: Seq[TypeParameter], ctx: WeaveTypeResolutionContext): ConstraintResult = {
    wt match {
      case tp: TypeParameter if tp eq t => {
        SolutionResult(emptySubstitution, Seq())
      }
      case UnionType(of) if of.exists(_ eq t) => {
        val uniqueTypes = of.filterNot(_ eq t)
        if (uniqueTypes.isEmpty) {
          SolutionResult(emptySubstitution, Seq())
        } else {
          buildSubstitution(t, TypeHelper.unify(uniqueTypes), typeParametersToBeResolved, ctx)
        }
      }
      case _ if Constraint.isNestedIn(t, wt) => {
        ErrorResult(Seq((t.location(), NestedTypeMessage(t, wt))))
      }
      //If the type parameter in the right belongs to the Type parameters we are trying to solve
      //Then we need to also add the counter part as we want to be able to replace all those type params.
      // Except for the DynamicTypeParams, this are not real Type parameters so we don't want it in the solution
      case tp: TypeParameter if !isDynamicTypeParameter(t) && (typeParametersToBeResolved.exists(_ eq tp)) => {
        SolutionResult(Substitution(IdentityHashMap((tp, t), (t, wt))), Seq())
      }
      case _ => {
        SolutionResult(Substitution(IdentityHashMap((t, wt))), Seq())
      }
    }
  }
}

class Substitution(val solutions: IdentityHashMap[TypeParameter, WeaveType]) {

  override def toString: String = s"Substitution(${solutions.toString()})"

  def solutionFor(typeParameter: TypeParameter): Option[WeaveType] = {
    //We meed to search by identity as two TypeParameter may have same name but are not the same type parameter
    solutions.get(typeParameter)
  }

  def apply(ctx: WeaveTypeResolutionContext, typeToSubstitute: WeaveType, messageCollector: MessageCollector = new MessageCollector()): WeaveType = {
    Constraint.substitute(typeToSubstitute, this, ctx, messageCollector = messageCollector, recursionDetector = TypeHelper.createRecursionDetector())
  }

  /**
    * Updates the constraints with this solution
    *
    * @param ctx         The context
    * @param constraints The constraints
    * @return The updated constraints
    */
  def update(ctx: WeaveTypeResolutionContext, constraints: Seq[Constraint]): Seq[Constraint] = {
    constraints.flatMap((constraint) => {
      val actualType = constraint.actualType
      val result = constraint.copy(expectedType = apply(ctx, constraint.expectedType), actualType = apply(ctx, actualType))
      if (actualType.isInstanceOf[DynamicReturnType]) {
        val constraintSet = Constraint.collectConstrains(result.expectedType, result.actualType, ctx)
        constraintSet match {
          case ConstrainProblem(constraints) => {
            //We add new constrains
            val resolvedConstrains = constraints.filterNot(_ == result)
            Seq(result) ++ resolvedConstrains
          }
          case _ => Seq(result)
        }
      } else {
        Seq(result)
      }

    })
  }

  def hasDynamicReturn(constraint: Constraint): Boolean = {
    FunctionTypeHelper.hasDynamicReturn(constraint.expectedType) || FunctionTypeHelper.hasDynamicReturn(constraint.actualType)
  }

  /**
    * Update this solution with the other solution and merge it with this
    *
    * @param ctx   The resolution context
    * @param other The other Substitution to compose with
    * @return A new Substitution
    */
  def compose(ctx: WeaveTypeResolutionContext, other: Substitution): Substitution = {
    val substitutedThis = solutions.mapValues(s => other.apply(ctx, s))
    other.mergeWith(substitutedThis)
  }

  private def mergeWith(substitutedThis: IdentityHashMap[TypeParameter, WeaveType]): Substitution = {
    Substitution(substitutedThis ++ solutions)
  }

}

object Substitution {
  def apply(solutions: IdentityHashMap[TypeParameter, WeaveType]) = new Substitution(solutions)

  def apply(solutions: Seq[(TypeParameter, WeaveType)]) = new Substitution(IdentityHashMap(solutions: _*))

  def apply() = new Substitution(new IdentityHashMap[TypeParameter, WeaveType]())
}

sealed trait ConstraintResult {}

case class ErrorResult(problems: Seq[(WeaveLocation, Message)]) extends ConstraintResult

case class SolutionResult(substitution: Substitution, warnings: Seq[Message]) extends ConstraintResult

object Constraint {

  val ASSIGNMENT = 0
  val TOP = 1
  val BOTTOM = 2

  def toString(constraintSet: ConstraintSet): String = {
    constraintSet match {
      case NoSolutionSet(problems) => {
        s"Problems:\n\t- ${problems.mkString("\n\t- ")}"
      }
      case MultiOptionConstrainProblem(constraintsSets) => {
        constraintsSets.map(toString).mkString("\nOR\n")
      }
      case ConstrainProblem(constraints) => {
        s"{\n\t${constraints.sortBy(_.actualType.toString).map((c) => c.expectedType + " <- " + c.actualType).mkString(",\n\t")}\n}"
      }
    }
  }

  /**
    *
    * Collects all the constrains or equations that need to be resolved so that this assignment to happen.
    *
    * Inplementation detail.
    *
    * We only support Top Types for extracting date We transform bottom expressions to a TopExpression
    *
    *  Something like T >: {||} ==> T <: {}
    *                T >: "" ==> T <: String
    * This transformation is not perfect as we are bounding T to the base type of that Type and we are discarding Any
    *
    *
    * @param expected The expected Type
    * @param actual The actual Type
    * @param ctx The context
    * @param recursionDetector Detects Recursion
    * @param constraintType The kind of Constrain Type . Equal, Top or Bottom
    * @return The set of constrains detected
    */
  def collectConstrains(expected: WeaveType, actual: WeaveType, ctx: WeaveTypeResolutionContext, recursionDetector: DoubleRecursionDetector[ConstraintSet] = DoubleRecursionDetector((_, _) => EmptyConstrainProblem), constraintType: Int = Constraint.ASSIGNMENT): ConstraintSet = {
    if (expected.isInstanceOf[ReferenceType] || actual.isInstanceOf[ReferenceType]) {
      return recursionDetector.resolve(expected, actual, (typePair) => {
        collectConstrains(typePair.left, typePair.right, ctx, recursionDetector, constraintType)
      })
    }

    actual match {
      case UnionType(of) => {
        val constraintSets = of.map((aType) => {
          collectConstrains(expected, aType, ctx, recursionDetector, constraintType)
        })
        val constraintSet: ConstraintSet = constraintSets.foldLeft[ConstraintSet](EmptyConstrainProblem)(
          (acc, value) => { acc.merge(value) })
        return groupByExpected(constraintSet, ctx)
      }
      case IntersectionType(of) => {
        val resolvedIntersectionType = TypeHelper.resolveIntersection(of)
        return resolvedIntersectionType match {
          case IntersectionType(actualOf) => {
            expected match {
              case IntersectionType(expectedOf) => {
                val constrains = SeqUtils
                  .combine(Seq(actualOf, expectedOf))
                  .toArray
                  .sliding(actualOf.size)
                  .map((combination) => {
                    combination
                      .map((pair) => {
                        collectConstrains(pair(1), pair(0), ctx, recursionDetector, constraintType)
                      })
                      .reduce(_.merge(_))
                  }).toSeq
                MultiOptionConstrainProblem(constrains)
              }
              case _ => actualOf
                .map(collectConstrains(expected, _, ctx, recursionDetector, constraintType))
                .reduce((acc, cur) => { acc.merge(cur) })
            }
          }
          case _ => collectConstrains(expected, resolvedIntersectionType, ctx, recursionDetector, constraintType)
        }
      }
      case drt: DynamicReturnType => {
        val problem = ConstrainProblem(Seq(Constraint(expected, drt, constraintType)))
        return expected match {
          case expectedTP: TypeParameter => {
            if (expectedTP.top.isDefined) {
              collectConstrains(expectedTP.top.get, actual, ctx, recursionDetector, constraintType).merge(problem)
            } else {
              problem
            }
          }
          case _ => problem
        }
      }
      case _ =>
    }

    val messageCollector = MessageCollector()
    val expectedConstraints: ConstraintSet = expected match {
      case ArrayType(expectedArrayType) => {
        actual match {
          case ArrayType(actualArrayType) => collectConstrains(expectedArrayType, actualArrayType, ctx, recursionDetector, constraintType)
          case _ => {
            val canBeSubstituted = TypeHelper.canBeSubstituted(actual, expected, ctx, messageCollector)
            if (canBeSubstituted) {
              EmptyConstrainProblem
            } else {
              noSolutionTypeMismatch(expected, actual, messageCollector)
            }
          }
        }
      }
      case ObjectType(eProps, _, _) => {
        val canBeSubstituted = TypeHelper.canBeSubstituted(actual, expected, ctx, messageCollector)
        actual match {
          case aobj: ObjectType if canBeSubstituted => {
            var matchedProperties = Seq[KeyValuePairType]()
            val constraintSets: Seq[ConstraintSet] = aobj.properties.map((aProp) => {
              val foundProp: Option[KeyValuePairType] = eProps.find((eProp) => TypeHelper.canBeSubstituted(aProp, eProp, ctx))
              foundProp match {
                case Some(eProp) => {
                  matchedProperties = matchedProperties :+ eProp
                  val constraintSet: ConstraintSet = collectConstrains(eProp.key, aProp.key, ctx, recursionDetector, constraintType)
                    .merge(collectConstrains(eProp.value, aProp.value, ctx, recursionDetector, constraintType))
                  constraintSet
                }
                case None => EmptyConstrainProblem
              }
            })

            val isEmptyOpenObject: Boolean = aobj.properties.isEmpty
            val anyKey: KeyType = KeyType(NameType())
            copyLocation(anyKey, aobj.location())
            //If it is close then is Nothing as there are no values it is never going to happen
            val anyValue: WeaveType = if (aobj.close) NothingType() else AnyType()
            copyLocation(anyValue, aobj.location())

            //If we have optional key value pairs that hasn't been match then add a constraint to UnknownType
            val optionalConstraints: Seq[ConstrainProblem] = eProps.flatMap((eProp) => {
              if (matchedProperties.contains(eProp)) {
                None
              } else if (isEmptyOpenObject && TypeHelper.canBeSubstituted(anyKey, eProp.key, ctx) && TypeHelper.canBeSubstituted(anyValue, eProp.value, ctx)) {
                //This means that is an {} so we are going to treat it like if it were {_?: Any}
                Some(ConstrainProblem(Seq(Constraint(eProp.key, anyKey, constraintType), Constraint(eProp.value, anyValue, constraintType))))
              } else {
                None
              }
            })

            //We merge all the matches
            val set: ConstraintSet = (constraintSets ++ optionalConstraints).foldLeft[ConstraintSet](EmptyConstrainProblem)((acc, value) => {
              acc.merge(value)
            })
            groupByExpected(set, ctx)
          }
          case _ if canBeSubstituted => collectBaseConstraints(expected, ctx, constraintType)
          case _ => {
            noSolutionTypeMismatch(expected, actual, messageCollector)
          }
        }
      }
      case KeyType(eName, eAttrs) => {
        actual match {
          case KeyType(aName, aAttrs) => {
            val attrConstrains: Seq[ConstraintSet] = aAttrs.map((aAttr) => {
              val foundAttr = eAttrs.find((eAttr) => TypeHelper.canBeSubstituted(aAttr, eAttr, ctx))
              foundAttr match {
                case Some(eAttr) =>
                  collectConstrains(eAttr.name, aAttr.name, ctx, recursionDetector, constraintType)
                    .merge(collectConstrains(eAttr.value, aAttr.value, ctx, recursionDetector, constraintType))
                case None => {
                  NoSolutionSet(Seq((actual.location(), TypeMismatch(expected, actual))))
                }
              }
            })
            attrConstrains.foldLeft(collectConstrains(eName, aName, ctx, recursionDetector, constraintType))((acc, value) => acc.merge(value))
          }
          case _ if TypeHelper.canBeSubstituted(actual, expected, ctx, messageCollector) => EmptyConstrainProblem
          case _ => NoSolutionSet(Seq((actual.location(), TypeMismatch(expected, actual))))
        }
      }
      case IntersectionType(of) => TypeHelper.resolveIntersection(of) match {
        case e: IntersectionType => {
          ConstrainProblem(Seq(Constraint(expectedType = e, actualType = actual, constraintType = constraintType)))
        }
        case e => collectConstrains(e, actual, ctx, recursionDetector, constraintType)
      }
      case UnionType(of) => {
        val parts = TypeHelper.inlineUnionTypes(of)
        //In the case of a union type on the expected we need to fork the constraint problems
        //This mean that it can be more than one solution and that some of the problems may not have a resolution
        val constraintSets = parts.map((eType) => {
          (eType, collectConstrains(eType, actual, ctx, recursionDetector, constraintType))
        })

        val correctOnes = constraintSets.filterNot((cs) => cs._2.isInstanceOf[NoSolutionSet])
        //Check if there is any possible solution
        if (correctOnes.isEmpty) {
          //If no solution the show the problem
          val errors = constraintSets
            .collect({
              case (wtype, nss: NoSolutionSet) if (!wtype.isInstanceOf[NullType]) => {
                val problems: Seq[(WeaveLocation, Message)] = nss.problems
                applyTrace(problems, expected)
              }
              case _ => Seq() //Filter out null options
            })
            .flatten
          NoSolutionSet(errors)
        } else if (correctOnes.size == 1) {
          correctOnes.head._2
        } else {
          MultiOptionConstrainProblem(correctOnes.map((pair) => pair._2))
        }
      }
      case tp: TypeParameter => {
        var constrainProblem: ConstraintSet = ConstrainProblem(Seq(Constraint(tp, actual, constraintType)))
        if (tp.top.isDefined) {
          constrainProblem = constrainProblem.merge(collectConstrains(tp.top.get, actual, ctx, recursionDetector, Constraint.TOP))
        }
        if (tp.bottom.isDefined) {
          constrainProblem = constrainProblem.merge(collectConstrains(TypeHelper.toTopType(tp.bottom.get), actual, ctx, recursionDetector, Constraint.TOP))
        }

        constrainProblem match {
          case NoSolutionSet(problems) => {
            problems.foreach(t => {
              t._2 match {
                // If top or bottom constraint failed we add trace back to the type parameter
                case tm: TypeMessage => tm.addTrace(tp)
                case _               =>
              }
            })
          }
          case _ =>
        }
        constrainProblem
      }
      case TypeType(et) => actual match {
        case TypeType(at) => collectConstrains(et, at, ctx, recursionDetector, constraintType)
        case _ if TypeHelper.canBeSubstituted(actual, expected, ctx, messageCollector) => EmptyConstrainProblem
        case _ => NoSolutionSet(Seq((actual.location(), TypeMismatch(expected, actual))))
      }
      case et @ FunctionType(_, expectedParams, expectedReturnType, _, _, _) => //TODO: add type parameter constraint
        actual match {
          case at @ FunctionType(_, actualParams, actualReturnType, actualOverloads, _, _) => {
            if (actualOverloads.isEmpty) {
              val argsConstraints: Seq[ConstraintSet] = actualParams
                .zip(expectedParams)
                .map((pairArg) => {
                  val actualParamType: WeaveType = pairArg._1.defaultValueType match {
                    case Some(weaveType) if (TypeHelper.isJustTypeParameter(pairArg._1.wtype)) => {
                      //We convert bottom types into top types this is a simple way to implement it
                      pairArg._1.wtype.asInstanceOf[TypeParameter].copy(top = Some(TypeHelper.toTopType(weaveType)))
                    }
                    case _ => {
                      pairArg._1.wtype
                    }
                  }
                  val expectedParamType: WeaveType = pairArg._2.defaultValueType match {
                    case Some(weaveType) if (TypeHelper.isJustTypeParameter(pairArg._2.wtype)) => {
                      //We convert bottom types into top types this is a simple way to implement it
                      pairArg._2.wtype.asInstanceOf[TypeParameter].copy(top = Some(TypeHelper.toTopType(weaveType)))
                    }
                    case _ => {
                      pairArg._2.wtype
                    }
                  }
                  val solution: ConstraintSet = collectConstrains(actualParamType, expectedParamType, ctx, recursionDetector, constraintType)
                  solution match {
                    case NoSolutionSet(_) => {
                      //We need to re create it
                      NoSolutionSet(Seq((actualParamType.location(), TypeMismatch(expectedParamType, actualParamType))))
                    }
                    case _ => solution
                  }
                })

              //We validate arity of the functions
              val messages: Seq[(WeaveLocation, Message)] = if (expectedParams.size < actualParams.size) {
                //If head has optional
                if (actualParams.headOption.exists(_.optional)) {
                  val delta: Int = actualParams.length - expectedParams.size
                  //Make sure all non validated params are optional
                  val paramsSplit = actualParams.splitAt(delta)
                  if (paramsSplit._1.exists(!_.optional)) {
                    Seq((at.location(), TooManyArgumentMessage(expectedParams.map(_.wtype), actualParams.map(_.wtype), et, actual)))
                  } else {
                    Seq()
                  }
                } else {
                  //Make sure all non validated params are optional
                  val assignedArgsSplitByDefault = actualParams.splitAt(expectedParams.size)
                  if (assignedArgsSplitByDefault._2.exists(!_.optional)) {
                    Seq((at.location(), TooManyArgumentMessage(expectedParams.map(_.wtype), actualParams.map(_.wtype), et, actual)))
                  } else {
                    Seq()
                  }
                }
              } else if (expectedParams.size != actualParams.size) {
                Seq((at.location(), NotEnoughArgumentMessage(expectedParams.map(_.wtype), actualParams.map(_.wtype), et, actual)))
              } else {
                Seq()
              }
              //If there are error in the arity return with no solution else merge the constraints
              if (messages.nonEmpty) {
                noSolutionTypeMismatch(expected, actual, messages)
              } else {
                argsConstraints.foldLeft(collectConstrains(expectedReturnType, actualReturnType, ctx, recursionDetector, constraintType))(_.merge(_))
              }
            } else {
              //We look for the first set of constraint that doesn't fail with basic types
              val constraintSets: Seq[ConstraintSet] = actualOverloads.toStream
                .map((actualFunction) => {
                  collectConstrains(expected, actualFunction, ctx, recursionDetector, constraintType)
                })
              //We collect all valid options
              val constrainProblems = constraintSets
                .collect({ case cs: ConstrainProblem => cs })
              //If there is none that is valid then we fail
              if (constrainProblems.isEmpty) {
                constraintSets.head
              } else if (constrainProblems.size == 1) {
                //If only one we just return that
                // one
                constrainProblems.head
              } else {
                //If multiple we build a multioption same as we do with union types
                MultiOptionConstrainProblem(constrainProblems)
              }

            }
          }
          case _ if TypeHelper.canBeSubstituted(actual, et, ctx) => collectBaseConstraints(et, ctx, constraintType)
          case _ => NoSolutionSet(Seq((actual.location(), TypeMismatch(expected, actual))))
        }
      case _ if TypeHelper.canBeSubstituted(actual, expected, ctx, messageCollector) => EmptyConstrainProblem
      case _ => noSolutionTypeMismatch(expected, actual, messageCollector)
    }

    //Only constraints we can draw from the actual types are given by their lower and upper bound
    val actualConstraints: ConstraintSet = actual match {
      case atp @ TypeParameter(_, top, bottom, _, _) if !TypeHelper.isArithmeticType(expected) => {
        val baseConstraint = if (!expected.isInstanceOf[TypeParameter]) {
          ConstrainProblem(Seq(Constraint(expected, atp, constraintType)))
        } else {
          EmptyConstrainProblem
        }
        val bottomConstraint = bottom match {
          case Some(actualBottom) => {
            ConstrainProblem(Seq(Constraint(TypeHelper.toTopType(actualBottom), atp, Constraint.BOTTOM)))
          }
          case None => EmptyConstrainProblem
        }
        val topConstraint = top match {
          case Some(actualTop) => {
            ConstrainProblem(Seq(Constraint(actualTop, atp, Constraint.TOP)))
          }
          case None => EmptyConstrainProblem
        }

        baseConstraint.merge(bottomConstraint).merge(topConstraint)
      }
      case _ => EmptyConstrainProblem
    }

    actualConstraints.merge(expectedConstraints)
  }

  private def collectBaseConstraints(wt: WeaveType, ctx: WeaveTypeResolutionContext, constraintType: Int): ConstraintSet = {
    wt match {
      case ObjectType(eProps, _, _) => {
        val optionalConstraints: Seq[ConstrainProblem] = eProps.flatMap(eProp => {
          val anyKey: KeyType = KeyType(NameType())
          val anyValue: WeaveType = AnyType()
          if (TypeHelper.canBeSubstituted(anyKey, eProp.key, ctx) && TypeHelper.canBeSubstituted(anyValue, eProp.value, ctx)) {
            //This means that is an {} so we are going to treat it like if it were {_?: Any}
            Some(ConstrainProblem(Seq(Constraint(eProp.key, anyKey, constraintType), Constraint(eProp.value, anyValue, constraintType))))
          } else {
            None
          }
        })

        val set: ConstraintSet = optionalConstraints.foldLeft[ConstraintSet](EmptyConstrainProblem)((acc, value) => {
          acc.merge(value)
        })
        groupByExpected(set, ctx)
      }
      case _ => EmptyConstrainProblem
    }
  }
  private def applyTrace(problems: Seq[(WeaveLocation, Message)], traces: WeaveType*) = {
    problems.foreach((p) => {
      p._2 match {
        case tm: TypeMessage => {
          traces.foreach((trace) => {
            tm.addTrace(trace)
          })
        }
        case _ =>
      }
    })
    problems
  }

  private def noSolutionTypeMismatch(expected: WeaveType, actual: WeaveType, messageCollector: MessageCollector): NoSolutionSet = {
    noSolutionTypeMismatch(expected, actual, messageCollector.errorMessages)
  }

  private def noSolutionTypeMismatch(expected: WeaveType, actual: WeaveType, errorMessages: Seq[(WeaveLocation, Message)]): NoSolutionSet = {
    if (errorMessages.isEmpty) {
      NoSolutionSet(Seq((actual.location(), TypeMismatch(expected, actual))))
    } else {
      NoSolutionSet(applyTrace(errorMessages, expected))
    }
  }

  def groupByExpected(set: ConstraintSet, ctx: WeaveTypeResolutionContext): ConstraintSet = {
    val unifiableConstraints: ArrayBuffer[(WeaveType, Int, ArrayBuffer[WeaveType])] = ArrayBuffer.empty

    set match {
      case ConstrainProblem(constraints) =>
        /**
          * Group constraints of a ConstrainProblem by expected type and constraint type
          * and return a new one where each of those groups is replaced by a new constraint where
          * its actual type is the unification of all respective actual types.
          *
          * e.g:
          *   ConstrainProblem => [(T1 -> Int), (T2 -> Number), (T1 -> {})]
          *   unify(ConstrainProblem) => [(T1 -> Int | {}), (T2 -> Number)]
          */
        constraints.foreach(constraint => {
          val entry = unifiableConstraints.find(
            uc => (uc._1 eq constraint.expectedType) && uc._2 == constraint.constraintType)
          entry match {
            case None        => unifiableConstraints += Tuple3(constraint.expectedType, constraint.constraintType, ArrayBuffer(constraint.actualType))
            case Some(value) => value._3 += constraint.actualType
          }
        })

        val unifiedConstraints = unifiableConstraints.map(entry => {
          val newActualType = TypeHelper.unify(entry._3)
          Constraint(entry._1, newActualType, entry._2)
        })
        ConstrainProblem(unifiedConstraints)
      case MultiOptionConstrainProblem(constraints) =>
        MultiOptionConstrainProblem(constraints.map((cs) => groupByExpected(cs, ctx)))
      case _ => set
    }
  }

  /**
    * Recursive nested calls are replaced with Any as they are recursive definitions and never stop the recursion
    * As the type is defined nested
    */
  def replaceRecursiveDefinition(value: WeaveType, functionNode: FunctionNode): WeaveType = {
    WeaveTypeTraverse.treeMap(
      value, {
      //It is recursive it is a DynReturn of the same function Then it will never cut
      case recursive: DynamicReturnType if recursive.node eq functionNode => {
        AnyType()
      }
    })
  }

  /**
    * Resolves type parameters from context
    *
    * @param weaveType    The return type to resolve
    * @param substitution The context
    * @return The return type resolved
    */
  def substitute(weaveType: WeaveType, substitution: Substitution, ctx: WeaveTypeResolutionContext, resolveDR: Boolean = false, messageCollector: MessageCollector = new MessageCollector(), recursionDetector: RecursionDetector[WeaveType] = TypeHelper.createRecursionDetector()): WeaveType = {
    WeaveTypeTraverse.treeMap(
      weaveType, {
      case tp: TypeParameter => {
        val parameter: Option[WeaveType] = substitution.solutionFor(tp)
        parameter match {
          case Some(replacement: TypeParameter) => replacement //If it is a TypeParameter no more to replace
          case Some(x)                          => substitute(x, substitution, ctx, resolveDR, messageCollector, recursionDetector)
          case None                             => tp
        }
      }
      case drt @ DynamicReturnType(arguments, functionNode, typeGraph, scopesNavigator, functionName, expectedReturnType, resolver) => {
        val substitutedParameters: Seq[FunctionTypeParameter] = arguments.map({
          case FunctionTypeParameter(name, wtype, optional, defaultValueType) => {
            val weaveType1 = substitute(wtype, substitution, ctx, resolveDR, messageCollector, recursionDetector)
            FunctionTypeParameter(name, weaveType1, optional, defaultValueType)
          }
        })
        val incomingArgTypes: Seq[WeaveType] = substitutedParameters.map(_.wtype)
        if (areArgumentsSubstituted(incomingArgTypes) && ctx.rootGraph != null) {
          val expectedReturnTypeSubstituted = expectedReturnType.map((rt) => {
            substitute(rt, substitution, ctx, resolveDR, messageCollector, recursionDetector)
          })
          val expressionType = FunctionTypeHelper.resolveReturnType(incomingArgTypes, expectedReturnTypeSubstituted, ctx, drt, resolveDR, messageCollector)
          if (expressionType.isDefined) {
            val value = expressionType.get
            replaceRecursiveDefinitionIfRequired(functionNode, value, ctx)
          } else {
            DynamicReturnType(substitutedParameters, functionNode, typeGraph, scopesNavigator, functionName, expectedReturnType, resolver)
          }
        } else {
          DynamicReturnType(substitutedParameters, functionNode, typeGraph, scopesNavigator, functionName, expectedReturnType, resolver)
        }
      }
    },
      recursionDetector)
  }

  private def replaceRecursiveDefinitionIfRequired(functionNode: FunctionNode, value: WeaveType, ctx: WeaveTypeResolutionContext): WeaveType = {
    value match {
      case returnType: DynamicReturnType if returnType.node eq functionNode =>
        value
      case UnionType(of) =>
        TypeHelper.unify(of.map((wtype) => replaceRecursiveDefinitionIfRequired(functionNode, wtype, ctx)))
      case _ =>
        replaceRecursiveDefinition(value, functionNode)
    }
  }

  private def areArgumentsSubstituted(incomingArgTypes: Seq[WeaveType]) = {
    !incomingArgTypes.exists(isDynamicTypeParameter)
  }

  def isNestedIn(typeParam: TypeParameter, weaveType: WeaveType): Boolean = {
    WeaveTypeTraverse.exists(weaveType, {
      case tp: TypeParameter => typeParam eq tp
      case _                 => false
    })
  }
}