package org.mule.weave.v2.api.tooling.impl.ts.catalog

import org.mule.weave.v2.api.tooling.impl.ts.catalog.DefaultDWTypeCatalog.WeaveTypeCatalogEmitter
import org.mule.weave.v2.api.tooling.ts.catalog.DWTypeCatalog
import org.mule.weave.v2.api.tooling.ts.catalog.DWTypeCatalogLoader
import org.mule.weave.v2.api.tooling.ts.catalog.ErrorHandler
import org.mule.weave.v2.codegen.StringCodeWriter
import org.mule.weave.v2.parser.ModuleParser
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.header.directives.TypeDirective
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.sdk.ParsingContextFactory
import org.mule.weave.v2.sdk.WeaveResourceFactory
import org.mule.weave.v2.api.tooling.ts.DWType
import org.mule.weave.v2.ts.AnyType
import org.mule.weave.v2.ts.ArrayType
import org.mule.weave.v2.ts.BinaryType
import org.mule.weave.v2.ts.BooleanType
import org.mule.weave.v2.ts.DateTimeType
import org.mule.weave.v2.ts.DynamicReturnType
import org.mule.weave.v2.ts.FunctionType
import org.mule.weave.v2.ts.FunctionTypeHelper
import org.mule.weave.v2.ts.IntersectionType
import org.mule.weave.v2.ts.KeyType
import org.mule.weave.v2.ts.KeyValuePairType
import org.mule.weave.v2.ts.LocalDateTimeType
import org.mule.weave.v2.ts.LocalDateType
import org.mule.weave.v2.ts.LocalTimeType
import org.mule.weave.v2.ts.Metadata
import org.mule.weave.v2.ts.MetadataConstraint
import org.mule.weave.v2.ts.NameType
import org.mule.weave.v2.ts.NameValuePairType
import org.mule.weave.v2.ts.NamespaceType
import org.mule.weave.v2.ts.NothingType
import org.mule.weave.v2.ts.NullType
import org.mule.weave.v2.ts.NumberType
import org.mule.weave.v2.ts.ObjectType
import org.mule.weave.v2.ts.PeriodType
import org.mule.weave.v2.ts.RangeType
import org.mule.weave.v2.ts.RecursionDetector
import org.mule.weave.v2.ts.RegexType
import org.mule.weave.v2.ts.SimpleReferenceType
import org.mule.weave.v2.ts.StringType
import org.mule.weave.v2.ts.TimeType
import org.mule.weave.v2.ts.TimeZoneType
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.TypeParameter
import org.mule.weave.v2.ts.TypeSelectorType
import org.mule.weave.v2.ts.TypeType
import org.mule.weave.v2.ts.UnionType
import org.mule.weave.v2.ts.UriType
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.ts.WeaveTypeTraverse
import org.mule.weave.v2.utils.Optionals.toJavaOptional
import org.mule.weave.v2.utils.StringEscapeHelper
import org.mule.weave.v2.utils.WeaveTypeMetadataEmitter

import java.util.Optional
import scala.collection.JavaConverters.mapAsScalaMapConverter
import scala.collection.mutable

class DefaultDWTypeCatalog private (private val typesProvider: () => Map[String, WeaveType]) extends DWTypeCatalog {

  private lazy val catalogTypes = typesProvider.apply()

  override def write(): String = {
    new WeaveTypeCatalogEmitter().emit(catalogTypes)
  }

  override def getType(name: String): Optional[DWType] = {
    val resultType: Option[DWType] = catalogTypes.get(name)
    resultType.asJava
  }

  override def getTypeNames: Array[String] = {
    catalogTypes.keys.toArray
  }
}

object DefaultDWTypeCatalog {

  private class DefaultWeaveTypeCatalogLoader extends DWTypeCatalogLoader {

    private var weaveTypes: mutable.Map[String, WeaveType] = mutable.Map()
    private var callBack: Optional[ErrorHandler] = Optional.empty()

    override def build(): DWTypeCatalog = {
      new DefaultDWTypeCatalog(() => weaveTypes.toMap)
    }

    override def withType(name: String, weaveType: DWType): DWTypeCatalogLoader = {
      weaveTypes.put(name, weaveType.asInstanceOf[WeaveType])
      this
    }

    override def withTypes(types: java.util.Map[String, DWType]): DWTypeCatalogLoader = {
      weaveTypes = mutable.Map(types.asScala.map(entry => (entry._1, entry._2.asInstanceOf[WeaveType])).toSeq: _*)
      this
    }

    override def withErrorHandler(callback: ErrorHandler): DWTypeCatalogLoader = {
      callBack = Optional.of(callback)
      this
    }

    override def fromCatalog(typeCatalog: String): DWTypeCatalog = {
      val typesProvider = () => {
        val parsingContext = ParsingContextFactory.createParsingContext()
        val parseResult = ModuleParser.parse(ModuleParser.scopePhase(), WeaveResourceFactory.fromContent(typeCatalog), parsingContext)
        if (parseResult.hasErrors()) {
          val errorMessages = parseResult.errorMessages().map(messages => {
            messages._2.message + "\nat \n" + messages._1.locationString
          }).toArray
          if (callBack.isPresent) {
            callBack.get().onError(errorMessages)
          }
        }
        val types: Map[String, WeaveType] = {
          if (parseResult.hasResult()) {
            val result = parseResult.getResult()
            val node = result.astNode
            val typeDirectives = AstNodeHelper.collectChildrenWith(node, classOf[TypeDirective])
            typeDirectives.map(td => {
              (td.variable.name, org.mule.weave.v2.ts.WeaveType(td.typeExpression, result.scope.referenceResolver))
            }).toMap
          } else {
            Map()
          }

        }
        types
      }
      new DefaultDWTypeCatalog(typesProvider)
    }

  }

  def loader(): DWTypeCatalogLoader = {
    new DefaultWeaveTypeCatalogLoader()
  }

  private case class WeaveTypeToEmit(weaveType: WeaveType, typeParameters: Seq[TypeParameter]) {
    def getWeaveType: WeaveType = weaveType

    def getTypeParameters: Seq[TypeParameter] = typeParameters
  }

  private object WeaveTypeToEmit {
    def apply(weaveType: WeaveType): WeaveTypeToEmit = WeaveTypeToEmit(weaveType, TypeHelper.collectTypeParameters(weaveType))
  }

  private class WeaveTypeCatalogEmitter {
    private val emittedTypes: mutable.Map[String, WeaveType] = mutable.Map()
    private var typesToEmit: mutable.Map[String, WeaveTypeToEmit] = mutable.Map()
    private val namespaces: mutable.Map[String, String] = mutable.Map()
    private var objectKeyAnnotationsToEmit: List[String] = List()

    def emit(types: Map[String, WeaveType], typeDeclarations: StringCodeWriter = new StringCodeWriter()): String = {
      generateTypeCatalog(types.map(e => (e._1, WeaveTypeToEmit(e._2))), typeDeclarations)
    }

    private def mergeEmitterMetadata(other: WeaveTypeCatalogEmitter): Unit = {
      emittedTypes ++= other.emittedTypes
      typesToEmit ++= other.typesToEmit
      namespaces ++= other.namespaces
      objectKeyAnnotationsToEmit ++= other.objectKeyAnnotationsToEmit
    }

    private def generateTypeCatalog(types: Map[String, WeaveTypeToEmit], typeDeclarations: StringCodeWriter = new StringCodeWriter()): String = {
      types.foreach(entry => {
        if (!emittedTypes.contains(entry._1)) {
          generateType(entry._1, entry._2, typeDeclarations)
        }
      })

      val header = new StringCodeWriter()
      header.println("%dw 2.0")
      if (namespaces.nonEmpty) {
        header.println("// namespaces")
        namespaces.foreach(namespace => {
          header.println(s"ns ${namespace._2} ${namespace._1}")
        })
      }
      if (objectKeyAnnotationsToEmit.nonEmpty) {
        objectKeyAnnotationsToEmit.foreach(objectKey => generateMetadataAnnotationDefinition(header, objectKey))
      }
      header.toString + typeDeclarations.toString
    }

    private def generateMetadataAnnotationDefinition(stringWriter: StringCodeWriter, objectKey: String) = {
      stringWriter.println()
      stringWriter.println(s"""@Metadata(key = "$objectKey")""")
      stringWriter.println(s"annotation ${objectKey.capitalize}(value: Any)")
    }

    private def generateType(rootName: String, typeToEmit: WeaveTypeToEmit, builder: StringCodeWriter = new StringCodeWriter()): String = {
      val typeParameters = typeToEmit.getTypeParameters
      val wtype = typeToEmit.getWeaveType
      val recursionDetector = RecursionDetector[String]((id, _) => {
        id.name
      })

      builder.print(s"\ntype $rootName")
      if (typeParameters.nonEmpty) {
        val boundedTypeEmitter = new WeaveTypeCatalogEmitter()
        builder.print("<")
        builder.printForeachWithSeparator(", ", typeParameters.distinct, (tp: WeaveType) => {
          boundedTypeEmitter.generate(tp, builder, recursionDetector)
        })
        mergeEmitterMetadata(boundedTypeEmitter)
        builder.print(">")
      }
      builder.print(" = ")
      emittedTypes.+=((rootName, wtype))
      generate(wtype, builder, RecursionDetector[String]((id, _) => {
        id.name
      }))
      val types = typesToEmit.toMap
      typesToEmit = mutable.Map[String, WeaveTypeToEmit]()
      generateTypeCatalog(types, builder)
    }

    private def getPrefixFor(namespace: String): String = {
      if (namespaces.contains(namespace)) {
        namespaces(namespace)
      } else {
        val prefix = "ns" + namespaces.size
        namespaces.put(namespace, prefix)
        prefix
      }
    }

    private def generate(wtype: WeaveType, builder: StringCodeWriter, recursionDetector: RecursionDetector[String], insideKey: Boolean = false, depth: Int = 0, evaluationParenthesis: Boolean = false): String = {
      val weaveTypeMetadataConstraints = wtype.metadataConstraints()
      val weaveTypeMetadata = filterWeaveTypeMetadata(wtype)
      wtype match {
        case ObjectType(properties, close, ordered) =>
          if (properties.isEmpty && (!close && !ordered)) {
            builder.print("Object")
          } else {
            builder.print("{")
            if (ordered)
              builder.print("-")
            if (close)
              builder.print("|")
            builder.indent()
            //In case is one property key value pair we don't print the new lines so simple types keep simple to read
            if (inlineObject(properties)) {
              builder.printSpace()
              properties.zipWithIndex.foreach(indexProp => {
                generate(indexProp._1, builder, recursionDetector, insideKey, depth + 1)
              })
              builder.printSpace()
            } else {
              properties.zipWithIndex.foreach(indexProp => {
                if (indexProp._2 > 0) {
                  builder.printSpace(",")
                }
                builder.println()
                builder.printIndent()

                generate(indexProp._1, builder, recursionDetector, insideKey, depth + 1)
              })
            }
            builder.dedent()
            if (!inlineObject(properties)) {
              builder.println()
              builder.printIndent()
            }
            if (close)
              builder.print("|")
            if (ordered)
              builder.print("-")
            builder.print("}")
          }
        case KeyValuePairType(key, value, optional, repeated) =>
          generate(key, builder, recursionDetector, insideKey = true, depth)
          if (repeated) {
            builder.print("*")
          }
          if (optional) {
            builder.print("?")
          }
          builder.printSpace(":")
          generate(value, builder, recursionDetector, insideKey, depth + 1)
        case kt: KeyType =>
          if (insideKey) {
            kt.metadata()
              .zipWithIndex.foreach(annotationWithIndex => {
                val annotation = annotationWithIndex._1
                if (!objectKeyAnnotationsToEmit.contains(annotation.name)) {
                  objectKeyAnnotationsToEmit = objectKeyAnnotationsToEmit :+ annotation.name
                }
                if (annotationWithIndex._2 > 0) {
                  builder.printSpace()
                }
                emitAnnotationReference(builder, annotation)
              })
            generate(kt.name, builder, recursionDetector, insideKey, depth + 1)
            if (kt.attrs.nonEmpty) {
              builder.print(" @(")
              kt.attrs.zipWithIndex.foreach(attr => {
                if (attr._2 > 0) {
                  builder.printSpace(",")
                }
                generate(attr._1, builder, recursionDetector, insideKey, depth + 1)
              })
              builder.print(")")
            }
          } else {
            builder.print("Key")
          }
        case NameValuePairType(name, value, optional) =>
          generate(name, builder, recursionDetector, insideKey = true, depth)
          if (optional) {
            builder.print("?")
          }
          builder.print(": ")
          generate(value, builder, recursionDetector, insideKey, depth + 1)
        case nt @ NameType(name) =>
          name match {
            case Some(qName) =>
              nt.metadata().zipWithIndex
                .foreach(annotationWithIndex => {
                  val annotation = annotationWithIndex._1
                  if (!objectKeyAnnotationsToEmit.contains(annotation.name)) {
                    objectKeyAnnotationsToEmit = objectKeyAnnotationsToEmit :+ annotation.name
                  }
                  if (annotationWithIndex._2 > 0) {
                    builder.printSpace()
                  }
                  emitAnnotationReference(builder, annotation)
                })
              if (qName.ns.isDefined) {
                builder.print(getPrefixFor(qName.ns.get))
                builder.print("#")
              }
              if (StringEscapeHelper.keyRequiresQuotes(qName.name)) {
                builder.printQuoted(qName.name)
              } else {
                builder.print(qName.name)
              }
            case None => if (insideKey) builder.print("_") else builder.print("Name")
          }
        case ArrayType(of) =>
          builder.print("Array")
          printTypeParameter(of, builder, recursionDetector, insideKey, depth)
        case UnionType(of) =>
          if (evaluationParenthesis || weaveTypeMetadataConstraints.nonEmpty || weaveTypeMetadata.nonEmpty) {
            builder.print("(")
          }

          of.zipWithIndex.foreach {
            case (item, index) =>
              if (index > 0) {
                builder.print(" | ")
              }
              generate(item, builder, recursionDetector, insideKey, depth + 1)
          }
          if (evaluationParenthesis || weaveTypeMetadataConstraints.nonEmpty || weaveTypeMetadata.nonEmpty) {
            builder.print(")")
          }
        case IntersectionType(of) =>
          if (weaveTypeMetadataConstraints.nonEmpty || weaveTypeMetadata.nonEmpty) {
            builder.print("(")
          }
          of.zipWithIndex.foreach(item => {
            if (item._2 > 0) {
              builder.print(" & ")
            }
            generate(item._1, builder, recursionDetector, insideKey, depth + 1, evaluationParenthesis = true)
          })
          if (weaveTypeMetadataConstraints.nonEmpty || weaveTypeMetadata.nonEmpty) {
            builder.print(")")
          }
        case StringType(Some(value)) =>
          builder.print(StringEscapeHelper.escapeString(value))
        case StringType(_) =>
          builder.print("String")
        case AnyType() => builder.print("Any")
        case BooleanType(Some(value), _) =>
          builder.print(value.toString)
        case BooleanType(_, _) =>
          builder.print("Boolean")
        case NumberType(Some(value)) => builder.print(value)
        case NumberType(_)           => builder.print("Number")
        case RangeType()             => builder.print("Range")
        case RegexType()             => builder.print("Regex")
        case UriType(_)              => builder.print("Uri")
        case NullType()              => builder.print("Null")
        case DateTimeType()          => builder.print("DateTime")
        case LocalDateTimeType()     => builder.print("LocalDateTime")
        case LocalDateType()         => builder.print("Date")
        case LocalTimeType()         => builder.print("LocalTime")
        case TimeType()              => builder.print("Time")
        case TimeZoneType()          => builder.print("TimeZone")
        case PeriodType()            => builder.print("Period")
        case BinaryType()            => builder.print("Binary")
        case TypeParameter(name, tt, bt, _, _) =>
          builder.print(name)
          if (bt.isDefined) {
            builder.print(" :> ")
            generate(bt.get, builder, recursionDetector, insideKey, depth + 1)
          } else if (tt.isDefined) {
            builder.print(" <: ")
            generate(tt.get, builder, recursionDetector, insideKey, depth + 1)
          }
        case FunctionType(_, args, returnType, _, _, _) =>
          builder.print("(")
          args.zipWithIndex.foreach(item => {
            if (item._2 > 0) {
              builder.print(", ")
            }

            if (!isImplicitParameter(item._1.name)) {
              builder.print(item._1.name)
              if (item._1.optional) {
                builder.print("?")
              }
              if (!FunctionTypeHelper.isDynamicTypeParameter(item._1.wtype)) {
                builder.printSpace(":")
                generate(item._1.wtype, builder, recursionDetector, insideKey, depth + 1)
              }
            } else {
              generate(item._1.wtype, builder, recursionDetector, insideKey, depth + 1)
            }
          })
          builder.printSpace(")")
          builder.printSpace("->")
          generate(returnType, builder, recursionDetector, insideKey, depth + 1)
        case TypeType(t) =>
          builder.print("Type")
          printTypeParameter(t, builder, recursionDetector, insideKey, depth)
        case NothingType()        => builder.print("Nothing")
        case NamespaceType(_, _)  => builder.print("Namespace")
        case _: DynamicReturnType => builder.print(s"?")
        case tsrt: TypeSelectorType =>
          generate(tsrt.referencedType, builder, recursionDetector, insideKey, depth + 1)
          builder.print(".")
          if (StringEscapeHelper.keyRequiresQuotes(tsrt.refName)) {
            builder.printQuoted(tsrt.refName)
          } else {
            builder.print(tsrt.refName)
          }
        case rt: SimpleReferenceType =>
          val text = recursionDetector.resolve(
            rt,
            referencedType => {
              val builder = new StringCodeWriter()
              val refName = rt.refName.parent() match {
                case Some(NameIdentifier.CORE_MODULE) =>
                  rt.refName.localName().name
                case _ =>
                  referencedType match {
                    case _: TypeParameter => rt.refName.name
                    case _ =>
                      rt.referencedTypeDef() match {
                        case Some(referenceTypeDefinition) =>
                          var fqnRefName = referenceTypeDefinition._1.fullQualifiedName().replace("::", "_")
                          // TODO: this could be configured at part of the emit method in catalog API in order to describe the moduleSource from where types to emit
                          // where loaded. For the time being as types to be included in catalog will not reference to themselves [#1, ObjectT], [#2, ReferenceTypeT(#1)]
                          // this will work and we assume that anonymous is the root module source.
                          if (referenceTypeDefinition._1.parent().exists(n => n.equals(NameIdentifier.ANONYMOUS_NAME))) {
                            fqnRefName = referenceTypeDefinition._1.localName().name
                          }
                          if (!emittedTypes.contains(fqnRefName)) {
                            if (referenceTypeDefinition._3.isDefined) {
                              typesToEmit.+=((fqnRefName, WeaveTypeToEmit(
                                referenceTypeDefinition._2,
                                referenceTypeDefinition._3.get)))
                            } else {
                              typesToEmit.+=((fqnRefName, WeaveTypeToEmit.apply(rt.resolveType())))
                            }
                          }
                          fqnRefName
                        case None => throw new RuntimeException("Catalog could not be serialized due to a non resolved reference: "
                          + rt + ". This is probably a bug.")
                      }
                  }
              }
              builder.print(refName)

              rt.typeParams match {
                case Some(tps) =>
                  builder.print("<")
                  builder.startIgnoringNewLines()
                  builder.printForeachWithSeparator(", ", tps, (tp: WeaveType) => {
                    generate(tp, builder, recursionDetector, insideKey, depth + 1)
                  })
                  builder.endIgnoringNewLines()
                  builder.print(">")
                case None =>
              }
              builder.toString
            })
          builder.print(text)
        case _ => builder.print(wtype.getClass.getSimpleName)
      }
      if (weaveTypeMetadataConstraints.nonEmpty) {
        builder.print(" {")
        builder.printForeachWithSeparator(
          ",\n",
          weaveTypeMetadataConstraints,
          (m: Any) => {
            m match {
              case constraint: MetadataConstraint =>
                builder.printQuoted(constraint.name).printSpace(":").print(StringEscapeHelper.escapeString(constraint.value.toString))
              case _ => // Nothing to do
            }

          })
        builder.print("}")
      }

      if (weaveTypeMetadata.nonEmpty) {
        val emitter = new WeaveTypeMetadataEmitter()
        builder.print(" <~ { ")
        builder.printForeachWithSeparator(
          ",\n",
          weaveTypeMetadata,
          (m: Any) => {
            m match {
              case metadata: Metadata =>
                builder.print(emitter.toString(metadata))
              case _ => // Nothing to do
            }

          })
        builder.print("}")
      }
      builder.codeContent()
    }

    private def filterWeaveTypeMetadata(wtype: WeaveType): Seq[Metadata] = {
      if (FunctionTypeHelper.isDynamicTypeParameter(wtype) || isKeyType(wtype) || isNameType(wtype)) {
        Seq.empty
      } else {
        wtype.metadata()
      }
    }

    private def isKeyType(wtype: WeaveType): Boolean = {
      wtype match {
        case KeyType(_, _) => true
        case _             => false
      }
    }

    private def isNameType(wtype: WeaveType): Boolean = {
      wtype match {
        case NameType(_) => true
        case _           => false
      }
    }

    private def printTypeParameter(t: WeaveType, builder: StringCodeWriter, recursionDetector: RecursionDetector[String], insideKey: Boolean, depth: Int) = {
      builder.print("<")
      builder.startIgnoringNewLines()
      generate(t, builder, recursionDetector, insideKey, depth + 1)
      builder.endIgnoringNewLines()
      builder.print(">")
    }

    private def inlineObject(properties: Seq[KeyValuePairType]) = {
      properties.isEmpty || (properties.size == 1 && isSimpleType(properties.head.value))
    }

    private def isSimpleType(weaveType: DWType): Boolean = {
      !WeaveTypeTraverse.exists(weaveType.asInstanceOf[WeaveType], {
        case ot: ObjectType => ot.properties.size > 1
        case _              => false
      })
    }

    private def isImplicitParameter(variable: String): Boolean = {
      variable.matches("""\$+""")
    }

  }

  private def emitAnnotationReference(builder: StringCodeWriter, annotation: Metadata) = {
    builder.print("@").print(annotation.name.capitalize).print("(value=")
    val emitter = new WeaveTypeMetadataEmitter()
    builder.print(emitter.printMetadataValue(annotation.value))
    builder.print(") ")
  }
}
