package org.mule.weave.v2

import org.mule.weave.v2.completion.AutoCompletionService
import org.mule.weave.v2.editor.EmptyVirtualFileSystem
import org.mule.weave.v2.editor.VirtualFileSystem
import org.mule.weave.v2.inspector.Inspector
import org.mule.weave.v2.inspector.NoInspector
import org.mule.weave.v2.parser.DocumentParser
import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.Position
import org.mule.weave.v2.parser.phase.DependencyGraph
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.ParsingResult
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.parser.phase.ScopeGraphResult
import org.mule.weave.v2.parser.phase.TypeCheckingResult
import org.mule.weave.v2.scope.AstNavigator
import org.mule.weave.v2.scope.ScopesNavigator
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.ts.ConstraintSet
import org.mule.weave.v2.ts.FunctionType
import org.mule.weave.v2.ts.TypeGraph
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.utils.LockFactory
import org.mule.weave.v2.utils.ReadWriteLock

class WeaveEditorSupport(
  val resource: WeaveResource,
  val buildParsingContext: () => ParsingContext,
  private var mayBeCursorLocation: Option[Int] = None,
  val vfs: VirtualFileSystem = EmptyVirtualFileSystem,
  val dependencyGraph: DependencyGraph = new DependencyGraph(),
  val inspector: Inspector = NoInspector) {

  val withRecovery: ParsingResults = new ParsingResults()
  val withoutRecovery: ParsingResults = new ParsingResults()

  def offsetOf(line: Int, column: Int): Int = {
    resource.offsetOf(line, column)
  }

  def positionOf(offset: Int): Position = {
    resource.positionOf(offset)
  }

  def updateCursor(cursorLocation: Int): Unit = {
    this.mayBeCursorLocation = Some(cursorLocation)
  }

  def elementAtCursor(): Option[AstNode] = {
    mayBeCursorLocation.flatMap((cursorLocation) => {
      astNavigator().flatMap((navigator) => {
        navigator.nodeAt(cursorLocation)
      })
    })
  }

  def withNewContent(content: String): WeaveEditorSupport = {
    new WeaveEditorSupport(WeaveResource(resource.url(), content), buildParsingContext, mayBeCursorLocation)
  }

  def withNewContent(content: String, cursorLocation: Int): WeaveEditorSupport = {
    new WeaveEditorSupport(WeaveResource(resource.url(), content), buildParsingContext, Some(cursorLocation))
  }

  def getTypeCheckingForModule(moduleName: NameIdentifier): PhaseResult[TypeCheckingResult[ModuleNode]] = {
    buildParsingContext().getTypeCheckingForModule(moduleName)
  }

  def getScopeGraphForModule(moduleName: NameIdentifier): PhaseResult[ScopeGraphResult[ModuleNode]] = {
    buildParsingContext().getScopeGraphForModule(moduleName)
  }

  def resolveFullQualifiedName(ni: NameIdentifier): Option[NameIdentifier] = {
    astNavigator().map((astNavigator) => {
      //We map here how to the full canonical name of the module for example
      tryToParseModule(ni) match {
        case Some(_) => ni
        case _ if ni.nameElements().length == 1 => {
          val maybeDirective = astNavigator
            .importDirectives()
            .find((importDirective) => {
              importDirective.importedModule.aliasOrLocalName.name.equals(ni.nameElements().head)
            })
          maybeDirective.map(_.importedModule.elementName).getOrElse(ni)
        }
        case _ => ni
      }
    })
  }

  def tryToParseModule(ni: NameIdentifier): Option[PhaseResult[ParsingResult[ModuleNode]]] = {
    buildParsingContext().tryToParseModule(ni)
  }

  def content(): String = resource.content()

  def reverseTypeGraph(): Option[TypeGraph] = {
    val value = runReverseCheck()
    value.map(_.typeGraph).mayBeResult
  }

  def cursorLineNumber(): Int = {
    if (mayBeCursorLocation.isDefined) {
      resource.lineNumberOf(mayBeCursorLocation.get)
    } else {
      -1
    }
  }

  def cursorLocation(): Int = {
    mayBeCursorLocation.getOrElse(0)
  }

  def runReverseCheck(context: ParsingContext = buildParsingContext(), withRecovery: Boolean = true): PhaseResult[TypeCheckingResult[_ <: AstNode]] = {
    val callback = () => {
      val localContext: ParsingContext = context.withMessageCollector(new MessageCollector)
      val parser = createDocumentParser(calculateErrorTrace(withRecovery))
      val value = runTypeCheck(localContext, withRecovery)
      parser.reverseTypeCheck(value, localContext)
    }
    val result: PhaseResult[TypeCheckingResult[_ <: AstNode]] = {
      if (withRecovery) {
        this.withRecovery.reverseCheck(callback)
      } else {
        this.withoutRecovery.reverseCheck(callback)
      }
    }
    context.messageCollector.mergeWith(result.messages())
    result
  }

  private def calculateErrorTrace(withRecovery: Boolean): Int = {
    if (withRecovery) {
      0
    } else {
      2
    }
  }

  def runTypeCheck(context: ParsingContext = buildParsingContext(), withRecovery: Boolean = true): PhaseResult[TypeCheckingResult[_ <: AstNode]] = {
    val callback = () => {
      val localContext: ParsingContext = context.withMessageCollector(new MessageCollector)
      if (withRecovery) {
        localContext.addImplicitInput(AutoCompletionService.INSERTED_TEXT, None)
      }
      val parser = createDocumentParser(calculateErrorTrace(withRecovery))
      val scopeCheck = runScopePhase(localContext, withRecovery)
      parser.typeCheck(scopeCheck, localContext)
    }

    val result: PhaseResult[TypeCheckingResult[_ <: AstNode]] = {
      if (withRecovery) {
        this.withRecovery.typeCheck(callback)
      } else {
        this.withoutRecovery.typeCheck(callback)
      }
    }
    context.messageCollector.mergeWith(result.messages())
    result
  }

  def runScopePhase(context: ParsingContext = buildParsingContext(), withRecovery: Boolean = true): PhaseResult[ScopeGraphResult[_ <: AstNode]] = {
    val callback = () => {
      val localContext: ParsingContext = context.withMessageCollector(new MessageCollector)
      if (withRecovery) {
        localContext.addImplicitInput(AutoCompletionService.INSERTED_TEXT, None)
      }
      val parser: DocumentParser = createDocumentParser(calculateErrorTrace(withRecovery))
      val parse: PhaseResult[ParsingResult[_ <: AstNode]] = runCanonical(localContext, withRecovery)
      val scopeCheck = parser.scopeCheck(parse, localContext)
      scopeCheck
    }
    val result: PhaseResult[ScopeGraphResult[_ <: AstNode]] = {
      if (withRecovery) {
        this.withRecovery.scopePhase(callback)
      } else {
        this.withoutRecovery.scopePhase(callback)
      }
    }
    context.messageCollector.mergeWith(result.messages())
    result
  }

  def runCanonical(context: ParsingContext = buildParsingContext(), withRecovery: Boolean = true): PhaseResult[ParsingResult[_ <: AstNode]] = {
    val callback = () => {
      val localContext: ParsingContext = context.withMessageCollector(new MessageCollector)
      if (withRecovery) {
        localContext.addImplicitInput(AutoCompletionService.INSERTED_TEXT, None)
      }
      val parser = createDocumentParser(calculateErrorTrace(withRecovery))
      val parseResult = if (withRecovery) runParseWithRecovery(localContext) else runParse(localContext)
      parser.canonical(parseResult, localContext)
    }

    val result: PhaseResult[ParsingResult[_ <: AstNode]] = {
      if (withRecovery) {
        this.withRecovery.canonicalPhase(callback)
      } else {
        this.withoutRecovery.canonicalPhase(callback)
      }
    }
    context.messageCollector.mergeWith(result.messages())
    result
  }

  def runParseWithRecovery(context: ParsingContext = buildParsingContext()): PhaseResult[ParsingResult[AstNode]] = {
    val callback: () => PhaseResult[ParsingResult[AstNode]] = () => {
      val localContext: ParsingContext = context.withMessageCollector(new MessageCollector)
      val parser = createDocumentParser(calculateErrorTrace(true))
      parser.parseWithRecovery(resource, localContext, mayBeCursorLocation)
    }
    val result: PhaseResult[ParsingResult[AstNode]] = withRecovery.parsePhase(callback)
    context.messageCollector.mergeWith(result.messages())
    dependencyGraph.loadDependenciesFrom(context.nameIdentifier, result)
    result
  }

  def runParse(context: ParsingContext = buildParsingContext()): PhaseResult[ParsingResult[AstNode]] = {
    val callback: () => PhaseResult[ParsingResult[AstNode]] = () => {
      val localContext: ParsingContext = context.withMessageCollector(new MessageCollector)
      val parser = createDocumentParser(calculateErrorTrace(false))
      parser.parse(resource, localContext)
    }
    val value: PhaseResult[ParsingResult[AstNode]] = withoutRecovery.parsePhase(callback)
    context.messageCollector.mergeWith(value.messages())
    dependencyGraph.loadDependenciesFrom(context.nameIdentifier, value)
    value
  }

  def astDocument(): Option[AstNode] = {
    runParse().mayBeResult.map(_.astNode)
  }

  def astNavigator(withRecovery: Boolean = true): Option[AstNavigator] = {
    if (withRecovery) {
      this.withRecovery.astNavigator(() => {
        val value: PhaseResult[ParsingResult[_ <: AstNode]] = runCanonical()
        if (value.hasResult()) {
          val astNode = value.getResult().astNode
          Option(new AstNavigator(astNode))
        } else {
          None
        }
      })
    } else {
      this.withoutRecovery.astNavigator(() => {
        val value: PhaseResult[ParsingResult[AstNode]] = runParse()
        if (value.hasResult()) {
          val astNode = value.getResult().astNode
          Option(new AstNavigator(astNode))
        } else {
          None
        }
      })
    }
  }

  def scopeGraph(): Option[ScopesNavigator] = {
    val scopePhase: PhaseResult[ScopeGraphResult[_ <: AstNode]] = runScopePhase()
    if (scopePhase.hasResult()) {
      Some(scopePhase.getResult().scope)
    } else {
      None
    }
  }

  def typeGraph(): Option[TypeGraph] = {
    val value: PhaseResult[TypeCheckingResult[_ <: AstNode]] = runTypeCheck()
    if (value.hasResult()) {
      Some(value.getResult().typeGraph)
    } else {
      None
    }
  }

  private def createDocumentParser(errorTrace: Int): DocumentParser = {
    DocumentParser(errorTrace, inspector)
  }

}

case class CalculatedConstraints(ft: FunctionType, args: Array[WeaveType], constraintSet: ConstraintSet)

object WeaveEditorSupport {

  def apply(resource: WeaveResource, buildParsingContext: () => ParsingContext, cursorLocation: Int, virtualFileSystem: VirtualFileSystem): WeaveEditorSupport = {
    WeaveEditorSupport(resource, buildParsingContext, cursorLocation, virtualFileSystem, new DependencyGraph)
  }

  def apply(resource: WeaveResource, buildParsingContext: () => ParsingContext, cursorLocation: Int, virtualFileSystem: VirtualFileSystem, dependencyGraph: DependencyGraph): WeaveEditorSupport = {
    new WeaveEditorSupport(resource, buildParsingContext, Some(cursorLocation), virtualFileSystem, dependencyGraph)
  }

  def apply(resource: WeaveResource, buildParsingContext: () => ParsingContext, virtualFileSystem: VirtualFileSystem): WeaveEditorSupport = {
    WeaveEditorSupport(resource, buildParsingContext, virtualFileSystem, new DependencyGraph)
  }

  def apply(resource: WeaveResource, buildParsingContext: () => ParsingContext, virtualFileSystem: VirtualFileSystem, dependencyGraph: DependencyGraph): WeaveEditorSupport = {
    new WeaveEditorSupport(resource, buildParsingContext, None, virtualFileSystem, dependencyGraph)
  }

  def apply(resource: WeaveResource, buildParsingContext: () => ParsingContext, virtualFileSystem: VirtualFileSystem, dependencyGraph: DependencyGraph, inspector: Inspector): WeaveEditorSupport = {
    new WeaveEditorSupport(resource, buildParsingContext, None, virtualFileSystem, dependencyGraph, inspector)
  }
}

class ParsingResults() {

  private var reverseCheck: PhaseResult[TypeCheckingResult[_ <: AstNode]] = null
  private var typeChecker: PhaseResult[TypeCheckingResult[_ <: AstNode]] = null
  private var scopePhase: PhaseResult[ScopeGraphResult[_ <: AstNode]] = null
  private var canonicalPhase: PhaseResult[ParsingResult[_ <: AstNode]] = null
  private var parsePhase: PhaseResult[ParsingResult[AstNode]] = null
  private val readWriteLock: ReadWriteLock = LockFactory.createReadWriteLock()
  private var astNavigator: Option[AstNavigator] = None

  def parsePhase(provider: () => PhaseResult[ParsingResult[AstNode]]): PhaseResult[ParsingResult[AstNode]] = {
    val result: PhaseResult[ParsingResult[AstNode]] = readWriteLock.readLock({
      parsePhase
    })
    if (result == null) {
      readWriteLock.writeLock({
        if (parsePhase == null) {
          parsePhase = provider()
        }
        parsePhase
      })
    } else {
      result
    }
  }

  def canonicalPhase(provider: () => PhaseResult[ParsingResult[_ <: AstNode]]): PhaseResult[ParsingResult[_ <: AstNode]] = {
    val result = readWriteLock.readLock({
      canonicalPhase
    })
    if (result == null) {
      readWriteLock.writeLock({
        if (canonicalPhase == null) {
          canonicalPhase = provider()
        }
        canonicalPhase
      })
    } else {
      result
    }
  }

  def scopePhase(provider: () => PhaseResult[ScopeGraphResult[_ <: AstNode]]): PhaseResult[ScopeGraphResult[_ <: AstNode]] = {
    val result = readWriteLock.readLock({
      scopePhase
    })
    if (result == null) {
      readWriteLock.writeLock({
        if (scopePhase == null) {
          scopePhase = provider()
        }
        scopePhase
      })
    } else {
      result
    }
  }

  def typeCheck(provider: () => PhaseResult[TypeCheckingResult[_ <: AstNode]]): PhaseResult[TypeCheckingResult[_ <: AstNode]] = {
    val result = readWriteLock.readLock({
      typeChecker
    })
    if (result == null) {
      readWriteLock.writeLock({
        if (typeChecker == null) {
          typeChecker = provider()
        }
        typeChecker
      })
    } else {
      result
    }
  }

  def reverseCheck(provider: () => PhaseResult[TypeCheckingResult[_ <: AstNode]]): PhaseResult[TypeCheckingResult[_ <: AstNode]] = {
    val result = readWriteLock.readLock({
      reverseCheck
    })
    if (result == null) {
      readWriteLock.writeLock({
        if (reverseCheck == null) {
          reverseCheck = provider()
        }
        reverseCheck
      })
    } else {
      result
    }
  }

  def astNavigator(provider: () => Option[AstNavigator]): Option[AstNavigator] = {
    val result = readWriteLock.readLock({
      astNavigator
    })
    if (result.isEmpty) {
      readWriteLock.writeLock({
        if (astNavigator.isEmpty) {
          astNavigator = provider()
        }
        astNavigator
      })
    } else {
      result
    }

  }
}
