package org.mule.weave.v2.mapping

import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.ts.WeaveType

import scala.collection.mutable.ListBuffer

/**
  * A mapping models a transformation from one structure to another one.
  * It's comprised of many assignments generated from interactions in the UI (drag & drop, etc..) or other sources.
  */
case class DataMapping(parentMapping: Option[DataMapping], source: NamePathElement, target: NamePathElement) {
  private val childMappingList = new ListBuffer[DataMapping]
  private val fieldAssignmentList = new ListBuffer[FieldAssignment]
  private val expressionAssignmentList = new ListBuffer[ExpressionAssignment]

  def childMappings(): Array[DataMapping] = {
    this.childMappingList.toArray
  }

  def fieldAssignments(): Array[FieldAssignment] = {
    this.fieldAssignmentList.toArray
  }

  def expressionAssignments(): Array[ExpressionAssignment] = {
    this.expressionAssignmentList.toArray
  }

  def addMapping(innerMapping: DataMapping): Unit = {
    childMappingList += innerMapping
  }

  def addArrow(source: NamePathElement, target: NamePathElement, sourceType: Option[WeaveType], targetType: Option[WeaveType]): Unit = {
    fieldAssignmentList += FieldAssignment(source, target, sourceType, targetType)
  }

  def addExpression(target: NamePathElement, expressionNode: AstNode, sourceType: Option[WeaveType], targetType: Option[WeaveType]): Unit = {
    expressionAssignmentList += ExpressionAssignment(target, expressionNode, sourceType, targetType)
  }

  def findInnerMapping(source: NamePathElement, target: NamePathElement): Option[DataMapping] = {
    childMappingList.find(m =>
      m.target == target)
  }

  def findOrCreateInnerMapping(source: NamePathElement, target: NamePathElement): DataMapping = {
    findInnerMapping(source, target) match {
      case Some(innerMapping) => innerMapping
      case None =>
        val innerMapping = DataMapping(Some(this), source, target)
        addMapping(innerMapping)
        innerMapping
    }
  }

  /**
    * Returns the target path, relative to the target of the parent mapping
    */
  def relativeTargetPath(): Array[NamePathElement] = {
    parentMapping match {
      case Some(parent) =>
        target.relativePathFrom(parent.target)
      case None =>
        target.relativePath()
    }
  }

  def isRootMapping(): Boolean = {
    source.isRoot() && target.isRoot()
  }

  private lazy val indent: Int = {
    parentMapping match {
      case Some(parent) => parent.indent + 2
      case None         => 0
    }
  }

  def indentStr(n: Int): String = " " * n

  def getMappingIndex: Int = {
    parentMapping
      .map((parent) => parent.getMappingIndex)
      .getOrElse(-2) + 1
  }

  override def toString: String = {
    def printSeq(arr: Seq[Object]) = {
      if (arr.nonEmpty) arr.mkString("[", ", ", "]") else "[]"
    }

    val indentStr = " " * indent
    s"""{
       |$indentStr  "mapping": "$source -> $target"
       |$indentStr  "fieldAssignments": ${printSeq(fieldAssignmentList)},
       |$indentStr  "expressionAssignments": ${printSeq(expressionAssignmentList)},
       |$indentStr  "childMappings": ${printSeq(childMappingList)}
       |$indentStr}""".stripMargin
  }

}
