package org.mule.weave.v2.parser

import java.io.File
import java.io.InputStream

import org.parboiled2.ParserInput.DefaultParserInput

import scala.collection.mutable.ArrayBuffer
import scala.io.Source
import scala.io.Source.fromInputStream
import scala.io.Source.fromFile

class SafeStringBasedParserInput(val content: String) extends DefaultParserInput {
  lazy val lineIndexes = {
    val indexes = new ArrayBuffer[Int]()
    indexes += -1 //line numbers start at 1
    indexes += 0 //line 1 has index 0
    var newLineIndex = content.indexOf('\n')
    while (newLineIndex >= 0) {
      indexes += newLineIndex + 1
      newLineIndex = content.indexOf('\n', newLineIndex + 1)
    }
    indexes
  }

  def getLineAndColumn(implicit index: Int): (Int, Int) = {
    if (index < 0 || index > content.length) {
      throw new IndexOutOfBoundsException(s"Invalid index: $index")
    }
    val line = lineForIndex(index)
    val column = index - lineIndexes(line) + 1 // column numbers start at 1
    (line, column)
  }

  private def lineForIndex(index: Int): Int = {
    def lineForIndexRec(minLine: Int, maxLine: Int): Int = {
      if (maxLine - minLine <= 1) {
        val line = if (index >= lineIndexes(maxLine)) maxLine else minLine
        return line
      }
      val midLine = (maxLine + minLine) / 2
      val midIndex = lineIndexes(midLine)
      if (index < midIndex) {
        lineForIndexRec(minLine, midLine)
      } else if (index > midIndex) {
        lineForIndexRec(midLine, maxLine)
      } else {
        midLine // the index is exactly the line start
      }
    }

    val lastLine = lineIndexes.length - 1
    lineForIndexRec(1, lastLine)
  }

  override def getLine(line: Int): String = {
    val lineIndex = getLineIndex(line)
    val nextLineIndex = getLineIndex(line + 1)
    if (lineIndex == -1) {
      //line out of range
      ""
    } else if (nextLineIndex == -1) {
      //it's the last line
      content.substring(lineIndex)
    } else {
      content.substring(lineIndex, nextLineIndex)
    }
  }

  private def getLineIndex(line: Int): Int = {
    val maxIndexedLine = lineIndexes.length - 1
    if (maxIndexedLine >= line && line > 0) {
      lineIndexes(line)
    } else {
      -1
    }
  }

  def charAt(ix: Int) = {
    if (ix >= content.length) {
      org.parboiled2.EOI
    } else {
      content.charAt(ix)
    }
  }

  override def toString: String = content

  def length = content.length

  def sliceString(start: Int, end: Int) = {
    content.substring(start, safeIndex(end))
  }

  //noinspection ScalaUselessExpression
  def sliceCharArray(start: Int, end: Int) = {
    val chars = new Array[Char](end - start)
    content.getChars(start, end, chars, 0)
    chars
  }

  def safeIndex(ix: Int): Int = {
    if (ix > content.length) {
      content.length
    } else {
      ix
    }
  }

  def canEqual(other: Any): Boolean = other.isInstanceOf[SafeStringBasedParserInput]

  override def equals(other: Any): Boolean = other match {
    case that: SafeStringBasedParserInput =>
      (that canEqual this) &&
        content == that.content
    case _ => false
  }

  override def hashCode(): Int = {
    val state = Seq(content)
    state.map(_.hashCode()).foldLeft(0)((a, b) => 31 * a + b)
  }
}

object SafeStringBasedParserInput {
  def apply(content: String) = new SafeStringBasedParserInput(content)

  def apply(content: InputStream) = {
    val source = fromInputStream(content)
    try {
      new SafeStringBasedParserInput(source.mkString)
    } finally {
      source.close()
    }
  }

  def apply(content: Source) = {
    try {
      new SafeStringBasedParserInput(content.mkString)
    } finally {
      content.close()
    }
  }

  def apply(content: File) = {
    val source = fromFile(content)
    try {
      new SafeStringBasedParserInput(source.mkString)
    } finally {
      source.close()
    }
  }
}
