package org.mule.weave.v2.codegen

import org.mule.weave.v2.annotations.WeaveApi
import org.mule.weave.v2.grammar._
import org.mule.weave.v2.grammar.literals.TypeLiteral
import org.mule.weave.v2.parser.ErrorAstNode
import org.mule.weave.v2.parser.annotation.BooleanNotTypeAnnotation
import org.mule.weave.v2.parser.annotation.BracketSelectorAnnotation
import org.mule.weave.v2.parser.annotation.CustomInterpolationAnnotation
import org.mule.weave.v2.parser.annotation.EnclosedMarkAnnotation
import org.mule.weave.v2.parser.annotation.InfixNotationFunctionCallAnnotation
import org.mule.weave.v2.parser.annotation.InterpolationExpressionAnnotation
import org.mule.weave.v2.parser.annotation.NotType
import org.mule.weave.v2.parser.annotation.SingleKeyValuePairNodeAnnotation
import org.mule.weave.v2.parser.annotation.TrailingCommaAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.CommentNode
import org.mule.weave.v2.parser.ast.CommentType
import org.mule.weave.v2.parser.ast.ContainerAstNode
import org.mule.weave.v2.parser.ast.LiteralValueAstNode
import org.mule.weave.v2.parser.ast.UndefinedExpressionNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationArgumentNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.conditional.DefaultNode
import org.mule.weave.v2.parser.ast.conditional.IfNode
import org.mule.weave.v2.parser.ast.conditional.UnlessNode
import org.mule.weave.v2.parser.ast.functions._
import org.mule.weave.v2.parser.ast.header.HeaderNode
import org.mule.weave.v2.parser.ast.header.directives._
import org.mule.weave.v2.parser.ast.logical.AndNode
import org.mule.weave.v2.parser.ast.logical.OrNode
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.operators._
import org.mule.weave.v2.parser.ast.patterns._
import org.mule.weave.v2.parser.ast.selectors.ExistsSelectorNode
import org.mule.weave.v2.parser.ast.selectors.NullSafeNode
import org.mule.weave.v2.parser.ast.selectors.NullUnSafeNode
import org.mule.weave.v2.parser.ast.structure._
import org.mule.weave.v2.parser.ast.structure.schema.SchemaNode
import org.mule.weave.v2.parser.ast.structure.schema.SchemaPropertyNode
import org.mule.weave.v2.parser.ast.types._
import org.mule.weave.v2.parser.ast.updates.ArrayIndexUpdateSelectorNode
import org.mule.weave.v2.parser.ast.updates.AttributeNameUpdateSelectorNode
import org.mule.weave.v2.parser.ast.updates.FieldNameUpdateSelectorNode
import org.mule.weave.v2.parser.ast.updates.MultiFieldNameUpdateSelectorNode
import org.mule.weave.v2.parser.ast.updates.UpdateExpressionNode
import org.mule.weave.v2.parser.ast.updates.UpdateExpressionsNode
import org.mule.weave.v2.parser.ast.updates.UpdateNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.utils.StringEscapeHelper

class CodeGenerator(printer: CodeWriter, settings: CodeGeneratorSettings = CodeGeneratorSettings()) {
  private var insideParameterList = false

  def generatePattern(pexn: PatternExpressionNode): Unit = {
    pexn match {
      case RegexPatternNode(pattern, name, function) => {
        if (!isImplicitParameter(name)) {
          generate(name)
          printer.printSpace()
        }
        printer.printSpace("matches")
        generate(pattern)
        printer.printSpace(" ->")
        generate(function)

      }
      case EmptyArrayPatternNode(onMatch) => {
        printer.print("[]")
        printer.printSpace(" ->")
        generate(onMatch)
      }
      case EmptyObjectPatternNode(onMatch) => {
        printer.print("{}")
        printer.printSpace(" ->")
        generate(onMatch)
      }
      case DeconstructArrayPatternNode(head, tail, onMatch) => {
        printer.print("[")
        generate(head)
        printer.print(" ~ ")
        generate(tail)
        printer.print("]")
        printer.printSpace(" ->")
        generate(onMatch)
      }
      case DeconstructObjectPatternNode(headKey, headValue, tail, onMatch) => {
        printer.print("{")
        generate(headKey)
        printer.printSpace(":")
        generate(headValue)
        printer.print(" ~ ")
        generate(tail)
        printer.print("}")
        printer.printSpace(" ->")
        generate(onMatch)
      }
      case ExpressionPatternNode(pattern, name, function) => {
        generate(name)
        printer.print(" if ")
        generate(pattern)
        printer.printSpace(" ->")
        generate(function)
      }
      case TypePatternNode(pattern, name, function) => {
        if (!isImplicitParameter(name)) {
          generate(name)
          printer.printSpace()
        }
        printer.printSpace("is")
        generate(pattern)
        printer.printSpace(" ->")
        generate(function)
      }
      case LiteralPatternNode(pattern, name, function) => {
        if (!isImplicitParameter(name)) {
          generate(name)
          printer.print(": ")
        }
        generate(pattern)
        printer.printSpace(" ->")
        generate(function)
      }
      case DefaultPatternNode(value, name) => {
        printer.printSpace("else")
        if (!isImplicitParameter(name)) {
          generate(name)
        }
        printer.printSpace("->")
        generate(value)
      }
    }
  }

  def generate(node: AstNode): Unit = {

    val comments: Seq[CommentNode] = node.comments
    //Comments divided by the ones above and the ones below
    val commentsHeadAndTail: (Seq[CommentNode], Seq[CommentNode]) = comments
      .sortBy(_.location().startPosition.index)
      .partition((comment) => comment.location().endPosition.index <= node.location().startPosition.index)

    commentsHeadAndTail._1.foreach((comment) => {
      //This is to keep parenthesis and comments keep their relationship
      //Comments outside parent will remain outside
      //Parents inside will keep inside
      if (comment.location().startPosition.index <= node.semanticStartPosition().index) {
        if (printer.hasNewLineListener()) {
          printer.println()
        }
        printComment(comment)
        printer.println()
      }
    })

    val encloseParenthesis = node.annotationsBy(classOf[EnclosedMarkAnnotation])
    if (encloseParenthesis.nonEmpty) {
      encloseParenthesis.foreach(_ => printer.print("("))
    }

    commentsHeadAndTail._1.foreach((comment) => {
      if (comment.location().startPosition.index > node.semanticStartPosition().index) {
        if (printer.hasNewLineListener()) {
          printer.println()
        }
        printComment(comment)
        printer.println()
      }
    })

    commentsHeadAndTail._2.zipWithIndex.foreach((comment) => {
      printer.onNewLine((printer) => {
        if (printer.nonEmpty()) {
          printer.printSpace()
        }
        printComment(comment._1)
      })
    })

    //The or so that it can work with the cannonical model
    if (AstNodeHelper.notInjectedNode(node) || node.isInstanceOf[FunctionNode] || AstNodeHelper.isMetadataAdditionNode(node)) {
      doGenerateCode(node)
    }

    if (encloseParenthesis.nonEmpty) {
      encloseParenthesis.foreach(_ => printer.print(")"))
    }
  }

  private def doGenerateCode(node: AstNode) = {
    node match {
      case UndefinedExpressionNode() => printer.print("???")
      case ModuleNode(name, elements) => {
        if (printer.hasNewLineListener()) {
          printer.println()
        }
        elements.foreach((node) => {
          generate(node)
          printer.println()
        })
      }
      case AndNode(lhs, rhs, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        generate(lhs)
        printer.print(" and ")
        generate(rhs)
      }
      case OrNode(lhs, rhs, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        generate(lhs)
        printer.print(" or ")
        generate(rhs)
      }
      case _: ErrorAstNode =>
      case PatternMatcherNode(lhs, patterns, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        generate(lhs)
        printer.printSpace(" match")
        generate(patterns)
      }
      case UpdateNode(expression, matchers, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        generate(expression)
        printer.printSpace(" update")
        generate(matchers)
      }
      case UpdateExpressionsNode(expressions) => {
        printer.println("{")
        printer.indent()
        expressions.foreach(e => {
          printer.printIndent()
          generate(e)
          printer.println()
        })
        printer.dedent()
        printer.printIndent()
        printer.println("}")
      }
      case UpdateExpressionNode(name, index, selector, forceCreate, condition, updateExpression) => {
        printer.printSpace("case")
        if (!isImplicitParameter(name)) {
          if (!isImplicitParameter(index)) {
            printer.print("(")
            generate(name)
            printer.printSpace(",")
            generate(index)
            printer.print(")")
          } else {
            generate(name)
          }
          printer.printSpace(" at")
        }
        generate(selector)

        if (forceCreate) {
          printer.print("!")
        }

        if (condition.isDefined) {
          printer.printSpace().print("if").print("(")
          generate(condition.get)
          printer.print(")")
        }
        printer.printSpace(" ->")
        generate(updateExpression)
      }
      case FieldNameUpdateSelectorNode(name, child) => {
        printer.print(".")
        generate(name)
        if (child.isDefined) {
          generate(child.get)
        }
      }
      case AttributeNameUpdateSelectorNode(name, child) => {
        printer.print(".@")
        generate(name)
        if (child.isDefined) {
          generate(child.get)
        }
      }
      case MultiFieldNameUpdateSelectorNode(name, child) => {
        printer.print(".*")
        generate(name)
        if (child.isDefined) {
          generate(child.get)
        }
      }
      case ArrayIndexUpdateSelectorNode(name, child) => {
        printer.print("[")
        generate(name)
        if (child.isDefined) {
          generate(child.get)
        }
        printer.print("]")
      }
      case pexn: PatternExpressionNode => {
        generatePattern(pexn)
      }

      case PatternExpressionsNode(patterns) => {
        printer.print("{")
        printer.indent()
        patterns.zipWithIndex.foreach((pattern) => {
          printer.println()
          if (!pattern._1.isInstanceOf[DefaultPatternNode]) {
            printer.printSpace("case")
          }
          generate(pattern._1)
        })
        printer.dedent()
        printer.println()
        printer.print("}")
      }
      case DirectiveOptionName(name) =>
        printer.print(name)
      case FunctionParameter(variable, defaultValue, wtype, annotations) => {
        printAnnotations(annotations, " ")
        generate(variable)
        generateTypeDeclaration(wtype)
        if (defaultValue.isDefined) {
          printer.printSpace(" =")
          generate(defaultValue.get)
        }
      }
      case VariableReferenceNode(variable, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        generate(variable)
      }
      case sin @ StringInterpolationNode(elements) => {
        val quoteChar = sin.quotedBy().getOrElse('"')
        printer.print(quoteChar)
        stringInterpolation(elements)
        printer.print(quoteChar)
      }
      case NameIdentifier(name, loader) => {
        loader.foreach((loader) => printer.print(loader).print("!"))
        printer.print(s"${name}")
      }
      case ConditionalNode(value, cond) => {
        printer.print("(")
        generate(value)
        printer.printSpace(")")
        printer.printSpace("if")
        printer.print("(")
        generate(cond)
        printer.print(")")
      }
      case DynamicKeyNode(keyName, attr) => {
        generate(keyName)
        if (attr.isDefined) {
          generate(attr.get)
        }
        printer.printSpace(":")
      }
      case NullSafeNode(selector, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        generate(selector)
      }
      case NullUnSafeNode(selector, codeAnnotations) =>
        printAnnotations(codeAnnotations)
        generate(selector)
        printer.print("!")
      case ArrayNode(elements, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        printer.print("[")
        printer.indent()
        val itemsNewLine = elements.nonEmpty && !elements.head.isInstanceOf[LiteralValueAstNode]
        elements.zipWithIndex.foreach((item) => {
          if (item._2 > 0) {
            if (itemsNewLine)
              printer.print(",")
            else
              printer.printSpace(",")
          }
          if (itemsNewLine) {
            printer.println()
          }
          generate(item._1)
        })
        node.annotation(classOf[TrailingCommaAnnotation]).foreach(annotation => {
          printer.print(",")
        })
        printer.dedent()

        if (itemsNewLine) {
          printer.println()
        }
        printer.print("]")
      }
      case ExistsSelectorNode(selectable) => {
        generate(selectable)
        printer.print("?")
      }
      case AttributesNode(attrs) => {
        printer.print(" @(")
        attrs.zipWithIndex.foreach((atr) => {
          if (atr._2 > 0) {
            printer.print(", ")
          }
          generate(atr._1)
        })
        printer.print(")")
      }
      case ObjectNode(elements, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        if (elements.isEmpty) {
          printer.print("{}")
        } else {
          val notSingleKeyValuePair = node.annotation(classOf[SingleKeyValuePairNodeAnnotation]).isEmpty
          if (notSingleKeyValuePair) {
            printer.print("{")
          }
          printer.indent()
          elements.zipWithIndex.foreach((item) => {
            if (item._2 > 0) {
              printer.println(",")
            } else {
              printer.println()
            }
            generate(item._1)
          })
          printer.dedent()
          if (notSingleKeyValuePair) {
            printer.println()
            printer.print("}")
          } else {
            printer.println()
          }
        }
      }
      case NameValuePairNode(key, value, cond) => {
        if (cond.isDefined) {
          printer.print("(")
        }
        generate(key)
        printer.printSpace(":")
        generate(value)
        if (cond.isDefined) {
          printer.printSpace(")")
          printer.printSpace("if")
          printer.print("(")
          generate(cond.get)
          printer.print(")")
        }
      }
      case NamespaceNode(prefix) => printer.print(prefix.name)
      case KeyValuePairNode(key, value, cond) => {
        if (cond.isDefined) {
          printer.print("(")
        }
        generate(key)
        if (isConditionalNode(value)) {
          printer.indent()
          printer.println()
          generate(value)
          printer.dedent()
        } else {
          generate(value)
        }
        if (cond.isDefined) {
          printer.printSpace(")")
          printer.printSpace("if")
          printer.print("(")
          generate(cond.get)
          printer.print(")")
        }
      }
      case NameNode(keyName, ns, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        if (ns.isDefined) {
          generate(ns.get)
          printer.print("#")
        }
        generate(keyName)
      }
      case DynamicNameNode(keyName) => {
        generate(keyName)
      }
      case KeyNode(keyName, ns, attr, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        if (ns.isDefined) {
          generate(ns.get)
          printer.print("#")
        }
        generate(keyName)
        if (attr.isDefined) {
          generate(attr.get)
        }
        printer.printSpace(":")
      }
      case UnlessNode(ifExpr, condition, elseExpr, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        printer.print("unless (")
        generate(condition)
        printer.println(")")

        if (isDoBlock(elseExpr)) {
          printer.printSpace()
          generate(ifExpr)
          printer.printSpace()
        } else {
          printer.indent()
          printer.println()
          generate(ifExpr)
          printer.dedent()
          printer.println()
        }

        printer.print("else")

        if (isConditionalNode(elseExpr) || isDoBlock(elseExpr)) {
          printer.printSpace()
          generate(elseExpr)
        } else {
          printer.indent()
          printer.println()
          generate(elseExpr)
          printer.dedent()
        }
      }
      case IfNode(ifExpr, condition, elseExpr, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        printer.print("if (")
        generate(condition)
        printer.print(")")

        if (isDoBlock(elseExpr)) {
          printer.printSpace()
          generate(ifExpr)
          printer.printSpace()
        } else {
          printer.indent()
          printer.println()
          generate(ifExpr)
          printer.dedent()
          printer.println()
        }

        printer.print("else")

        if (isConditionalNode(elseExpr) || isDoBlock(elseExpr)) {
          printer.printSpace()
          generate(elseExpr)
        } else {
          printer.indent()
          printer.println()
          generate(elseExpr)
          printer.dedent()
        }
      }
      case DefaultNode(lhs, rhs, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        generate(lhs)
        printer.printSpace(" default")
        generate(rhs)
      }
      case fcn @ FunctionCallNode(function, args, typeParameters, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        val binaryFunction = args.args.size == 2 && function.isInstanceOf[VariableReferenceNode]
        val annotationIsPresent = fcn.annotation(classOf[InfixNotationFunctionCallAnnotation]).isDefined
        val useInfix = (annotationIsPresent && settings.infixNotation == InfixOptions.KEEP) || settings.infixNotation == InfixOptions.ALWAYS
        val isInfixFunctionCall: Boolean = useInfix && binaryFunction
        if (isInfixFunctionCall) {
          generate(args.args.head)
          printer.printSpace()
          generate(function)
          typeParameters match {
            case Some(value) => generate(value)
            case None        =>
          }
          printer.printSpace()
          generate(args.args.last)
        } else {
          val isCustomInterpolation: Boolean = fcn.annotation(classOf[CustomInterpolationAnnotation]).isDefined
          if (isCustomInterpolation) {
            val stringsNode = args.args.head.asInstanceOf[ArrayNode]
            val valuesNode = args.args(1).asInstanceOf[ArrayNode]
            val strings = stringsNode.elements
            val dynExpressions = valuesNode.elements
            generate(function)
            typeParameters match {
              case Some(value) => generate(value)
              case None        =>
            }
            printer.printSpace()
            printer.print("`")
            val literal = strings.zipWithIndex.flatMap((stringExpr) => {
              if (dynExpressions.size > stringExpr._2)
                Seq(stringExpr._1, dynExpressions(stringExpr._2))
              else
                Seq(stringExpr._1)
            })
            stringInterpolation(literal)
            printer.print("`")
          } else {
            generate(function)
            typeParameters match {
              case Some(value) => {
                generate(value)
              }
              case None =>
            }
            generate(args)
          }
        }
      }
      case DocumentNode(header, root) => {
        generate(header)
        generate(root)
      }
      case HeaderNode(directiveNodes) => {
        val directives = if (settings.orderDirectives) {
          directiveNodes.sortBy({
            case _: VersionDirective        => 0
            case _: ImportDirective         => 10
            case _: AnnotationDirectiveNode => 15
            case _: InputDirective          => 17
            case _: NamespaceDirective      => 20
            case _: TypeDirective           => 30
            case _: VarDirective            => 40
            case _: FunctionDirectiveNode   => 40
            case _: OutputDirective         => 60
            case _                          => 33

          })
        } else {
          directiveNodes
        }
        if (directives.nonEmpty) {
          if (printer.hasNewLineListener()) {
            printer.println()
          }
          var realDirectives = directives
            .filter((node) => AstNodeHelper.notInjectedNode(node))

          if (settings.alwaysInsertVersion && !realDirectives.exists(_.isInstanceOf[VersionDirective])) {
            realDirectives = Seq(new VersionDirective()) ++ realDirectives
          }

          if (realDirectives.nonEmpty) {
            realDirectives
              .foreach((directive) => {
                val requiresNewLine = directive.hasWeaveDoc || (settings.newLineBetweenFunctions && directive.isInstanceOf[FunctionDirectiveNode])
                if (requiresNewLine) {
                  printer.println()
                }
                generate(directive)
              })
            printer.printIndent()
            printer.println("---")
          }
        }
      }
      case DoBlockNode(header, body, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        printer.print("do {")
        printer.println()
        printer.indent()
        generate(header)
        generate(body)
        printer.dedent()
        printer.println()
        printer.print("}")
      }
      case fn @ FunctionNode(args, body, wtype, typeParameterList) => {
        if (AstNodeHelper.notInjectedNode(fn)) {
          if (typeParameterList.isDefined) {
            generateBracketedTypeList(typeParameterList.get.typeParameters)
          }
          if (!isImplicitParameters(args)) {
            generate(args)
            generateTypeDeclaration(wtype)
            printer.print(" -> ")
          }

          val shouldNextLine = body.isInstanceOf[UsingNode] || body.isInstanceOf[FunctionNode]
          val shouldIndent = shouldNextLine || !(body.isInstanceOf[ArrayNode] || body.isInstanceOf[ObjectNode])

          if (shouldIndent) printer.indent()
          if (shouldNextLine) printer.println()

          generate(body)

          if (shouldIndent) printer.dedent()
        } else {
          generate(body)
        }
      }
      case TypeParametersApplicationListNode(typeParameters) => {
        generateBracketedTypeList(typeParameters)
      }
      case DirectiveOption(name, value) => {
        printer.print(name.name)
        printer.print("=")
        generate(value)
      }
      case SchemaNode(properties) => {
        printer.print("{")
        properties.zipWithIndex.foreach((prop) => {
          if (prop._2 > 0) {
            printer.print(", ")
          }
          generate(prop._1)
        })
        printer.print("}")
      }
      case SchemaPropertyNode(name, value, condition) => {
        if (condition.isDefined) {
          printer.print("(")
        }
        generate(name)
        printer.printSpace(":")
        generate(value)
        if (condition.isDefined) {
          printer.print(") if")
          generate(condition.get)
        }
      }
      case VersionMajor(v)          => printer.print(v)
      case VersionMinor(v)          => printer.print(v)
      case on: OpNode               => generateOpNode(on)
      case dn: DirectiveNode        => generateDirectiveNode(dn)
      case cn: ContainerAstNode     => generateContainers(cn)
      case lvn: LiteralValueAstNode => generateLiterals(lvn)
      case UsingVariableAssignments(vars) => {
        printer.print("(")
        vars.zipWithIndex.foreach((varWithIndex) => {
          if (varWithIndex._2 > 0) {
            printer.print(", ")
          }
          generate(varWithIndex._1.name)
          printer.printSpace(" =")
          generate(varWithIndex._1.value)
        })
        printer.print(")")
      }
      case UsingNode(vars, expr, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        printer.printSpace("using")
        generate(vars)
        printer.indent()
        printer.println()
        generate(expr)
        printer.dedent()
      }
      case typeNode: WeaveTypeNode => generateType(typeNode)
      case HeadTailArrayNode(head, tail, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        printer.print("[")
        generate(head)
        printer.print(" ~ ")
        generate(tail)
        printer.print("]")
      }
      case HeadTailObjectNode(headKey, headValue, tail, annotations) => {
        printAnnotations(annotations)
        printer.print("{")
        generate(headKey)
        generate(headValue)
        printer.print(" ~ ")
        generate(tail)
        printer.print("}")
      }
    }
  }

  private def generateBracketedTypeList(types: Seq[WeaveTypeNode]) = {
    val previousValue = insideParameterList
    insideParameterList = true
    printer
      .print("<")
      .printForeachWithSeparator(", ", types, generateType)
      .print(">")
    insideParameterList = previousValue
  }

  private def stringInterpolation(literal: Seq[AstNode]): Unit = {
    literal.foreach {
      case str @ StringNode(value, codeAnnotations) if str.annotation(classOf[InterpolationExpressionAnnotation]).isEmpty => {
        printAnnotations(codeAnnotations)
        printer.print(value)
      }
      case element => {
        val maybeAnnotation = element.annotation(classOf[InterpolationExpressionAnnotation])
        if (maybeAnnotation.isDefined && !maybeAnnotation.get.enclosedExpression) {
          element match {
            case VariableReferenceNode(name, _) if isImplicitParameter(name) => generate(element)
            case _ => {
              printer.print("$")
              generate(element)
            }
          }
        } else {
          printer.print("$(")
          generate(element)
          printer.print(")")
        }
      }
    }
  }

  private def isConditionalNode(value: AstNode): Boolean = {
    value.isInstanceOf[IfNode] || value.isInstanceOf[UnlessNode]
  }

  private def isDoBlock(value: AstNode): Boolean = {
    value.isInstanceOf[DoBlockNode]
  }

  private def printComment(comment: CommentNode): CodeWriter = {

    val commentText = comment.literalValue.trim
    comment.commentType match {
      case CommentType.BlockComment =>
        printer.printSpace("/*")
        printer.printSpace(commentText)
        printer.print("*/")
      case CommentType.DocComment =>
        printer.println("/**")
        commentText.linesIterator.foreach((line) => printer.printSpace("*").println(line))
        printer.print("**/")
      case CommentType.LineComment =>
        printer.printSpace("//").print(commentText)
    }
  }

  def generateTypeDeclaration(wtype: Option[WeaveTypeNode], printSpaceAfterwards: Boolean = false): Unit = {
    if (wtype.isDefined) {
      printer.printSpace(":")
      generate(wtype.get)
      if (printSpaceAfterwards) {
        printer.printSpace()
      }
    }
  }

  def generateType(typeNode: WeaveTypeNode): Any = {

    val maybeAnnotation = typeNode.annotation(classOf[EnclosedMarkAnnotation])
    val encloseParenthesis: Boolean = maybeAnnotation.isDefined
    val endParenthesisIndex: Int = maybeAnnotation.map(_.location.endPosition.index).getOrElse(0)
    var parenthesisClosed = false

    def closeParenthesisIfMetadataIsOutside(schemaNode: SchemaNode): Boolean = {
      if (encloseParenthesis && endParenthesisIndex < schemaNode.location().startPosition.index) {
        printer.print(")")
        true
      } else {
        false
      }
    }

    if (encloseParenthesis) {
      printer.print("(")
    }

    typeNode match {
      case ObjectTypeNode(properties, schema, typeSchema, close, ordered, annotations) => {
        printAnnotations(annotations)
        if (properties.isEmpty && !close && !ordered) {
          printer.print(TypeLiteral.OBJECT_TYPE_NAME)
        } else {
          printer.print("{")
          if (ordered) {
            printer.print("-")
          }
          if (close) {
            printer.print("|")
          }
          if (settings.multilineTypes && properties.nonEmpty && !insideParameterList) {
            printer.println()
            printer.indent()
          } else {
            printer.printSpace()
          }
          properties.zipWithIndex.foreach((prop) => {
            if (prop._2 > 0) {
              printer.print(",")
              if (settings.multilineTypes && !insideParameterList) {
                printer.println()
              } else {
                printer.printSpace()
              }
            }
            generate(prop._1)
          })
          if (settings.multilineTypes && properties.nonEmpty && !insideParameterList) {
            printer.println()
            printer.dedent()
          } else {
            printer.printSpace()
          }

          if (close) {
            printer.print("|")
          }
          if (ordered) {
            printer.print("-")
          }
          printer.print("}")
        }
        if (schema.isDefined) {
          printer.printSpace()
          generate(schema.get)
        }
        if (typeSchema.isDefined && AstNodeHelper.notInjectedNode(typeSchema.get)) {
          parenthesisClosed = closeParenthesisIfMetadataIsOutside(typeSchema.get)
          printer.print(" <~ ")
          generate(typeSchema.get)
        }
      }
      case _: DynamicReturnTypeNode => printer.print("?")
      case KeyValueTypeNode(key, value, repeated, optional) => {
        if (!key.isInstanceOf[KeyTypeNode]) {
          printer.print("(")
        }
        generateType(key)
        if (!key.isInstanceOf[KeyTypeNode]) {
          printer.print(")")
        }

        if (repeated) {
          printer.print("*")
        }
        if (optional) {
          printer.print("?")
        }
        printer.printSpace(":")
        generateType(value)
      }
      case NameValueTypeNode(name, value, optional) => {
        if (name.ns.isDefined) {
          printer.print(name.ns.get.prefix.name)
          printer.print("#")
        }
        if (name.localName.isDefined) {
          if (StringEscapeHelper.keyRequiresQuotes(name.localName.get)) {
            printer.printQuoted(name.localName.get)
          } else {
            printer.print(name.localName.get)
          }
        } else {
          printer.print("_")
        }

        if (optional) {
          printer.print("?")
        }
        printer.printSpace(":")
        generate(value)
      }
      case KeyTypeNode(name, attrs, annotations, _, _) => {
        printAnnotations(annotations)
        if (name.ns.isDefined) {
          printer.print(name.ns.get.prefix.name)
          printer.print("#")
        }
        if (name.localName.isDefined) {
          if (StringEscapeHelper.keyRequiresQuotes(name.localName.get)) {
            printer.printQuoted(name.localName.get)
          } else {
            printer.print(name.localName.get)
          }
        } else {
          printer.print("_")
        }
        if (attrs.nonEmpty) {
          printer.print(" @(")
          attrs.zipWithIndex.foreach((attr) => {
            if (attr._2 > 0) {
              printer.printSpace(",")
            }
            generate(attr._1)
          })
          printer.print(")")
        }
      }
      case TypeParameterNode(name, base) => {
        printer.print(name.name)
        if (base.isDefined) {
          printer.print(" <: ")
          generate(base.get)
        }
      }
      case FunctionParameterTypeNode(name, valueType, optional) => {
        if (name.isDefined) {
          generate(name.get)
          if (optional)
            printer.print("?")
          printer.printSpace(":")
        }
        generate(valueType)
      }
      case FunctionTypeNode(args, returnType, schema, typeSchema, annotations) => {
        printAnnotations(annotations)
        printer.print("(")
        args.zipWithIndex.foreach((arg) => {
          if (arg._2 > 0) {
            printer.printSpace(",")
          }
          generate(arg._1)
        })
        printer.print(") -> ")
        generate(returnType)
        if (schema.isDefined) {
          printer.printSpace()
          generate(schema.get)
        }
        if (typeSchema.isDefined && AstNodeHelper.notInjectedNode(typeSchema.get)) {
          parenthesisClosed = closeParenthesisIfMetadataIsOutside(typeSchema.get)
          printer.print(" <~ ")
          generate(typeSchema.get)
        }
      }
      case UnionTypeNode(elems, schema, typeSchema, annotations) => {
        printAnnotations(annotations)
        val definedSchema = schema.isDefined
        if (definedSchema) {
          printer.print("(")
        }
        elems.zipWithIndex.foreach(t => {
          generate(t._1)
          if (t._2 < elems.size - 1) {
            printer.print(" | ")
          }
        })
        if (definedSchema) {
          printer.print(")")
          printer.printSpace()
          generate(schema.get)
        }
        if (typeSchema.isDefined && AstNodeHelper.notInjectedNode(typeSchema.get)) {
          parenthesisClosed = closeParenthesisIfMetadataIsOutside(typeSchema.get)
          printer.print(" <~ ")
          generate(typeSchema.get)
        }
      }
      case NativeTypeNode(typeId, asSchema) => {
        printer.println(typeId)
        if (asSchema.isDefined && asSchema.get.properties.nonEmpty) {
          printer.printSpace()
          generate(asSchema.get)
        }
      }
      case IntersectionTypeNode(elems, asSchema, typeSchema, annotations) => {
        printAnnotations(annotations)
        val definedSchema = asSchema.isDefined
        if (definedSchema) {
          printer.print("(")
        }
        elems.zipWithIndex.foreach(t => {
          generate(t._1)
          if (t._2 < elems.size - 1) {
            printer.print(" & ")
          }
        })
        if (definedSchema) {
          printer.print(")")
          printer.printSpace()
          generate(asSchema.get)
        }
        if (typeSchema.isDefined && AstNodeHelper.notInjectedNode(typeSchema.get)) {
          parenthesisClosed = closeParenthesisIfMetadataIsOutside(typeSchema.get)
          printer.print(" <~ ")
          generate(typeSchema.get)
        }
      }
      case TypeReferenceNode(variable, typeParameters, asSchema, typeSchema, annotations) => {
        printAnnotations(annotations)
        if (variable.loader.isDefined) {
          printer.print(variable.loader.get).print('!')
        }
        printer.print(variable.name)
        if (typeParameters.isDefined) {
          generateBracketedTypeList(typeParameters.get)
        }
        if (asSchema.isDefined && asSchema.get.properties.nonEmpty) {
          printer.printSpace()
          generate(asSchema.get)
        }
        if (typeSchema.isDefined && AstNodeHelper.notInjectedNode(typeSchema.get)) {
          parenthesisClosed = closeParenthesisIfMetadataIsOutside(typeSchema.get)
          printer.print(" <~ ")
          generate(typeSchema.get)
        }
      }
      case TypeSelectorNode(selector, weaveTypeNode, asSchema, typeSchema, annotations) => {
        printAnnotations(annotations)
        generate(weaveTypeNode)
        printer.print(".")
        generate(selector)
        if (asSchema.isDefined && asSchema.get.properties.nonEmpty) {
          printer.printSpace()
          generate(asSchema.get)
        }
        if (typeSchema.isDefined && AstNodeHelper.notInjectedNode(typeSchema.get)) {
          parenthesisClosed = closeParenthesisIfMetadataIsOutside(typeSchema.get)
          printer.print(" <~ ")
          generate(typeSchema.get)
        }
      }
      case LiteralTypeNode(value, asSchema, typeSchema, annotations) => {
        printAnnotations(annotations)
        generate(value)
        if (asSchema.isDefined && asSchema.get.properties.nonEmpty) {
          printer.printSpace()
          generate(asSchema.get)
        }
        if (typeSchema.isDefined && AstNodeHelper.notInjectedNode(typeSchema.get)) {
          parenthesisClosed = closeParenthesisIfMetadataIsOutside(typeSchema.get)
          printer.print(" <~ ")
          generate(typeSchema.get)
        }
      }
    }

    if (encloseParenthesis && !parenthesisClosed) {
      printer.print(")")
    }
  }

  def isImplicitParameters(args: FunctionParameters): Boolean = {
    args.paramList.nonEmpty && args.paramList.forall(isImplicitParameter)
  }

  def generateOpNode(on: OpNode): Unit = {
    on match {
      case UnaryOpNode(opId, rhs, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        generateUnaryNode(opId, rhs, on)
      }
      case bon: BinaryOpNode => {
        generateBinaryNode(bon)
      }
    }
  }

  def generateContainers(cn: ContainerAstNode): Unit = {
    cn match {
      case FunctionCallParametersNode(args) => {
        printer.print("(")
        args.zipWithIndex.foreach((arg) => {
          if (arg._2 > 0) {
            printer.printSpace(",")
          }
          generate(arg._1)
        })
        printer.print(")")
      }
      case fp: FunctionParameters => {
        val previousValue = insideParameterList
        insideParameterList = true
        if (!isImplicitParameters(fp)) {
          printer.print("(")
          fp.paramList.zipWithIndex.foreach((arg) => {
            if (!isImplicitParameter(arg._1)) {
              if (arg._2 > 0) {
                printer.printSpace(",")
              }
              generate(arg._1)
            }
          })
          printer.print(")")
        }
        insideParameterList = previousValue
      }
      case _ =>
    }
  }

  def isImplicitParameter(arg: FunctionParameter): Boolean = {
    val variable: NameIdentifier = arg.variable
    isImplicitParameter(variable)
  }

  def isImplicitParameter(variable: NameIdentifier): Boolean = {
    variable.name.matches("""\$+""")
  }

  def generateLiterals(lvn: LiteralValueAstNode): Unit = {
    lvn match {
      case DateTimeNode(literalValue, codeAnnotations)      => printLiteralValue(literalValue, codeAnnotations)
      case TimeNode(literalValue, codeAnnotations)          => printLiteralValue(literalValue, codeAnnotations)
      case LocalDateTimeNode(literalValue, codeAnnotations) => printLiteralValue(literalValue, codeAnnotations)
      case UriNode(literalValue, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        printer.print(literalValue)
      }
      case PeriodNode(literalValue, codeAnnotations) => printLiteralValue(literalValue, codeAnnotations)
      case RegexNode(literalValue, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        printer.print("/").print(RegexNode.escapeString(literalValue)).print("/")
      }
      case NullNode(codeAnnotations) => {
        printAnnotations(codeAnnotations)
        printer.print("null")
      }
      case LocalTimeNode(literalValue, codeAnnotations) => printLiteralValue(literalValue, codeAnnotations)
      case NumberNode(literalValue, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        printer.print(literalValue)
      }
      case BooleanNode(literalValue, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        printer.print(literalValue)
      }
      case TimeZoneNode(literalValue, codeAnnotations)  => printLiteralValue(literalValue, codeAnnotations)
      case LocalDateNode(literalValue, codeAnnotations) => printLiteralValue(literalValue, codeAnnotations)
      case sn: StringNode => {
        printAnnotations(sn.codeAnnotations)
        val quotes = sn.quotedBy()
        if (quotes.isDefined) {
          printer.print(quotes.get.toString)
        }
        printer.print(sn.value)
        if (quotes.isDefined) {
          printer.print(quotes.get.toString)
        }
      }
      case _ =>
    }
  }

  private def printLiteralValue(literalValue: String, codeAnnotations: Seq[AnnotationNode]): Unit = {
    printAnnotations(codeAnnotations)
    printer.print("|").print(literalValue).print("|")
  }

  def isNativeFunctionCall(body: AstNode): Boolean = {
    body match {
      case FunctionCallNode(VariableReferenceNode(NameIdentifier("native", None), _), _, _, _) => true
      case _ => false
    }
  }

  private def printFunctionDirective(variable: NameIdentifier, literal: FunctionNode) = {
    printer.printSpace("fun")
    generate(variable)
    if (literal.typeParameterList.isDefined) {
      generateBracketedTypeList(literal.typeParameterList.get.typeParameters)
    }
    generate(literal.params)
    generateTypeDeclaration(literal.returnType)
    val requiresNewLine = !isNativeFunctionCall(literal.body) && !isDoBlock(literal.body)

    if (requiresNewLine)
      printer.print(" =")
    else
      printer.printSpace(" =")

    if (requiresNewLine) {
      printer.indent()
      printer.println()
    }

    generate(literal.body)

    if (requiresNewLine) {
      printer.dedent()
    }

    printer.println()
  }

  def printAnnotations(annotations: Seq[AnnotationNode], separator: String = System.lineSeparator()): Unit = {
    annotations.foreach(annotation => {
      printer.print("@")
      generate(annotation.name)
      if (annotation.args.isDefined) {
        printer.print("(")
        val notInjectedNodeArgs = annotation.args.get.args.filter(arg => AstNodeHelper.notInjectedNode(arg))
        printer.printForeachWithSeparator(", ", notInjectedNodeArgs, (arg: AnnotationArgumentNode) => {
          printer.printSpace(arg.name.name)
          printer.printSpace("=")
          generate(arg.value)
        })
        printer.print(")")
      }
      printer.print(separator)
    })
  }

  def generateDirectiveNode(dn: DirectiveNode): Unit = {
    dn match {
      case FunctionDirectiveNode(variable, literal: FunctionNode, annotations) => {
        printAnnotations(annotations)
        printFunctionDirective(variable, literal)
      }
      //TODO: Review this
      case FunctionDirectiveNode(variable, literal: OverloadedFunctionNode, _) => {
        literal.functions.foreach(printFunctionDirective(variable, _))
      }
      case AnnotationDirectiveNode(nameIdentifier, params, annotations) => {
        printAnnotations(annotations)
        printer.printSpace("annotation").print(nameIdentifier.name)
        printer
          .print("(")
          .printForeachWithSeparator(", ", params.paramList, (p: AnnotationParameterNode) => {
            printer.print(p.nameIdentifier.name).printSpace(":")
            generateType(p.weaveType)
            if (p.defaultValue.isDefined) {
              printer.printSpace(" =")
              generate(p.defaultValue.get)
            }
          })
          .print(")")
        printer.println()
      }
      case ImportDirective(moduleIdentifier, subElements, codeAnnotations) => {
        printAnnotations(codeAnnotations)
        printer.printSpace("import")
        subElements.elements.zipWithIndex.foreach((componentWithIndex) => {
          val index: Int = componentWithIndex._2
          if (index > 0) {
            printer.print(", ")
          }
          val importedElement: ImportedElement = componentWithIndex._1
          printer.print(importedElement.elementName.name.trim)
          if (importedElement.alias.isDefined) {
            printer.print(" as ")
            printer.print(importedElement.alias.get.name.trim)
          }
        })

        if (subElements.elements.nonEmpty) {
          printer.print(" from ")
        }
        val moduleName: NameIdentifier = moduleIdentifier.elementName
        if (moduleName.loader.isDefined) {
          printer.print(moduleName.loader.get).print('!')
        }
        printer.print(moduleName.name)

        if (moduleIdentifier.alias.isDefined) {
          printer.print(" as ")
          printer.print(moduleIdentifier.alias.get.name)
        }

        printer.println()
      }
      case InputDirective(variable, dataFormat, mime, options, wtype, annotationNodes) => {
        printAnnotations(annotationNodes)
        printer.printSpace("input")
        generate(variable)
        generateTypeDeclaration(wtype)
        printer.printSpace()
        if (dataFormat.isDefined) {
          printer.printSpace(dataFormat.get.id)
        } else {
          printer.printSpace(mime.get.mime)
        }
        printer.printSpace()
        if (options.isDefined) {
          options.get.zipWithIndex.foreach((opt) => {
            if (opt._2 > 0) {
              printer.printSpace(",")
            }
            generate(opt._1)
          })
        }
        printer.println()
      }
      case NamespaceDirective(prefix, uri, annotationNodes) => {
        printAnnotations(annotationNodes)
        printer.printSpace("ns")
        printer.printSpace(prefix.name)
        generate(uri)
        printer.println()
      }
      case OutputDirective(dataFormat, mime, options, wtype, annotationNodes) => {
        printAnnotations(annotationNodes)
        printer.printSpace("output")
        generateTypeDeclaration(wtype, true)
        if (dataFormat.isDefined) {
          if (mime.isDefined) {
            printer.printSpace(mime.get.mime)
            printer.printSpace("with")
          }
          printer.print(dataFormat.get.id)
        } else {
          printer.printSpace(mime.get.mime)
        }
        printer.printSpace()
        if (options.isDefined) {
          options.get.zipWithIndex.foreach((opt) => {
            if (opt._2 > 0) {
              printer.printSpace(",")
            }
            generate(opt._1)
          })
        }
        printer.println()
      }
      case TypeDirective(variable, typeParametersListNode, typeExpression, annotationNodes) => {
        printAnnotations(annotationNodes)
        printer.printSpace("type")
        generate(variable)
        if (typeParametersListNode.isDefined) {
          generateBracketedTypeList(typeParametersListNode.get.typeParameters)
        }

        printer.print(" = ")

        typeExpression match {
          case NativeTypeNode(_, _) => printer.print("???")
          case _                    => generateType(typeExpression)
        }
        printer.println()
      }
      case VarDirective(variable, value, wtype, annotations) => {
        printAnnotations(annotations)
        printer.printSpace("var")
        generate(variable)
        generateTypeDeclaration(wtype)
        printer.print(" = ")
        generate(value)
        printer.println()
      }
      case VersionDirective(major, minor, annotationNodes) => {
        printAnnotations(annotationNodes)
        printer.printSpace("%dw")
        generate(major)
        printer.print(".")
        generate(minor)
        printer.println()
      }
    }
  }

  def generateUnaryNode(opId: UnaryOpIdentifier, rhs: AstNode, unaryNode: OpNode): Unit = {
    opId match {
      case MinusOpId => {
        printer.print("-")
        generate(rhs)
      }
      case AllSchemaSelectorOpId => {
        generate(rhs)
        printer.print(".^")
      }
      case DescendantsSelectorOpId => {
        generate(rhs)
        printer.print("..")
      }
      case NotOpId => {
        unaryNode.annotation(classOf[BooleanNotTypeAnnotation]).map(_.notType) match {
          case Some(NotType.Word) => {
            printer.printSpace("not")
            generate(rhs)
          }
          case Some(NotType.Exclamation) => {
            printer.print("!")
            generate(rhs)
          }
          case None => { //In case the annotation is not there
            printer.printSpace("not")
            generate(rhs)
          }
        }
      }
      case AllAttributesSelectorOpId => {
        generate(rhs)
        printer.print(".@")
      }
      case NamespaceSelectorOpId => {
        generate(rhs)
        printer.print(".#")
      }
    }
  }

  def isBracketSelector(binaryOpNode: BinaryOpNode) = {
    binaryOpNode.annotation(classOf[BracketSelectorAnnotation]).isDefined
  }

  def generateSelector(binaryOpNode: BinaryOpNode, selectorStr: String): Unit = {
    val lhs = binaryOpNode.lhs
    val rhs = binaryOpNode.rhs

    generate(lhs)
    if (isBracketSelector(binaryOpNode)) {
      printer.print("[")
      printer.print(selectorStr)
      generate(rhs)
      printer.print("]")
    } else {
      val strBuilder = new StringBuilder()
      if (!isDescendantsSelector(lhs))
        strBuilder.append(".")
      strBuilder.append(selectorStr)
      printer.print(strBuilder.toString())
      generate(rhs)
    }
  }

  def generateBinaryNode(binaryOpNode: BinaryOpNode): Unit = {
    val opId: BinaryOpIdentifier = binaryOpNode.opId
    val lhs: AstNode = binaryOpNode.lhs
    val rhs: AstNode = binaryOpNode.rhs

    opId match {
      case AttributeValueSelectorOpId => {
        generateSelector(binaryOpNode, "@")
      }
      case MultiValueSelectorOpId => {
        generateSelector(binaryOpNode, "*")
      }
      case MultiAttributeValueSelectorOpId => {
        generateSelector(binaryOpNode, "*@")
      }
      case GreaterOrEqualThanOpId => {
        generate(lhs)
        printer.print(" >= ")
        generate(rhs)
      }
      case AdditionOpId => {
        generate(lhs)
        printer.print(" + ")
        generate(rhs)
      }
      case IsOpId => {
        generate(lhs)
        printer.print(" is ")
        generate(rhs)
      }
      case SubtractionOpId => {
        generate(lhs)
        printer.print(" - ")
        generate(rhs)
      }
      case DivisionOpId => {
        generate(lhs)
        printer.print(" / ")
        generate(rhs)
      }
      case MultiplicationOpId => {
        generate(lhs)
        printer.print(" * ")
        generate(rhs)
      }
      case LeftShiftOpId => {
        generate(lhs)
        printer.print(" << ")
        generate(rhs)
      }
      case RightShiftOpId => {
        generate(lhs)
        printer.print(" >> ")
        generate(rhs)
      }
      case DynamicSelectorOpId => {
        generate(lhs)
        printer.print("[")
        generate(rhs)
        printer.print("]")
      }
      case EqOpId => {
        generate(lhs)
        printer.print(" == ")
        generate(rhs)
      }
      case NotEqOpId => {
        generate(lhs)
        printer.print(" != ")
        generate(rhs)
      }
      case GreaterThanOpId => {
        generate(lhs)
        printer.print(" > ")
        generate(rhs)
      }
      case SchemaValueSelectorOpId => {
        generate(lhs)
        printer.print(".^")
        generate(rhs)
      }
      case ValueSelectorOpId => {
        generateSelector(binaryOpNode, "")
      }
      case ObjectKeyValueSelectorOpId => {
        generateSelector(binaryOpNode, "&")
      }
      case FilterSelectorOpId => {
        generate(lhs)
        printer.print("[?(")
        generate(rhs.asInstanceOf[FunctionNode].body)
        printer.print(")]")
      }
      case SimilarOpId => {
        generate(lhs)
        printer.print(" ~= ")
        generate(rhs)
      }
      case RangeSelectorOpId => {
        generate(lhs)
        printer.print("[")
        generate(rhs)
        printer.print("]")
      }
      case LessThanOpId => {
        generate(lhs)
        printer.print(" < ")
        generate(rhs)
      }
      case LessOrEqualThanOpId => {
        generate(lhs)
        printer.print(" <= ")
        generate(rhs)
      }
      case AsOpId => {
        generate(lhs)
        printer.print(" as ")
        generate(rhs)
      }
      case MetadataInjectorOpId =>
        generate(lhs)
        printer.print(" <~ ")
        generate(rhs)
      case MetadataAdditionOpId =>
        generate(lhs)
    }
  }

  private def isDescendantsSelector(lhs: AstNode) = {
    lhs.isInstanceOf[UnaryOpNode] && (lhs.asInstanceOf[UnaryOpNode].opId == DescendantsSelectorOpId)
  }
}

object InfixOptions {
  val NEVER = 0
  val ALWAYS = 1
  val KEEP = 2

}

case class CodeGeneratorSettings(infixNotation: Int = InfixOptions.KEEP, alwaysInsertVersion: Boolean = false, newLineBetweenFunctions: Boolean = true, orderDirectives: Boolean = true, multilineTypes: Boolean = false) {}

@WeaveApi(Seq("Studio"))
object CodeGenerator {
  def generate(node: AstNode): String = {
    generate(node, CodeGeneratorSettings())
  }

  def generate(node: AstNode, settings: CodeGeneratorSettings): String = {
    val printer: StringCodeWriter = new StringCodeWriter()
    new CodeGenerator(printer, settings).generate(node)
    printer.codeContent()
  }
}
