package org.mule.weave.v2.ts.resolvers

import org.mule.weave.v2.sdk.ObjectSubtractionTypeResolver
import org.mule.weave.v2.ts.ArrayType
import org.mule.weave.v2.ts.CustomTypeResolver
import org.mule.weave.v2.ts.FunctionType
import org.mule.weave.v2.ts.FunctionTypeParameter
import org.mule.weave.v2.ts.IntersectionType
import org.mule.weave.v2.ts.KeyType
import org.mule.weave.v2.ts.KeyValuePairType
import org.mule.weave.v2.ts.NameType
import org.mule.weave.v2.ts.NullType
import org.mule.weave.v2.ts.NumberType
import org.mule.weave.v2.ts.ObjectType
import org.mule.weave.v2.ts.ReferenceType
import org.mule.weave.v2.ts.StringType
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.TypeNode
import org.mule.weave.v2.ts.UnionType
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.ts.WeaveTypeResolutionContext

object CustomFunctionTypeResolver {
  private val customTypeResolvers = Seq(
    ("mapObject", MapObjectCustomTypeResolver),
    ("reduce", ReduceCustomTypeResolver),
    ("--", RemoveKeysObjectCustomTypeResolver))

  def findRegisteredCustomTypeResolver(name: String, params: Seq[FunctionTypeParameter], ctx: WeaveTypeResolutionContext): Option[CustomTypeResolver] = {
    customTypeResolvers.find((typeResolver) => typeResolver._1.equals(name) && typeResolver._2.appliesTo(params, ctx)).map(_._2)
  }
}

object ReduceCustomTypeResolver extends CustomTypeResolver {

  override def resolve(invocationTypes: Seq[WeaveType], ctx: WeaveTypeResolutionContext, node: TypeNode, resolvedReturnType: WeaveType): Option[WeaveType] = {
    val mayBeFunctionType = TypeHelper.asFunctionType(invocationTypes.last)
    mayBeFunctionType match {
      case Some(functionType) => {
        val secondParamDefaultValue = functionType.params(1).defaultValueType
        if (secondParamDefaultValue.isDefined) {
          val arrayType = invocationTypes.head
          val mayBeArrayOf = TypeHelper.getArrayType(arrayType)
          mayBeArrayOf match {
            case Some(arrayItemType) => {
              val functionReturnType = FunctionCallNodeResolver.resolveReturnType(functionType, Seq(arrayItemType, secondParamDefaultValue.get), Seq(), node, ctx)
              functionReturnType.map((rt) => {
                TypeHelper.unify(Seq(secondParamDefaultValue.get, rt))
              })
            }
            case None => {
              None
            }
          }
        } else {
          Some(resolvedReturnType)
        }
      }
      case None => None
    }
  }
}

object MapObjectCustomTypeResolver extends CustomTypeResolver {

  override def resolve(invocationTypes: Seq[WeaveType], ctx: WeaveTypeResolutionContext, node: TypeNode, resolvedReturnType: WeaveType): Option[WeaveType] = {
    val functionType: FunctionType = invocationTypes.last.asInstanceOf[FunctionType]
    val objectType: WeaveType = invocationTypes.head
    resolve(objectType, functionType, ctx, node, resolvedReturnType)
  }

  @scala.annotation.tailrec
  private def resolve(objectType: WeaveType, functionType: FunctionType, ctx: WeaveTypeResolutionContext, node: TypeNode, resolvedReturnType: WeaveType): Option[WeaveType] = {
    objectType match {
      case ObjectType(properties, _, _) => {
        val mappedProperties = properties.map((property) => {
          val returnType = FunctionCallNodeResolver.resolveReturnType(functionType, Seq(property.value, property.key, NumberType()), Seq(), node, ctx)
          ObjectType(getObjectProps(returnType))
        })
        if (mappedProperties.isEmpty) {
          Some(ObjectType())
        } else {
          Some(TypeHelper.resolveIntersection(mappedProperties))
        }
      }
      case IntersectionType(of) => {
        TypeHelper.resolveIntersection(of) match {
          case _: IntersectionType => Some(resolvedReturnType)
          case otherType           => resolve(otherType, functionType, ctx, node, resolvedReturnType)
        }
      }
      case rt: ReferenceType => resolve(rt.resolveType(), functionType, ctx, node, resolvedReturnType)
      case NullType()        => Some(NullType())
      case _ => {
        Some(resolvedReturnType)
      }
    }
  }

  @scala.annotation.tailrec
  private def getObjectProps(returnType: Option[WeaveType]): Seq[KeyValuePairType] = {
    returnType match {
      case Some(ObjectType(objectProps, _, _)) => objectProps
      case Some(rt: ReferenceType)             => getObjectProps(Some(rt.resolveType()))
      case _                                   => Seq()
    }
  }
}

object RemoveKeysObjectCustomTypeResolver extends CustomTypeResolver {

  override def appliesTo(params: Seq[FunctionTypeParameter], ctx: WeaveTypeResolutionContext): Boolean = {
    params.size == 2 && TypeHelper.canBeAssignedTo(params.head.wtype, ObjectType(), ctx) && (TypeHelper.canBeAssignedTo(params.last.wtype, ArrayType(KeyType(NameType())), ctx) || TypeHelper.canBeAssignedTo(params.last.wtype, ArrayType(StringType()), ctx))
  }

  override def resolve(invocationTypes: Seq[WeaveType], ctx: WeaveTypeResolutionContext, node: TypeNode, resolvedReturnType: WeaveType): Option[WeaveType] = {
    val objectType = invocationTypes.head
    val keysToRemove = invocationTypes.last
    objectType match {
      case IntersectionType(of) => resolve(TypeHelper.resolveIntersection(of), keysToRemove, ctx)
      case _                    => resolve(objectType, keysToRemove, ctx)
    }
  }

  private def resolve(objects: WeaveType, keys: WeaveType, ctx: WeaveTypeResolutionContext): Option[WeaveType] = {
    objects match {
      case objectType: ObjectType => {
        keys match {
          case ArrayType(of) => {
            ObjectSubtractionTypeResolver.resolve(objectType, of, ctx)
          }
          case _ => {
            Some(ObjectType())
          }
        }
      }
      case UnionType(of) => {
        Some(TypeHelper.unify(of.flatMap((wtype) => resolve(wtype, keys, ctx))))
      }
      case rt: ReferenceType => resolve(rt.resolveType(), keys, ctx)
      case _                 => Some(ObjectType())
    }
  }
}