package org.mule.weave.v2.editor.refactor

import org.mule.weave.v2.WeaveEditorSupport
import org.mule.weave.v2.editor.CodeRefactor
import org.mule.weave.v2.grammar.Tokens
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.DirectivesCapableNode
import org.mule.weave.v2.parser.ast.functions.DoBlockNode
import org.mule.weave.v2.parser.ast.functions.FunctionNode
import org.mule.weave.v2.parser.ast.header.directives.DirectiveNode
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.structure.KeyNode
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.types.WeaveTypeNode
import org.mule.weave.v2.scope.AstNavigator
import org.mule.weave.v2.scope.DependenciesAnalyzerService
import org.mule.weave.v2.scope.ScopesNavigator
import org.mule.weave.v2.scope.VariableDependency
import org.mule.weave.v2.scope.VariableScope
import org.mule.weave.v2.utils.WeaveNameHelper.inferVariableName

/**
  * This service provides a set of common refactors
  * @param editor The editor
  */
class RefactorService(editor: WeaveEditorSupport) {

  /**
    * Extracts a variable or a type in the most near scope.
    * @param startOffset The start offset of the selection to extract
    * @param endOffset The end offset
    * @return The Extract variable refactor if selection is valid
    */
  def extractVariable(startOffset: Int, endOffset: Int, toRootScope: Boolean = false): Option[CodeRefactor] = {
    editor
      .astNavigator()
      .flatMap((navigator) => {
        val maybeNode: Option[AstNode] = navigator.nodeAt(startOffset, endOffset)
        maybeNode match {
          case Some(nodeToExtract) if (AstNodeHelper.isExpressionNode(nodeToExtract) || AstNodeHelper.isWeaveTypeNode(nodeToExtract)) => {
            val tokenType = nodeToExtract match {
              case _: WeaveTypeNode => Tokens.TYPE
              case _                => Tokens.VAR
            }
            //As we are extracting a selector or a key we should put it into a string interpolation
            val inToStringInterpolation = navigator.parentWithType(nodeToExtract, classOf[NameNode]).isDefined || navigator.parentWithType(nodeToExtract, classOf[KeyNode]).isDefined
            editor
              .scopeGraph()
              .flatMap((scopeGraph) => {
                val maybeScope: Option[VariableScope] = if (toRootScope) {
                  scopeGraph
                    .scopeOf(nodeToExtract)
                    .map((vs) => vs.rootScope())
                } else {
                  calculateTargetScope(nodeToExtract, scopeGraph)
                }
                maybeScope.flatMap((vs) => {
                  val defaultVarName: String = inferVariableName(nodeToExtract, navigator, "Var", vs)
                  val valueScope: VariableScope = detectCorrectScope(vs)
                  valueScope.astNode match {
                    case dcn: DirectivesCapableNode => {
                      val index: Int = calculateInsertionIndex(navigator, nodeToExtract, dcn)
                      val insertSeparator: Boolean = calculateRequiresSeparator(dcn)
                      val refactor = new ExtractVariableRefactor(false, insertSeparator, inToStringInterpolation, nodeToExtract.location(), index, vs.astNode.location(), defaultVarName, tokenType)
                      Some(refactor)
                    }
                    case _ => {
                      val refactor = new ExtractVariableRefactor(true, true, inToStringInterpolation, nodeToExtract.location(), vs.astNode.location().startPosition.index, vs.astNode.location(), defaultVarName, tokenType)
                      Some(refactor)
                    }
                  }
                })
              })
          }
          case _ => None
        }
      })
  }

  /**
    * Extracts the selected block into a function
    * @param startOffset The start offset of the selection
    * @param endOffset The end offset of the selection
    * @return
    */
  def extractFunction(startOffset: Int, endOffset: Int): Option[CodeRefactor] = {
    editor
      .astNavigator()
      .flatMap((navigator) => {
        val maybeNode: Option[AstNode] = navigator.nodeAt(startOffset, endOffset)
        maybeNode match {
          case Some(nodeToExtract) if (AstNodeHelper.isExpressionNode(nodeToExtract)) => {
            //As we are extracting a selector or a key we should put it into a string interpolation
            val inToStringInterpolation = navigator.parentWithType(nodeToExtract, classOf[NameNode]).isDefined || navigator.parentWithType(nodeToExtract, classOf[KeyNode]).isDefined
            editor
              .scopeGraph()
              .flatMap((scopeGraph) => {
                val maybeScope: Option[VariableScope] = calculateTargetScope(nodeToExtract, scopeGraph)
                maybeScope.flatMap((vs) => {
                  val defaultVarName: String = inferVariableName(nodeToExtract, navigator, "Fun", vs)
                  val valueScope: VariableScope = vs.rootScope()
                  val dependenciesAnalyzerService = new DependenciesAnalyzerService(editor)
                  val dependencies: Array[VariableDependency] = dependenciesAnalyzerService.externalScopeDependencies(startOffset, endOffset, Some(valueScope))
                  valueScope.astNode match {
                    case dcn: DirectivesCapableNode => {
                      val index: Int = calculateInsertionIndex(navigator, nodeToExtract, dcn)
                      val insertSeparator: Boolean = calculateRequiresSeparator(dcn)
                      val refactor =
                        new ExtractFunctionRefactor(false, insertSeparator, inToStringInterpolation, nodeToExtract.location(), index, vs.astNode.location(), defaultVarName, dependencies)
                      Some(refactor)
                    }
                    case _ => {
                      val refactor = new ExtractFunctionRefactor(
                        true,
                        true,
                        inToStringInterpolation,
                        nodeToExtract.location(),
                        vs.astNode.location().startPosition.index,
                        vs.astNode.location(),
                        defaultVarName,
                        dependencies)
                      Some(refactor)
                    }
                  }
                })
              })
          }
          case _ => None
        }
      })
  }

  private def calculateTargetScope(nodeToExtract: AstNode, scopeGraph: ScopesNavigator): Option[VariableScope] = {
    def toValidParent(maybeScope: Option[VariableScope]): Option[VariableScope] = {
      maybeScope.flatMap((vs) => {
        vs.astNode match {
          case _: WeaveTypeNode => toValidParent(vs.parentScope)
          case _                => Some(vs)
        }
      })
    }

    val maybeScope = scopeGraph.scopeOf(nodeToExtract)
    toValidParent(maybeScope)
  }

  private def calculateRequiresSeparator(dcn: DirectivesCapableNode) = {
    dcn match {
      case DoBlockNode(header, _, _) => header.directives.isEmpty
      case DocumentNode(header, _)   => header.directives.forall((n) => AstNodeHelper.isInjectedNode(n))
      case _                         => false
    }
  }

  private def calculateInsertionIndex(navigator: AstNavigator, nodeToExtract: AstNode, dcn: DirectivesCapableNode) = {

    if (dcn.directives.isEmpty) {
      dcn match {
        case DoBlockNode(_, body, _) => body.location().startPosition.index
        case DocumentNode(_, root)   => root.location().startPosition.index
        case _                       => dcn.location().startPosition.index
      }
    } else {
      val maybeDirectiveNode: Option[DirectiveNode] = navigator.parentWithType(nodeToExtract, classOf[DirectiveNode])
      maybeDirectiveNode match {
        case Some(dn) => {
          val i: Int = dcn.directives.indexOf(dn)
          if (i <= 0) {
            dn.location().startPosition.index
          } else {
            dcn.directives(i - 1).location().endPosition.index
          }
        }
        case None => {
          dcn.directives
            .maxBy(_.location().endPosition.index)
            .location()
            .endPosition
            .index
        }
      }

    }

  }

  private def detectCorrectScope(vs: VariableScope): VariableScope = {
    vs.astNode match {
      case _: FunctionNode => detectCorrectScope(vs.parentScope.get)
      case _               => vs
    }
  }
}
