package org.mule.weave.v2.module.protobuf.utils

import com.google.protobuf.ByteString
import com.google.protobuf.Descriptors.Descriptor
import com.google.protobuf.Descriptors.EnumValueDescriptor
import com.google.protobuf.Message
import com.google.protobuf.UnknownFieldSet
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.structure.schema.Schema
import org.mule.weave.v2.model.structure.schema.SchemaProperty
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.values.BinaryValue
import org.mule.weave.v2.model.values.BooleanValue
import org.mule.weave.v2.model.values.KeyValue
import org.mule.weave.v2.model.values.NumberValue
import org.mule.weave.v2.model.values.ObjectValueBuilder
import org.mule.weave.v2.model.values.StringValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.math.Number
import org.mule.weave.v2.module.protobuf.exception.ProtoBufParsingException
import org.mule.weave.v2.module.protobuf.utils.CommonValues.ENUM_INDEX_PROPERTY_NAME
import org.mule.weave.v2.module.protobuf.utils.CommonValues.unrecognizedEnum
import org.mule.weave.v2.module.protobuf.utils.ProtobufWireTypes.WIRE_TYPE_PROPERTY_NAME
import org.mule.weave.v2.parser.location.UnknownLocation

import scala.collection.JavaConverters.asScalaBuffer
import scala.collection.JavaConverters.collectionAsScalaIterableConverter

object ProtoToDWConverter {
  /**
    * Extract an ObjectValueBuilder from an UnknownFieldSet by setting
    * keys to "-<index>"
    *
    * @param unks
    * @return
    */
  def unknownFieldSetToDWValue(unks: UnknownFieldSet): ObjectValueBuilder = {
    def schemaForWireType(wt: ProtobufWireTypes.WireTypes): Schema = {
      Schema(Seq(SchemaProperty(
        StringValue(WIRE_TYPE_PROPERTY_NAME),
        StringValue(wt))))
    }

    val builder = new ObjectValueBuilder()

    unks.asMap().forEach((idx, field) => {
      val key: KeyValue = KeyValue(QualifiedName(s"-${idx}", None), None)

      field.getGroupList.forEach((unks) => {
        builder.addPair(key, unknownFieldSetToDWValue(unks).withSchema(schemaForWireType(ProtobufWireTypes.Group)))
      })

      field.getVarintList.forEach((n) => {
        builder.addPair(key, NumberValue(Number(n), UnknownLocationCapable, Some(schemaForWireType(ProtobufWireTypes.Varint))))
      })
      field.getFixed32List.forEach((n) => {
        builder.addPair(key, NumberValue(Number(n), UnknownLocationCapable, Some(schemaForWireType(ProtobufWireTypes._32Bit))))
      })
      field.getFixed64List.forEach((n) => {
        builder.addPair(key, NumberValue(Number(n), UnknownLocationCapable, Some(schemaForWireType(ProtobufWireTypes._64Bit))))
      })

      field.getLengthDelimitedList.forEach((bs) => {
        builder.addPair(key, BinaryValue(bs.toByteArray, UnknownLocationCapable, Some(schemaForWireType(ProtobufWireTypes.LengthDelimited))))
      })

    })

    builder
  }

  def protoMessageToDWValue(msg: Message): Value[_] = {
    MessageParser.parseMessage(msg).getOrElse(
      {
        val desc: Descriptor = msg.getDescriptorForType()
        val fields = asScalaBuffer(desc.getFields)
        val builder = new ObjectValueBuilder()
        fields.foreach((fd) => {
          val n = fd.getName

          FieldParser
            .parseFieldFrom(msg, fd)
            .map(value => builder.addPair(n, value))
            .getOrElse({
              val value = msg.getField(fd)
              if (fd.isRepeated) {
                value.asInstanceOf[java.util.List[Object]].asScala.foreach(f => builder.addPair(n, protoToDWValue(f)))
              } else if (msg.hasField(fd) || fd.getContainingOneof == null) {
                builder.addPair(n, protoToDWValue(value))
              }
            })
        })

        builder.mergeFrom(unknownFieldSetToDWValue(msg.getUnknownFields))

        builder.build
      })
  }

  def protoToDWValue(value: Object): Value[_] = {
    if (value.isInstanceOf[Double]) {
      NumberValue.safe(value.asInstanceOf[Double])
    } else if (value.isInstanceOf[Float]) {
      NumberValue.safe(value.asInstanceOf[Float])
    } else if (value.isInstanceOf[Long]) {
      NumberValue.safe(value.asInstanceOf[Long])
    } else if (value.isInstanceOf[Int]) {
      NumberValue.safe(value.asInstanceOf[Int])
    } else if (value.isInstanceOf[Boolean]) {
      BooleanValue(value.asInstanceOf[Boolean])
    } else if (value.isInstanceOf[Message]) {
      protoMessageToDWValue(value.asInstanceOf[Message])
    } else if (value.isInstanceOf[ByteString]) {
      BinaryValue(value.asInstanceOf[ByteString].toByteArray)
    } else if (value.isInstanceOf[String]) {
      StringValue(value.asInstanceOf[String])
    } else if (value.isInstanceOf[EnumValueDescriptor]) {
      val enumValue = value.asInstanceOf[EnumValueDescriptor]
      EnumParser.parseEnum(enumValue).getOrElse({
        val schema = Schema(Seq(SchemaProperty(StringValue(ENUM_INDEX_PROPERTY_NAME), NumberValue(enumValue.getNumber))))
        if (enumValue.getIndex == -1) {
          // If it's not present  on the descriptor, it doesn't have an index
          StringValue(unrecognizedEnum, Some(schema))
        } else {
          StringValue(enumValue.getName, Some(schema))
        }
      })
    } else {
      // This shouldn't happen
      throw new ProtoBufParsingException("The given protobuf value is not supported", UnknownLocation)
    }
  }
}
