package org.mule.weave.v2.module.protobuf.utils

import com.google.protobuf.ByteString
import com.google.protobuf.Descriptors.Descriptor
import com.google.protobuf.Descriptors.EnumDescriptor
import com.google.protobuf.Descriptors.EnumValueDescriptor
import com.google.protobuf.Descriptors.FieldDescriptor
import com.google.protobuf.Descriptors.FieldDescriptor.JavaType
import com.google.protobuf.Descriptors.OneofDescriptor
import com.google.protobuf.DynamicMessage
import com.google.protobuf.Message
import com.google.protobuf.UnknownFieldSet
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.types.BinaryType
import org.mule.weave.v2.model.types.BooleanType
import org.mule.weave.v2.model.types.KeyType
import org.mule.weave.v2.model.types.NumberType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values
import org.mule.weave.v2.module.protobuf.exception.ProtoBufWritingException
import org.mule.weave.v2.module.protobuf.utils.CommonValues.ENUM_INDEX_PROPERTY_NAME
import org.mule.weave.v2.module.protobuf.utils.CommonValues.unrecognizedEnum
import org.mule.weave.v2.module.protobuf.utils.ProtobufWireTypes.WIRE_TYPE_PROPERTY_NAME
import org.mule.weave.v2.module.protobuf.utils.ProtobufWireTypes.WireTypes

import scala.collection.mutable

object DWToProtoConverter {
  def DWValueToProto(value: Value[_], fdesc: FieldDescriptor)(implicit ctx: EvaluationContext): Any = {
    fdesc.getJavaType match {
      case JavaType.DOUBLE      => DWValueToDouble(value)
      case JavaType.FLOAT       => DWValueToFloat(value)
      case JavaType.LONG        => DWValueToLong(value)
      case JavaType.INT         => DWValueToInt(value)
      case JavaType.BOOLEAN     => DWValueToBool(value)
      case JavaType.STRING      => DWValueToString(value)
      case JavaType.MESSAGE     => DWValueToMessage(value, fdesc.getMessageType)
      case JavaType.BYTE_STRING => DWValueToByteString(value)
      case JavaType.ENUM        => DWValueToEnum(value, fdesc.getEnumType)
      case other =>
        // This shouldn't happen
        throw new ProtoBufWritingException(value.location(), s"The ProtoBuf ${other} is not supported by the DataWeave writer")
    }
  }

  private def DWValueToDouble(value: Value[_])(implicit ctx: EvaluationContext): Double = {
    if (NumberType.accepts(value)) {
      val v = NumberType.coerce(value).evaluate
      v.toDouble
    } else {
      throw new ProtoBufWritingException(value.location(), s"Can't write as a Double")
    }
  }

  private def DWValueToFloat(value: Value[_])(implicit ctx: EvaluationContext): Float = {
    if (NumberType.accepts(value)) {
      val v = NumberType.coerce(value).evaluate
      v.toFloat
    } else {
      throw new ProtoBufWritingException(value.location(), s"Can't write as a Float")
    }
  }

  private def DWValueToLong(value: Value[_])(implicit ctx: EvaluationContext): Long = {
    if (NumberType.accepts(value)) {
      val v = NumberType.coerce(value).evaluate
      v.toLong
    } else if (KeyType.accepts(value)) {
      try {
        val v = NumberType.coerce(value).evaluate
        v.toLong
      } catch {
        case e: Exception =>
          throw new ProtoBufWritingException(value.location(), s"Can't parse as a Long")
      }
    } else {
      throw new ProtoBufWritingException(value.location(), s"Can't write as a Long")
    }
  }

  private def DWValueToInt(value: Value[_])(implicit ctx: EvaluationContext): Int = {
    if (NumberType.accepts(value)) {
      val v = NumberType.coerce(value).evaluate
      v.toInt
    } else if (KeyType.accepts(value)) {
      try {
        val v = NumberType.coerce(value).evaluate
        v.toInt
      } catch {
        case e: Exception =>
          throw new ProtoBufWritingException(value.location(), s"Can't parse as an Int")
      }
    } else {
      throw new ProtoBufWritingException(value.location(), s"Can't write as an Int")
    }
  }

  private def DWValueToBool(value: Value[_])(implicit ctx: EvaluationContext): Boolean = {
    if (BooleanType.accepts(value)) {
      val v = BooleanType.coerce(value).evaluate
      v
    } else if (KeyType.accepts(value)) {
      try {
        val v = BooleanType.coerce(value).evaluate
        v
      } catch {
        case e: Exception =>
          throw new ProtoBufWritingException(value.location(), s"Can't parse as a Boolean")
      }
    } else {
      throw new ProtoBufWritingException(value.location(), s"Can't write as a Boolean")
    }
  }

  private def DWValueToString(value: Value[_])(implicit ctx: EvaluationContext): String = {
    if (StringType.accepts(value) || KeyType.accepts(value)) {
      val v = StringType.coerce(value).evaluate
      v.toString
    } else {
      throw new ProtoBufWritingException(value.location(), s"Can't write as a String")
    }
  }

  private def DWValueToByteString(value: Value[_])(implicit ctx: EvaluationContext): ByteString = {
    // Is this ok to take Binaries to ByteString?
    if (BinaryType.accepts(value)) {
      ByteString.readFrom(BinaryType.coerce(value).evaluate)
    } else {
      throw new ProtoBufWritingException(value.location(), s"Can't write as a Binary")
    }
  }

  private def DWValueToEnum(value: Value[_], edesc: EnumDescriptor)(implicit ctx: EvaluationContext): EnumValueDescriptor = {
    def getIndexFromValue(value: Value[_]): Option[values.math.Number] = {
      for {
        schema <- value.schema
        indexPropt <- schema.valueOf(ENUM_INDEX_PROPERTY_NAME)
        if (NumberType.accepts(indexPropt))
        indexEvaluated = NumberType.coerce(indexPropt).evaluate
      } yield {
        indexEvaluated
      }
    }

    EnumParser.writeEnum(value, edesc).getOrElse(
      if (StringType.accepts(value)) {
        val v = StringType.coerce(value).evaluate

        if (v == unrecognizedEnum) {
          getIndexFromValue(value) match {
            case Some(index) =>
              edesc.findValueByNumberCreatingIfUnknown(index.toInt)
            case None =>
              throw new ProtoBufWritingException(value.location(), s"Can't write as an Enum since no index was specified")

          }
        } else {
          val enumValue = edesc.findValueByName(v.toString)
          if (enumValue == null)
            throw new ProtoBufWritingException(value.location(), s"Can't find Enum value ${v.toString}")
          else {
            getIndexFromValue(value) match {
              case Some(index) if index.toInt != enumValue.getNumber =>
                throw new ProtoBufWritingException(value.location(), s"Enum ${v.toString} value has index ${enumValue.getNumber}, but ${index} was specified")
              case _ => enumValue
            }
          }
        }
      } else {
        throw new ProtoBufWritingException(value.location(), s"Can't write as an Enum")
      })
  }

  def DWValueToUnknown(value: Value[_])(implicit ctx: EvaluationContext): UnknownFieldSet.Field = {
    val builder = UnknownFieldSet.Field.newBuilder()

    def getWireTypeFromValue(value: Value[_]) = {
      for {
        schema <- value.schema
        wireTypeProp <- schema.valueOf(WIRE_TYPE_PROPERTY_NAME)
        if (StringType.accepts(wireTypeProp))
        wireTypeEvaluated = StringType.coerce(wireTypeProp).evaluate
        foundWireType <- ProtobufWireTypes.values.find(_ == wireTypeEvaluated)
      } yield {
        foundWireType
      }
    }

    if (NumberType.accepts(value)) {
      lazy val numberValue = NumberType.coerce(value).evaluate
      val maybeWireType: Option[WireTypes] = getWireTypeFromValue(value)

      maybeWireType match {
        case Some(wireType) => wireType match {
          case ProtobufWireTypes.Varint => builder.addVarint(numberValue.toLong)
          case ProtobufWireTypes._32Bit => builder.addFixed32(numberValue.toInt)
          case ProtobufWireTypes._64Bit => builder.addFixed64(numberValue.toLong)
          case ProtobufWireTypes.Group | ProtobufWireTypes.LengthDelimited =>
            throw new ProtoBufWritingException(
              value.location(),
              s"Can't set it to an unknown number with wireType ${wireType}")
        }
        case None =>
          // Using VarInt as default, is it ok?
          builder.addVarint(numberValue.toLong)
      }
    } else if (BinaryType.accepts(value)) {
      if (getWireTypeFromValue(value).exists(_ != ProtobufWireTypes.LengthDelimited)) {
        throw new ProtoBufWritingException(value.location(), "Can only set Binaries to length delimited wire types.")
      }
      val binaryValue = BinaryType.coerce(value).evaluate
      builder.addLengthDelimited(ByteString.readFrom(binaryValue))
      //    } else if (ObjectType.accepts(value)) {
      //      val objectValue = ObjectType.coerce(value).evaluate
      //      builder.addGroup()
    } else {
      throw new ProtoBufWritingException(value.location(), "Can't set it to an unknown. Only Numbers and Binaries are allowed")
    }

    builder.build()
  }

  val unknownFieldRegex = raw"-(\d*)".r

  def DWValueToMessage(value: Value[_], desc: Descriptor)(implicit ctx: EvaluationContext): Message = {
    MessageParser.writeMessage(value, desc).getOrElse(
      if (ObjectType.accepts(value)) {
        val presentOneOf = mutable.Set[OneofDescriptor]()
        val presentKeys = mutable.Set[FieldDescriptor]()
        val objectSeq: ObjectSeq = ObjectType.coerce(value).evaluate
        val b = DynamicMessage.newBuilder(desc)

        objectSeq.toIterator().foreach(f => {
          val name = f._1.evaluate.name
          val fdesc = desc.findFieldByName(name)
          if (fdesc == null) {
            name match {
              case unknownFieldRegex(index) =>
                b.mergeUnknownFields(UnknownFieldSet.newBuilder().addField(index.toInt, DWValueToUnknown(f._2)).build())
              case _ =>
                throw new ProtoBufWritingException(f._1.location(), "Field name not supported. Only field names present in the schema and unknowns are valid.")
            }
          } else {
            if (!fdesc.isRepeated && presentKeys.contains(fdesc)) {
              throw new ProtoBufWritingException(f._1.location(), s"The field ${fdesc.getFullName} has already been set and is not repeated")
            }
            FieldParser
              .writeField(f._2, fdesc)
              .map(value => {
                b.setField(fdesc, value)
              })
              .getOrElse({
                val fdescOneOf = fdesc.getContainingOneof
                if (fdescOneOf != null) {
                  if (presentOneOf.contains(fdescOneOf)) {
                    throw new ProtoBufWritingException(f._1.location(), s"The oneof ${fdescOneOf.getFullName} has already been set with another field name.")
                  } else {
                    presentOneOf.add(fdescOneOf)
                  }
                }
                if (fdesc.isRepeated) {
                  b.addRepeatedField(fdesc, DWValueToProto(f._2, fdesc))
                } else {
                  presentKeys.add(fdesc)
                  b.setField(fdesc, DWValueToProto(f._2, fdesc))
                }
              })
          }
        })

        // this will check that required fields are present
        b.build()
      } else {
        throw new ProtoBufWritingException(value.location(), s"Can't write as an Object")
      })
  }
}
