package org.mule.weave.v2.module.protobuf.utils

import com.google.protobuf.Descriptors.EnumDescriptor
import com.google.protobuf.Descriptors.EnumValueDescriptor
import com.google.protobuf.NullValue
import org.mule.weave.v2.core.exception.ExecutionException
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.types.NullType
import org.mule.weave.v2.model.values
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.protobuf.exception.ProtoBufParsingException
import org.mule.weave.v2.module.protobuf.exception.ProtoBufWritingException
import org.mule.weave.v2.parser.location.UnknownLocation

trait EnumParser[ProtoBufType, DWType] {
  def accepts(fdesc: EnumDescriptor): Boolean

  def fromDw(value: Value[_], desc: EnumDescriptor)(implicit ctx: EvaluationContext): ProtoBufType = {
    try {
      doFromDw(value, desc)
    } catch {
      case e: ExecutionException =>
        throw new ProtoBufWritingException(value.location(), e.message)
    }
  }

  protected def doFromDw(value: Value[_], desc: EnumDescriptor)(implicit ctx: EvaluationContext): ProtoBufType

  def toDw(en: EnumValueDescriptor): Value[DWType] = {
    if (!accepts(en.getType)) {
      throw new ProtoBufParsingException(s"Can't transform ${en.getFullName}", UnknownLocation)
    } else {
      doToDw(en)
    }
  }

  protected def doToDw(en: EnumValueDescriptor): Value[DWType]
}

object EnumParser {
  val enumParsers = Seq(NullValueParser)

  def parseEnum(en: EnumValueDescriptor): Option[Value[Any]] = {
    enumParsers.find(_.accepts(en.getType)).map(_.toDw(en))
  }

  def writeEnum(value: Value[_], desc: EnumDescriptor)(implicit ctx: EvaluationContext): Option[EnumValueDescriptor] = {
    enumParsers.find(_.accepts(desc)).map(_.fromDw(value, desc).getValueDescriptor)
  }
}

object NullValueParser extends EnumParser[NullValue, Null] {
  override def accepts(fdesc: EnumDescriptor): Boolean = fdesc.getFullName == "google.protobuf.NullValue"

  override protected def doFromDw(value: Value[_], desc: EnumDescriptor)(implicit ctx: EvaluationContext): NullValue = {
    if (NullType.accepts(value))
      NullValue.NULL_VALUE
    else
      throw new ProtoBufWritingException(value.location(), "Could not write as NullValue.")
  }

  override protected def doToDw(en: EnumValueDescriptor): Value[Null] = {
    if (en.getName == "NULL_VALUE")
      values.NullValue
    else
      throw new ProtoBufParsingException(s"Can't parse ${en.toString} as null", UnknownLocation)
  }
}
