package org.mule.weave.v2.helper

import org.mule.weave.v2.core.io.FileHelper
import org.mule.weave.v2.interpreted.Dumper
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.module.core.json.JsonDataFormat
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.runtime.CompilationResult
import org.mule.weave.v2.runtime.WeaveCompiler
import org.mule.weave.v2.sdk.WeaveResourceFactory
import org.scalatest.funspec.AnyFunSpec
import org.scalatest.matchers.should.Matchers

import java.io.BufferedOutputStream
import java.io.File
import java.io.FileInputStream
import java.io.FileOutputStream
import java.nio.charset.Charset

class AbstractDumperTest extends AnyFunSpec
    with Matchers
    with FolderBasedTest
    with ParsingContextTestAware
    with AbstractMatcherTest
    with ReadCapableTest {

  scenarios.foreach {
    case (scenario, inputs, transform, expectedDump) => {
      it(scenario) {
        implicit val evaluationContext: EvaluationContext = EvaluationContext.dumpOnErrorCtx()
        val workingDirectory = evaluationContext.serviceManager.settingsService.dumper().workingDirectory()
        val theFolder = new File(workingDirectory, Dumper.DUMP_FOLDER)
        if (theFolder.exists()) {
          FileHelper.deleteDirectory(theFolder)
          theFolder.mkdirs()
        }
        intercept[Exception] {
          val readers = buildReaders(inputs)
          val context: ParsingContext = createTestParsingContext(transform)
          context.disableCommonSubExpressionElimination()
          inputs.foreach((f) => {
            context.addImplicitInput(FileHelper.baseName(f), None)
          })
          val compilerResult: PhaseResult[CompilationResult[DocumentNode]] = WeaveCompiler.compile(WeaveResourceFactory.fromFile(transform), context)
          val executable = compilerResult.getResult().executable
          executable.writeWith(executable.implicitWriterOption().getOrElse(new JsonDataFormat().writer(None)), readers, Map())
        }

        val dumpFolder: File = ls(theFolder).maxBy(_.lastModified())
        ls(expectedDump)
          .map((expectedFile: File) => {
            val actualFileDumped: File = new File(dumpFolder, expectedFile.getName)
            assert(
              actualFileDumped.exists(),
              s"Expected File dump ${expectedFile.getName}, not exist. Actual files are: \n-${ls(dumpFolder).map(_.getAbsolutePath).mkString("\n-")}")
            assertMatches(new FileInputStream(actualFileDumped), Charset.forName("UTF-8"), expectedFile)
          })
        assert(ls(expectedDump).length == ls(dumpFolder).length, s"The amount of files dumped is different from the expected one. Actual files are: \n-${ls(dumpFolder).map(_.getAbsolutePath).mkString("\n-")}")
        FileHelper.deleteDirectory(theFolder)
      }
    }
  }

  private def ls(folder: File): Array[File] = {
    Option(folder.listFiles()).getOrElse(Array())
  }

  override def updateResult(ext: String, result: Array[Byte], expectedFile: File): Unit = {
    val testCaseFolder = getTestCaseFolder(expectedFile)
    val srcAst: File = new File(testCaseFolder, expectedFile.getParentFile.getParentFile.getName + File.separator + expectedFile.getParentFile.getName + File.separator + expectedFile.getName)
    println("WARNING -------------------- Updating ->" + srcAst.getCanonicalPath)
    val bos = new BufferedOutputStream(new FileOutputStream(srcAst))
    bos.write(result)
    bos.close()
  }

  override def getTestCaseFolder(expectedFile: File): File = {
    val parts: Array[String] = getClass.getPackage.getName.split("\\.")
    val testCaseFolder = new File(expectedFile.getParentFile, s"../../../../../${"../" * parts.length}src/test/resources/${parts.mkString("/")}")
    testCaseFolder
  }

  def scenarios: Seq[(String, Array[File], File, File)] = {
    val folders = getTestFoldersFor(getClass)
    folders
      .flatMap(
        (folder) =>
          ls(folder)
            .filter(acceptScenario)
            .map((file: File) => {
              (file.getName, ls(file).filter(_.getName.matches("in[0-9]+\\.[a-zA-Z]+")), new File(file, "transform.dwl"), new File(file, "expectedDump"))
            }))
      .sortBy((testEntry) => testEntry._1)
  }
}
