package org.mule.weave.v2.helper

import org.mule.weave.v2.core.io.FileHelper
import org.mule.weave.v2.core.util.TypeAliases
import org.mule.weave.v2.interpreted.RuntimeModuleNodeCompiler
import org.mule.weave.v2.interpreted.extension.ParsingContextCreator
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.SPIWeaveServicesProvider
import org.mule.weave.v2.model.ServiceManager
import org.mule.weave.v2.model.WeaveServicesProvider
import org.mule.weave.v2.model.service.LanguageLevelService
import org.mule.weave.v2.model.service.SettingsService
import org.mule.weave.v2.module.CompositeDataFormatExtensionsLoaderService
import org.mule.weave.v2.module.DataFormatExtensionsLoaderService
import org.mule.weave.v2.module.DefaultDataFormatExtensionsLoaderService
import org.mule.weave.v2.module.reader.Reader
import org.mule.weave.v2.module.writer.Writer
import org.mule.weave.v2.interpreted.extension.WeaveBasedDataFormatExtensionLoaderService
import org.mule.weave.v2.model.service.DefaultSecurityManagerService
import org.mule.weave.v2.model.service.SecurityManagerService
import org.mule.weave.v2.model.service.WeaveRuntimePrivilege
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.module.MimeType
import org.mule.weave.v2.parser.phase.CompilationException
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.runtime.ExecutableWeave
import org.mule.weave.v2.runtime.WeaveCompiler
import org.mule.weave.v2.sdk.ClassLoaderWeaveResourceResolver
import org.mule.weave.v2.sdk.WeaveResourceFactory
import org.scalatest.funspec.AnyFunSpec
import org.scalatest.matchers.should.Matchers

abstract class AbstractEngineTest extends AnyFunSpec
    with Matchers
    with FolderBasedTest
    with AbstractMatcherTest
    with ParsingContextTestAware
    with BaseDataWeaveTest
    with ReadCapableTest
    with WriteCapableTest
    with ConfigPropertyAwareTest {

  scenarios(getClass).foreach { scenario =>
    it(scenario.name) {
      var i = 0
      val languageLevelService = createLanguageLevelService(scenario.configProperty)
      val settingsService = createSettingsService(scenario.configProperty, languageLevelService)
      val context: ParsingContext = setupParsingContext(createTestParsingContext(scenario.transform, settingsService.parsingContext()))
      val moduleParserManager = context.moduleParserManager
      val moduleNodeLoader: RuntimeModuleNodeCompiler = RuntimeModuleNodeCompiler()
      val loaderService = WeaveBasedDataFormatExtensionLoaderService(ParsingContextCreator(moduleParserManager), ClassLoaderWeaveResourceResolver.noContextClassloader(), moduleNodeLoader)
      val manager = ServiceManager(
        WeaveServicesProvider(
          WeaveServicesProvider(
            Map[Class[_], Any](
              //Map Of Services
              classOf[LanguageLevelService] -> languageLevelService,
              classOf[SettingsService] -> settingsService,
              classOf[DataFormatExtensionsLoaderService] -> CompositeDataFormatExtensionsLoaderService(DefaultDataFormatExtensionsLoaderService, loaderService))),
          new SPIWeaveServicesProvider))
      scenario.inputs.foreach(f => {
        context.addImplicitInput(FileHelper.baseName(f), None)
      })
      val compilerResult = if (System.getProperty("noCheck") != null) {
        WeaveCompiler.compileWithNoCheck(WeaveResourceFactory.fromFile(scenario.transform), context, moduleNodeLoader)
      } else {
        WeaveCompiler.compile(WeaveResourceFactory.fromFile(scenario.transform), context, moduleNodeLoader)
      }
      //Check no issues with on the compilation
      if (compilerResult.hasErrors()) {
        throw new CompilationException(compilerResult.messages())
      }

      val outputExtension = FileHelper.getExtension(scenario.output)
      val executableWeave = compilerResult.getResult().executable

      try {
        while (i < executionCount()) {
          val start = System.currentTimeMillis()
          if (executionCount() > 1) {
            println(s"Start Execution number: ${i}")
          }
          implicit val evaluationContext: EvaluationContext = EvaluationContext(manager)

          val readers: Map[String, Reader] = buildReaders(scenario.inputs)

          val writer = buildWriter(outputExtension, executableWeave.outputDataFormatMimeType.map(MimeType.fromSimpleString))
          executableWeave.configureWriter(writer)
          val tuple = writeWith(executableWeave, readers, writer)
          val writerResult = tuple._1
          val encoding = tuple._2
          val end = System.currentTimeMillis()
          if (executionCount() > 1) {
            println(s"Time of ${i}: " + (end - start) + "ms")
          }
          assertMatches(writerResult, encoding, scenario.output)
          i = i + 1
        }
      } catch {
        case t: Throwable => fail(t)
      }
    }
  }

  def setupParsingContext(context: ParsingContext): ParsingContext = context

  def executionCount(): Int = Integer.parseInt(System.getProperty("times", "1"))

  protected def writeWith(executableWeave: ExecutableWeave[DocumentNode], readers: Map[String, Reader], writer: Writer)(implicit ctx: EvaluationContext): (Any, TypeAliases#JCharset) = {
    executableWeave.writeWith(writer, readers)
  }
}
