package org.mule.weave.v2.helper

import org.mule.weave.v2.core.io.FileHelper

import java.io._
import java.text.NumberFormat
import java.util.Locale
import org.mule.weave.v2.interpreted.RuntimeModuleNodeCompiler
import org.mule.weave.v2.interpreted.extension.ParsingContextCreator
import org.mule.weave.v2.interpreted.extension.WeaveBasedDataFormatExtensionLoaderService
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.ServiceManager
import org.mule.weave.v2.model.service.DefaultLoggingService
import org.mule.weave.v2.module.CompositeDataFormatExtensionsLoaderService
import org.mule.weave.v2.module.DataFormatExtensionsLoaderService
import org.mule.weave.v2.module.DataFormatManager
import org.mule.weave.v2.module.DefaultDataFormatExtensionsLoaderService
import org.mule.weave.v2.module.writer.Writer
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.runtime.CompilationResult
import org.mule.weave.v2.runtime.WeaveCompiler
import org.mule.weave.v2.sdk.ClassLoaderWeaveResourceResolver
import org.mule.weave.v2.sdk.WeaveResourceFactory
import org.scalatest.funspec.AnyFunSpec
import org.scalatest.matchers.should.Matchers

abstract class AbstractLoadTest extends AnyFunSpec
    with Matchers
    with FolderBasedTest
    with ParsingContextTestAware
    with ReadCapableTest {

  val moduleNodeLoader = RuntimeModuleNodeCompiler()

  scenarios.foreach {
    case (scenario, inputFiles, transform, output) =>
      it(scenario) {
        val context = createTestParsingContext(transform)
        implicit val evaluationContext: EvaluationContext = createEvaluationContext(context)
        val encoding = if (scenario.endsWith("utf16")) "UTF-16" else "UTF-8"
        val readers = inputFiles.zipWithIndex.map((f) => (s"in${f._2}", buildReader(f._1, encoding))).toMap
        val outputFile = File.createTempFile("output", FileHelper.getExtension(output))
        outputFile.deleteOnExit()
        val os = new FileOutputStream(outputFile)
        val writer = buildWriter(outputFile)
        try {
          println(s"Starting Load test for ${scenario}")
          val context = createTestParsingContext(transform)
          inputFiles.zipWithIndex.foreach((f) => {
            context.addImplicitInput("in" + f._2, None)
          })
          val compilerResult = WeaveCompiler.compile(WeaveResourceFactory.fromFile(transform), context, moduleNodeLoader)

          //Check no issues with on the compilation
          assert(
            compilerResult.noErrors(),
            compilerResult
              .errorMessages()
              .map((message) => {
                s"${message._2.message} at ${message._1.startPosition.line}:${message._1.startPosition.column}"
              })
              .mkString("\n"))

          val executable = compilerResult.getResult().executable
          executable.configureWriter(writer)
          executable.writeWith(writer, readers)
          evaluationContext.close()
          println("Output is :" + outputFile)

          val result = new BufferedReader(new InputStreamReader(new FileInputStream(outputFile), encoding))
          val expected = new BufferedReader(new InputStreamReader(new FileInputStream(output), encoding))
          var rline = result.readLine()
          var eline = expected.readLine()
          var i = 0
          while (rline != null && eline != null) {
            assert(rline.equals(eline), s"Line:`${i}` are different:\nActual:\n${rline}\nExpected:\n${eline}")
            rline = result.readLine()
            eline = expected.readLine()
            i = i + 1
          }
        } finally {
          os.close()
        }
      }
  }

  private def createEvaluationContext(context: ParsingContext) = {
    val moduleParserManager = context.moduleParserManager
    val loaderService = WeaveBasedDataFormatExtensionLoaderService(ParsingContextCreator(moduleParserManager), ClassLoaderWeaveResourceResolver.noContextClassloader(), moduleNodeLoader)
    val manager = ServiceManager(
      DefaultLoggingService,
      Map(classOf[DataFormatExtensionsLoaderService] -> CompositeDataFormatExtensionsLoaderService(DefaultDataFormatExtensionsLoaderService, loaderService)))
    EvaluationContext(manager)
  }
  def buildWriter(file: File)(implicit ctx: EvaluationContext): Writer = {
    val stream = new FileOutputStream(file)
    ctx.registerCloseable(stream)
    DataFormatManager.byExtension(FileHelper.getExtension(file)).get.writer(Some(stream))
  }

  def scenarios: Seq[(String, Array[File], File, File)] = {
    val folders = getTestFoldersFor(getClass)
    val context = createTestParsingContext()

    folders
      .flatMap(
        (scenarios) =>
          scenarios.listFiles
            .filter(acceptScenario)
            .map((testFolder: File) => {
              val inputFiles = testFolder.listFiles
                .filter(_.getName.matches("in[0-9]+-template\\.[a-zA-Z]+"))
                .sortBy(_.getName)
                .zipWithIndex
                .map((f) => {
                  val ctx: EvaluationContext = createEvaluationContext(context)
                  val inputFile: File = FileHelper.createTempFile(s"${testFolder.getName}-input-${f._2}", FileHelper.getExtension(f._1))
                  inputFile.deleteOnExit()
                  println("Input : " + inputFile.getPath + " from file " + f._1.getPath)
                  val result: PhaseResult[CompilationResult[DocumentNode]] = WeaveCompiler.compile(WeaveResourceFactory.fromFile(f._1), context, moduleNodeLoader)
                  val writer = buildWriter(inputFile)(ctx)
                  val executable = result.getResult().executable
                  executable.configureWriter(writer)(ctx)
                  executable.writeWith(writer)(ctx)
                  ctx.close()
                  val sizeInKB = inputFile.length() / 1024
                  val formatter = NumberFormat.getIntegerInstance(Locale.US)
                  val formattedSize = formatter.format(sizeInKB)
                  println(s"Input size: $formattedSize KB")
                  inputFile
                })

              val expectedFile = testFolder.listFiles
                .find((f) => f.getName.matches("out-template.[a-zA-Z]+"))
                .map((f) => {
                  implicit val ctx: EvaluationContext = createEvaluationContext(context)
                  val outputFile: File = FileHelper.createTempFile(s"${testFolder.getName}-output", FileHelper.getExtension(f))
                  outputFile.deleteOnExit()
                  println("Output : " + outputFile.getPath)
                  val result: PhaseResult[CompilationResult[DocumentNode]] = WeaveCompiler.compile(WeaveResourceFactory.fromFile(f), context, moduleNodeLoader)
                  val executable = result.getResult().executable
                  val writer = buildWriter(outputFile)
                  executable.configureWriter(writer)
                  executable.writeWith(writer)
                  ctx.close()
                  val sizeInKB = outputFile.length() / 1024
                  val formatter = NumberFormat.getIntegerInstance(Locale.US)
                  val formattedSize = formatter.format(sizeInKB)
                  println(s"Output size: $formattedSize KB")
                  outputFile
                })
                .orNull

              (testFolder.getName, inputFiles, new File(testFolder, "transform.dwl"), expectedFile)
            }))
      .sortBy(_._1)
  }

}
